#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import base64
import datetime
import re
import time
from urllib import urlencode

from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from ...dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from ...synchronizerdaemon import Synchronizer

app = None  # type: Optional[Synchronizer]

logger_auth = LoggerFactory.get_logger(u'AUTHENTICATION')


def user_login():
    user = app.get_user_auth()
    err = app.request.GET.get(u'error', None)
    origin_url = app.request.GET.get(u'origin_url', None)
    to_return = {u'app': app, u'error': err, u'origin_url': origin_url}
    
    for mod in app.modules_manager.get_internal_instances():
        if getattr(mod, u'login_placeholder', None):
            to_return[u'login_placeholder'] = mod.login_placeholder
            break
    
    if user:
        if not user.is_admin() and not user.is_expert():
            to_return[u'error'] = app._(u'login.isnt_admin') % user.get(u'contact_name', u'unknown')
        else:
            app.redirect(u'/main')
    return to_return


def user_login_redirect():
    app.redirect(u'/user/login')
    return {}


def _clean_cookie():
    # delete_cookie want str not unicode
    app.response.delete_cookie('user', secret=app.auth_secret, path='/')


def _set_valid_cookie(user_id):
    expires = datetime.datetime(2100, 1, 1)
    # expires = 'Fri, 01 Jan 2100 00:00:00 GMT'
    # set_cookie want str not unicode
    app.response.set_cookie('user', user_id, secret=app.auth_secret, path='/', expires=expires, httponly=True)


def user_logout():
    start_time = time.time()
    
    user = app.get_user_auth()
    if user:
        user_name = user.get_name()
        app.logger_user_authentication.logout(
            user_uuid=user.get(u'_id', u'(uuid unavailable)'),
            user_name=user_name if user_name else u'(name unavailable)',
            run_time=time.time() - start_time,
            requester_ip=app.request.remote_addr,
            requester_module=u'UI Configuration'
        )
        logger_auth.debug(u'User "%s" logged out' % user.get_name())
    
    _clean_cookie()
    
    app.redirect(u'/user/login')
    return {}


def user_auth():
    login = app.request.forms.get(u'shinken_login', '').decode(u'utf-8', u'ignore')
    password = app.request.forms.get(u'password', '').decode(u'utf-8', u'ignore')
    origin_url = app.request.forms.get(u'origin_url', '').decode(u'utf-8', u'ignore')
    
    authenticated_user = app.authenticate_user_by_modules(login, password, u'UI Configuration')
    is_valid = authenticated_user and authenticated_user.get(u'enabled', u'1') == u'1'
    
    error_txt = u''
    is_bad = False
    url = base64.b64decode(origin_url) if origin_url else u'/main'
    
    if is_valid:
        _set_valid_cookie(authenticated_user[u'_id'])
        
        if not authenticated_user.is_admin() and not authenticated_user.is_expert():
            _clean_cookie()
            logger_auth.warning(u'User "%s" is not Shinken admin or SI admin. Cancel authentication' % authenticated_user.get_name())
            error_txt = app._(u'login.isnt_admin') % login
            is_bad = True
    else:
        _clean_cookie()
        is_bad = True
        error_txt = app._(u'login.bad_auth')
    
    if is_bad:
        params = {
            u'error'     : error_txt.encode(u'utf8'),
            u'origin_url': origin_url if origin_url else u'',
        }
        url = u'/user/login?%s' % (urlencode(params))
    
    return {u'url': url}


# manage the /. If the user is known, go to problems page.
# Should be /dashboard in the future. If not, go login :)
def get_root():
    user = app.request.get_cookie('user', secret=app.auth_secret)
    if user:
        app.redirect(u'/main')
    elif app.remote_user_variable in app.request.headers and app.remote_user_enable == u'1':
        user_name = app.request.headers[app.remote_user_variable].strip()
        
        if app.http_remote_user_case_sensitive:
            user_auth = app.datamanagerV2.find_item_by_name_with_case_sensitive(user_name, ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING)
        else:
            user_auth = app.datamanagerV2.find_item_by_name(user_name, ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING)
        
        if not user_auth:
            print "Warning: You need to have a contact having the same name as your user %s" % user_name
            app.redirect(u'/user/login')
        else:
            _set_valid_cookie(user_auth.get_uuid())
            app.redirect(u'/main')
    else:
        app.redirect(u'/user/login')


pages = {
    user_login         : {u'routes': [u'/user/login', u'/user/login/'], u'view': u'login', u'static': True, u'wrappers': []},
    user_login_redirect: {u'routes': [u'/login'], u'static': True, u'wrappers': []},
    user_auth          : {u'routes': [u'/user/auth'], u'view': u'auth', u'method': u'POST', u'static': True, u'wrappers': []},
    user_logout        : {u'routes': [u'/user/logout', u'/logout'], u'static': True, u'wrappers': []},
    get_root           : {u'routes': [u'/'], u'static': True, u'wrappers': []},
}
