#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import shinkensolutions.shinkenjson as json
from shinken.misc.type_hint import TYPE_CHECKING
from ...business.sync_ui_common import syncuicommon
from ...component.component_manager import component_manager
from ...dao.def_items import ITEM_TYPE, DEF_ITEMS

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from ...synchronizerdaemon import Synchronizer

## Will be populated with a instance of Synchronizer from syncui process
app = None  # type: Optional[Synchronizer]


def get_page():
    user = app.get_user_auth()
    mongo_component = component_manager.get_mongo_component()
    
    configuration_stats = syncuicommon.get_configuration_stats(user)
    
    # Format Type/Sub Type/Have change/Have URL
    display_def = []
    if user.is_admin():
        display_def.append({'type': ITEM_TYPE.CLUSTERS, 'sub_type': [(ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERS), (ITEM_TYPE.CLUSTERTPLS, 'TPLS')], 'have_change': False, 'have_url': True})
        display_def.append({'type': ITEM_TYPE.HOSTS, 'sub_type': [(ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTS), (ITEM_TYPE.HOSTTPLS, 'TPLS'), (ITEM_TYPE.HOSTGROUPS, 'GRPS')], 'have_change': False, 'have_url': True})
        display_def.append({'type': ITEM_TYPE.SERVICESHOSTS, 'sub_type': [(ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESHOSTS), (ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESHOSTTPLS), (ITEM_TYPE.SERVICESCLUSTERS, ITEM_TYPE.SERVICESCLUSTERS),
                                                                          (ITEM_TYPE.SERVICESCLUSTERTPLS, ITEM_TYPE.SERVICESCLUSTERTPLS), (ITEM_TYPE.SERVICETPLS, 'TPLS')], 'have_change': False, 'have_url': True})
        
        display_def.append({'type': ITEM_TYPE.PACK, 'sub_type': [], 'have_change': False, 'have_url': False})
        display_def.append({'type': ITEM_TYPE.CONTACTS, 'sub_type': [(ITEM_TYPE.CONTACTS, ITEM_TYPE.CONTACTS), (ITEM_TYPE.CONTACTTPLS, 'TPLS'), (ITEM_TYPE.CONTACTGROUPS, 'GRPS')], 'have_change': False, 'have_url': True})
        display_def.append({'type': ITEM_TYPE.ESCALATIONS, 'sub_type': [(ITEM_TYPE.ESCALATIONS, ITEM_TYPE.ESCALATIONS)], 'have_change': False, 'have_url': True})
        display_def.append({'type': ITEM_TYPE.NOTIFICATIONWAYS, 'sub_type': [(ITEM_TYPE.NOTIFICATIONWAYS, ITEM_TYPE.NOTIFICATIONWAYS)], 'have_change': False, 'have_url': True})
        
        display_def.append({'type': ITEM_TYPE.COMMANDS, 'sub_type': [(ITEM_TYPE.COMMANDS, ITEM_TYPE.COMMANDS)], 'have_change': False, 'have_url': True})
        display_def.append({'type'       : ITEM_TYPE.BUSINESSIMPACTMODULATIONS,
                            'sub_type'   : [
                                (ITEM_TYPE.BUSINESSIMPACTMODULATIONS, "%s_%s" % (ITEM_TYPE.BUSINESSIMPACTMODULATIONS, 'min')),
                                (ITEM_TYPE.MACROMODULATIONS, "%s_%s" % (ITEM_TYPE.MACROMODULATIONS, 'min')),
                                (ITEM_TYPE.RESULTMODULATIONS, "%s_%s" % (ITEM_TYPE.RESULTMODULATIONS, 'min'))
                            ],
                            'have_change': False, 'have_url': True})
        display_def.append({'type': ITEM_TYPE.TIMEPERIODS, 'sub_type': [(ITEM_TYPE.TIMEPERIODS, ITEM_TYPE.TIMEPERIODS)], 'have_change': False, 'have_url': True})
        display_def.append({'type': 'hide', 'sub_type': [], 'have_change': False, 'have_url': False})
    
    else:
        display_def.append({'type': ITEM_TYPE.HOSTS, 'sub_type': [(ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTS)], 'have_change': False, 'have_url': True})
        display_def.append({'type': ITEM_TYPE.CONTACTS, 'sub_type': [(ITEM_TYPE.CONTACTS, ITEM_TYPE.CONTACTS)], 'have_change': False, 'have_url': True})
    
    if configuration_stats['error']:
        
        r = {'error'  : True,
             'overall': {
                 'working_area': [0, 0],
                 'proposed'    : [0, 0],
                 'new'         : 0,
                 'change'      : 0,
                 'stagging'    : 0,
                 'to_apply'    : 0},
             'detail' : {}}
        for item_type in DEF_ITEMS.keys():
            r['detail'][item_type] = {}
            r['detail'][item_type]['working_area'] = [0, 0]
            r['detail'][item_type]['proposed'] = [0, 0]
            r['detail'][item_type]['new'] = 0
            r['detail'][item_type]['stagging'] = 0
            r['detail'][item_type]['to_apply'] = 0
            r['detail'][item_type]['change'] = 0
        
        r['detail'][ITEM_TYPE.ELEMENTS] = r['overall']
        configuration_stats = r
    
    else:
        for definition in display_def:
            have_change = False
            if definition['sub_type']:
                for item_type in definition['sub_type']:
                    stats = configuration_stats['detail'].get(item_type[0], None)
                    if stats:
                        have_change |= (stats['change'] + stats['new'] > 0)
            else:
                stats = configuration_stats['detail'].get(definition['type'], None)
                if stats:
                    have_change = (stats['change'] + stats['new'] > 0)
            definition['have_change'] = have_change
    
    sources = app.get_api_sources_from_backend().values()
    sources.sort(key=lambda src: src['order'])
    
    discovery_modules = [mod for mod in app.source_module_instances if mod.properties['type'] == 'discovery-import']
    
    discovery_confs = []
    for discovery_mod in discovery_modules:
        discovery_confs.extend(discovery_mod.get_confs())
    discovery_confs = sorted(discovery_confs, key=lambda k: k['discovery_name'])
    
    # We are in a sub process, so we don't have access to the last_synchr value of theses objects main-process.
    for source in sources:
        sname = source['source_name']
        lst = [i for i in mongo_component.col_last_synchronization.find({'source_name': sname})]
        imports = None
        if len(lst) >= 1:
            imports = lst[0]
        if imports is None:
            source['last_import'] = 0
            source['state'] = 'PENDING'
            source['output'] = ''
    return {'app': app, 'user': user, 'sources': sources, 'discovery_confs': discovery_confs, 'configuration_stats': configuration_stats, 'display_def': display_def}


def stats_configuration():
    app.response.content_type = 'application/json'
    user = app.get_user_auth()
    diffs = syncuicommon.get_configuration_stats(user)
    return json.dumps(diffs)


pages = {
    get_page           : {'routes': ['/main/', '/main'], 'view': 'main', 'static': True, 'wrappers': ['auth']},
    stats_configuration: {'routes': ['/stats/configuration'], 'view': None, 'wrappers': ['auth']},
}
