# !/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.transactioner import transaction_protecter, DrainStopInProgressException

try:
    from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    from shinken.synchronizer.front_end.object.messages import ValidatorMessages, Message, MESSAGE
    from shinken.synchronizer.front_end.helper_frontend_response import FrontEndSaveResponse
    from shinken.synchronizer.business.item_controller.exceptions import BusinessException, UnauthorizedSaveException
except ImportError:
    from synchronizer.dao.transactions.transactions import DBTransaction
    from synchronizer.front_end.object.messages import ValidatorMessages, Message, MESSAGE
    from synchronizer.front_end.helper_frontend_response import FrontEndSaveResponse
    from synchronizer.business.item_controller.exceptions import BusinessException, UnauthorizedSaveException

if TYPE_CHECKING:
    from synchronizer.synchronizerdaemon import Synchronizer
    from shinken.misc.type_hint import Optional

app = None  # type: Optional[Synchronizer]


# ...###....##.....##.########.##.....##
# ..##.##...##.....##....##....##.....##
# .##...##..##.....##....##....##.....##
# ##.....##.##.....##....##....#########
# #########.##.....##....##....##.....##
# ##.....##.##.....##....##....##.....##
# ##.....##..#######.....##....##.....##


def abort_if_not_admin_nor_admin_si(func):
    def __wrap(*args, **kwargs):
        
        user = app.get_user_auth()
        if not user:
            logger.info("Invalid user on [%s]" % func.__name__)
            return app.abort(401, app.t('login.invalid_user'))
        
        if not user.is_admin() and not user.is_expert():
            user_name = user.get_name()
            logger.info("User %s isn't a Shinken administrator or a SI administrator." % user_name)
            return app.abort(403, app.t('login.isnt_admin') % user_name)
        return func(*args, **kwargs)
    
    
    return __wrap


def abort_if_not_admin(func):
    def __wrap(*args, **kwargs):
        user = app.get_user_auth()
        if not user:
            logger.info("Invalid user")
            return app.abort(401, app.t('login.invalid_user'))
        
        is_admin = user.is_admin()
        
        if not is_admin:
            user_name = user.get_name()
            logger.info("User %s isn't a Shinken administrator." % user_name)
            return app.abort(403, app.t('login.isnt_shinken_admin') % user_name)
        return func(*args, **kwargs)
    
    
    return __wrap


# If we get a BusinessException, we just return a ValidatorMessage containing the error
def format_validator_message_if_base_exception(func):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except BusinessException as e:
            if e.validator_messages:
                validator_message = e.validator_messages
            else:
                validator_message = ValidatorMessages()
                validator_message.add_message(Message(MESSAGE.STATUS_ERROR, e.text))
            response = FrontEndSaveResponse(validator_message, return_code=e.code, shinken_return_code=e.shinken_return_code)
            return response.get_response()
    
    
    return wrapper


# Similar to "abort_if_not_admin" but we will not abort and return a ValidatorMessage containing an UnauthorizedSaveException error
def format_validator_message_if_not_admin(func):
    @format_validator_message_if_base_exception
    def __wrap(*args, **kwargs):
        user = app.get_user_auth()
        if not user:
            logger.info("Invalid user")
            raise UnauthorizedSaveException
        
        is_admin = user.is_admin()
        
        if not is_admin and not user.is_expert():
            user_name = user.get_name()
            logger.info("User %s isn't a Shinken administrator or a SI administrator." % user_name)
            raise UnauthorizedSaveException
        
        if not is_admin:
            user_name = user.get_name()
            logger.info("User %s isn't a Shinken administrator." % user_name)
            raise UnauthorizedSaveException
        return func(*args, **kwargs)
    
    
    return __wrap


# ########.########.....###....##....##..######.....###.....######..########.####..#######..##....##..######.
# ...##....##.....##...##.##...###...##.##....##...##.##...##....##....##.....##..##.....##.###...##.##....##
# ...##....##.....##..##...##..####..##.##........##...##..##..........##.....##..##.....##.####..##.##......
# ...##....########..##.....##.##.##.##..######..##.....##.##..........##.....##..##.....##.##.##.##..######.
# ...##....##...##...#########.##..####.......##.#########.##..........##.....##..##.....##.##..####.......##
# ...##....##....##..##.....##.##...###.##....##.##.....##.##....##....##.....##..##.....##.##...###.##....##
# ...##....##.....##.##.....##.##....##..######..##.....##..######.....##....####..#######..##....##..######.


# This wrapper must be used for calls that modify objects, so during this
# call, the daemon won't stop (it have 30s to exit)
def protect_transaction(func):
    def __wrap(*args, **kwargs):
        try:
            with transaction_protecter:
                return func(*args, **kwargs)
        except DrainStopInProgressException:
            return app.bottle.abort(400, 'The daemon is currently stopping')
    
    
    return __wrap


# This must be used whenever a db transaction must be done
def protect_db_transaction(func):
    def __wrap(*args, **kwargs):
        with DBTransaction():
            return func(*args, **kwargs)
    
    
    return __wrap


# Some call are purely between the daemon and the HTTP process,
def protect_internal_call(func):
    def __wrap(*args, **kwargs):
        _private_key = app.request.GET.get('private_key', None)
        if not app.check_private_http_key(_private_key):
            app.abort(403, 'Bad private key')
        return func(*args, **kwargs)
    
    
    return __wrap
