#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import os
import threading
from multiprocessing import Process

from shinken.modules.base_module.basemodule import BaseModule
from .log import PART_INITIALISATION
from .misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from .misc.type_hint import Type
    from .moduleworker import ModuleWorker


# Class use to give object from module to worker
class FromModuleToWorkerContainer(object):
    pass


class FROM_DAEMON_TO_MODULE_MESSAGE_TYPES(object):
    ADD_REALM = 1
    DELETE_REALM = 2
    UPDATE_REALM = 3


class FromDaemonToModuleMessage(object):
    def __init__(self, type_, payload):
        self._type = type_
        self._payload = payload
    
    
    def get_type(self):
        return self._type
    
    
    def get_payload(self):
        return self._payload


class WithWorkersAndInventoryModule(BaseModule):
    is_worker_based = True
    
    # By default, the inventory module got not class.
    # USER class MUST set it
    MODULE_WORKER_CLASS = None  # type:Type[ModuleWorker]
    
    
    def __init__(self, mod_conf):
        super(WithWorkersAndInventoryModule, self).__init__(mod_conf)
        self._from_daemon_to_module_queue = None
        
        # Android are not managed, and the queue are not clean in a thread way
        try:
            import android
            raise Exception('The worker based modules are not available on android')
        except ImportError:
            pass
        
        # Precreate sub process inventories
        self._reset_inventories()
        
        self._nb_workers = int(getattr(mod_conf, 'broker_module_nb_workers', '1'))
        self._workers = {}
        self._worker_id = None  # will be set in the worker process
        
        # For perf boosting, we want to
        self._host_uuids_to_worker_cache = {}
        
        # Lock for commands from different threads racing for the result
        self.command_lock = threading.Lock()
    
    
    def check_worker_processes(self):
        for (worker_id, worker) in self._workers.iteritems():
            process = worker.get('process', None)
            if process is not None and not process.is_alive():
                self.logger.error('The worker process %s (pid=%s) is dead. We are stopping the module and we will restart it.' % (worker_id, process.pid))
                self.stop_all()
                return False
        return True
    
    
    def _reset_inventories(self):
        # On the main process, we need to know if we must push or not an inventory to the sub-process
        # IMPORTANT: this dict is only available on the MAIN PROCESS so NOT on the main() function!
        self._main_process_inventory_hashes = {}
        self.logger.debug(PART_INITIALISATION, 'Reset main process inventory')
    
    
    # The broker daemon is giving us its own inventories (one by realm) and we must look if:
    # * we don't have a realm => create it
    # * we have a realm that do not exist any more => delete it
    # * we have a realm but not up-to-date => replace it
    # NOTE: we are safe for the broker_realm_inventories as the broker is calling us from a locked protection for it
    def update_inventory_from_daemon(self, broker_realm_inventories, realms_inventory_differences):
        my_realm_inventories_set = set(self._main_process_inventory_hashes.keys())
        broker_realm_inventories_set = set(broker_realm_inventories.keys())
        
        # To add realms
        need_to_add_realms = broker_realm_inventories_set - my_realm_inventories_set
        
        # No more need realm
        need_to_remove_realms = my_realm_inventories_set - broker_realm_inventories_set
        
        # Need to look for update (we both have but with different hashes)
        need_to_update_realms = []
        
        both_have = my_realm_inventories_set.intersection(broker_realm_inventories_set)
        for realm_name in both_have:
            if broker_realm_inventories[realm_name].get_hash() != self._main_process_inventory_hashes[realm_name]:
                need_to_update_realms.append(realm_name)
        
        # Add new ones
        for realm_name in need_to_add_realms:
            self.logger.debug('update_inventory_from_daemon:: adding realm %s' % realm_name)
            broker_realm_inventory = broker_realm_inventories[realm_name]
            self._warn_sub_processes_about_new_realm(realm_name, broker_realm_inventory)
            self._main_process_inventory_hashes[realm_name] = broker_realm_inventory.get_hash()
        
        # Remove not exiting realms
        for realm_name in need_to_remove_realms:
            self.logger.debug('update_inventory_from_daemon:: removing realm %s' % realm_name)
            self._warn_sub_processes_about_deleted_realm(realm_name)
            del self._main_process_inventory_hashes[realm_name]
        
        # Update exiting one (replace them currently)
        for realm_name in need_to_update_realms:
            self.logger.debug('update_inventory_from_daemon:: updating realm %s' % realm_name)
            realm_inventory = broker_realm_inventories[realm_name]
            inventory_differences = realms_inventory_differences[realm_name]
            self._warn_sub_processes_about_updated_realm(realm_name, realm_inventory, inventory_differences)
            self._main_process_inventory_hashes[realm_name] = realm_inventory.get_hash()
    
    
    def _send_inventory_message_to_worker(self, worker_id, message):
        self._workers[worker_id]['instance']._send_inventory_message_to_worker(message)
    
    
    def _send_inventory_message_to_all_workers(self, message):
        for worker_id, worker in self._workers.iteritems():
            self._send_inventory_message_to_worker(worker_id, message)
    
    
    # We have a new realm, so split it into several inventories and send each sharded inventory into the good worker
    def _warn_sub_processes_about_new_realm(self, realm_name, realm_inventory):
        nb_workers = len(self._workers)
        worker_weights = [1 for _ in xrange(nb_workers)]
        for worker_id, worker_entry in self._workers.iteritems():
            worker_sharded_inventory = realm_inventory.get_new_inventory_from_us_and_take_only_a_shard_part(worker_id, 1, nb_workers, worker_id, worker_weights)
            self.logger.debug('[WORKER:%s] will receive a new realm inventory %s with %s elements' % (worker_id, realm_name, worker_sharded_inventory.get_len()))
            message = FromDaemonToModuleMessage(FROM_DAEMON_TO_MODULE_MESSAGE_TYPES.ADD_REALM, {'realm_name': realm_name, 'inventory': worker_sharded_inventory})
            self._send_inventory_message_to_worker(worker_id, message)
            # Save the computed inventory so we will be able to compute differences after
            worker_entry['inventories'][realm_name] = worker_sharded_inventory
        # Clean the host to workers cache
        self._host_uuids_to_worker_cache.clear()
    
    
    def _warn_sub_processes_about_deleted_realm(self, realm_name):
        message = FromDaemonToModuleMessage(FROM_DAEMON_TO_MODULE_MESSAGE_TYPES.DELETE_REALM, {'realm_name': realm_name, 'inventory': None})
        self._send_inventory_message_to_all_workers(message)
        # Clean the host to workers cache
        self._host_uuids_to_worker_cache.clear()
    
    
    # NOTE: in the broker, we have X workers with a sharded inventory, so the difference is too global
    # and we must recompute the x differences in each sharded_inventories
    def _warn_sub_processes_about_updated_realm(self, realm_name, realm_inventory, inventory_differences):
        nb_workers = len(self._workers)
        worker_weights = [1 for _ in xrange(nb_workers)]
        for worker_id, worker_entry in self._workers.iteritems():
            updated_worker_sharded_inventory = realm_inventory.get_new_inventory_from_us_and_take_only_a_shard_part(worker_id, 1, nb_workers, worker_id, worker_weights)
            current_worker_inventory = worker_entry['inventories'][realm_name]
            worker_inventory_differences = current_worker_inventory.compute_differences_from_new_inventory(updated_worker_sharded_inventory)
            message = FromDaemonToModuleMessage(FROM_DAEMON_TO_MODULE_MESSAGE_TYPES.UPDATE_REALM, {'realm_name': realm_name, 'inventory_differences': worker_inventory_differences})
            self._send_inventory_message_to_worker(worker_id, message)
            # Save the computed inventory so we will be able to compute differences after
            worker_entry['inventories'][realm_name] = updated_worker_sharded_inventory
        # Clean the host to workers cache
        self._host_uuids_to_worker_cache.clear()
    
    
    #############################################
    #        Workers start/live part            #
    #############################################
    
    def stop_all(self, update_stop_logger=None):
        if not update_stop_logger:
            update_stop_logger = self.logger
        
        update_stop_logger.info(u'Stopping all workers.')
        
        for worker in self._workers.values():
            process = worker['process']
            if process:
                self._do_stop_process(process)
        
        BaseModule.stop_all(self, update_stop_logger)
    
    
    def _get_module_worker(self, worker_id):
        if self.MODULE_WORKER_CLASS is None:
            raise Exception('[%s] Cannot create the workers for this module because the MODULE_WORKER_CLASS is not defined in the module class.' % (self.get_name()))
        
        worker_instance = self.MODULE_WORKER_CLASS(worker_id, self.myconf, self.get_name(), self.from_module_to_main_daemon_queue, self.queue_factory, self.daemon_display_name)
        worker_instance.set_from_module_to_worker_container(self.get_from_module_to_worker_container())
        return worker_instance
    
    
    # If we are a worker based module, start our workers
    def start_workers(self, daemon_display_name='UNSET'):
        self.daemon_display_name = daemon_display_name
        self.father_pid = os.getpid()
        
        # FIRST: clean our inventories because we can be dead and restarted
        self._reset_inventories()
        
        # NOTE: we will save process objects into a list temporary (when launching all workers)
        #       and then assign them to the self.workers object so when we fork() we don't have the
        #       process object in the self, because on windows it cannot be pickle, and so the fork()
        #       will fail
        _processes = []
        
        # If we already have workers, kill them all
        for worker_id in xrange(self._nb_workers):
            worker = self._workers.get(worker_id, None)
            # Maybe there was previously workers, but we did get killed maybe
            # so
            if worker is not None:
                process = worker.get('process', None)
                if process:
                    self._do_stop_process(process)
                # NOTE: we do not need to close queues as only the Android version need
                # to close/join them when droping
            instance = self._get_module_worker(worker_id)
            worker = {
                'id'         : worker_id,
                'process'    : None,
                'inventories': {},  # realm => inventory
                'instance'   : instance,
            }
            logger_worker_manager = self.logger.get_sub_part(u'WORKER MANAGER')
            logger_worker_manager.info(u'Starting worker %s' % worker_id)
            process = Process(target=instance._worker_main)
            process.start()
            logger_worker_manager.info(u'The worker %s is now started as pid:%d' % (worker_id, process.pid))
            self._workers[worker_id] = worker
            _processes.append(process)
        
        # Set AFTER the start
        for (worker_id, process) in enumerate(_processes):
            self._workers[worker_id]['process'] = process
    
    
    def get_raw_stats(self, param=''):
        data = self.send_command('get_raw_stats')
        return data
    
    
    def send_command(self, command_name, args=None):
        data = {'workers': {}}
        for worker_id, worker in self._workers.iteritems():
            worker_result = None
            # Maybe the worker is not ready to communicate with us
            if worker['instance'].command_from_module_to_worker and worker['process'].is_alive():
                worker_result = worker['instance'].command_queue_handler.send_command(command_name, args)
            
            data['workers'][worker_id] = worker_result
        return data
    
    
    def do_loop_turn(self):
        pass
    
    
    def loop_turn(self):
        pass
    
    
    def get_from_module_to_worker_container(self):  # noqa : this methode must be overload
        # type: () -> FromModuleToWorkerContainer
        return FromModuleToWorkerContainer()
