#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2020
# This file is part of Shinken Enterprise, all rights reserved.

import base64
import httplib
import json
import urllib

import time

from shinkensolutions.lib_checks.common import Result, EXIT_STATUS, Utils, ShinkenParser


class CheckError(StandardError):
    def __init__(self, error):
        self.error = error


result = Result()


def get_option_parser():
    parser = ShinkenParser()
    parser.add_option(u'-U', u'--url', dest=u'url', default=u'http://localhost:7767', help=u'Url of the webui to test')
    parser.add_option(u'-u', u'--username', dest=u'username', default=u'admin', help=u'Shinken user')
    parser.add_option(u'-p', u'--password', dest=u'password', default=u'admin', help=u'Password of Shinken user')
    parser.add_option(u'-W', u'--warning', dest=u'warning', default=2, type=float, help=u'Define the time limit in second where the check status is WARNING')
    parser.add_option(u'-C', u'--critical', dest=u'critical', default=4, type=float, help=u'Define the time limit in second where the check status is CRITICAL')
    return parser


def make_cookie(url, user, password):
    try:
        path = '%s/%s' % (url, 'auth')
        
        try:
            urllib.urlopen(url)
        except IOError as io:
            raise CheckError('Unable to access url : %s.<br><br>Cause : %s' % (url, io.strerror))
        
        http_params = urllib.urlencode({'login': base64.b64encode(user), 'password': base64.b64encode(password)})
        cookie_token = None
        
        try:
            http_buffer = urllib.urlopen(path, http_params)
        except IOError as io:
            raise CheckError('Cannot access to %s with <strong>username</strong> "%s" and <strong>password</strong> "*****" .<br><br>Cause : [%s] --> %s ' % (url, user, io[1], io[2]))
        
        token = ('%s' % http_buffer.headers).split('\n')
        for line in token:
            if line.startswith('Set-Cookie'):
                cookie_token = line.split('"')[1]
        if not cookie_token:
            raise CheckError('Unable to find cookie in token')
        
        header = {'Cookie': 'user=%s' % cookie_token}
    except CheckError as ce:
        raise ce
    except Exception as e:
        raise CheckError('An Exception occured %s' % str(e))
    return header


def make_call(url, path, headers, body=None):
    data = ''
    try:
        http_part, host = url.split('://')
        if http_part == 'https':
            conn = httplib.HTTPSConnection(host)
        else:
            conn = httplib.HTTPConnection(host)
        conn.request('POST', path, body, headers)
        r1 = conn.getresponse()
        if r1.status != 200:
            err = r1.read()
            conn.close()
            raise CheckError('%s : %s' % (r1.status, err))
        data = r1.read()
        conn.close()
    except CheckError as ce:
        CheckError(ce)
    except Exception as e:
        raise CheckError('An Error occurred %s ' % str(e))
    return data


def main():
    parser = get_option_parser()
    opts, args = parser.parse_args()
    
    state_warning_time_limit = opts.warning
    state_critical_time_limit = opts.critical
    
    url = opts.url
    
    try:
        header = make_cookie(url, opts.username, opts.password)
    except CheckError as ce:
        result.hard_exit(EXIT_STATUS.CRITICAL, ce.error)
        return
    
    start_time = time.time()
    try:
        response = make_call(url=opts.url, path='/api/list-elements-all?range=page:0~page_size:100', headers=header, body='{}')
    except CheckError as ce:
        result.hard_exit(EXIT_STATUS.CRITICAL, ce.error)
        return
    
    call_duration = time.time() - start_time
    
    exit_status = EXIT_STATUS.OK
    if call_duration >= state_critical_time_limit:
        exit_status = EXIT_STATUS.CRITICAL
    elif call_duration >= state_warning_time_limit:
        exit_status = EXIT_STATUS.WARNING
    result.add_check(exit_status, 'Call to /api/list-elements-all done in %s' % Utils.print_human_readable_period(call_duration))
    
    try:
        data = json.loads(response)
    except ValueError as ve:
        result.hard_exit(EXIT_STATUS.CRITICAL, ve.message)
        return
    
    result.add_perf_data('call_duration', call_duration)
    result.add_perf_data('nb_element_total', data['nb_element_total'])
    result.exit()


if __name__ == '__main__':
    main()
