#!/usr/bin/env python

# Copyright (C) 2013:
#     Gabes Jean, naparuba@gmail.com
#     Pasche Sebastien, sebastien.pasche@leshop.ch
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
#

import optparse
import re
import sys
import time

from pymongo import MongoClient

from shinken.util import get_sec_from_morning, get_wday

''''
 This script is a check for lookup at memory consumption over ssh without
 having an agent on the other side
'''

VERSION = '.01'

parser = optparse.OptionParser(
    "%prog [options]", version="%prog " + VERSION)
parser.add_option('-U', '--mongodb-uri',
                  dest="uri", help='Mongodb URI')
parser.add_option('-D', '--database',
                  dest="database", help='Database to connect to')
parser.add_option('-H', '--hostname',
                  dest="hostname", help='Hostname to lookup')
parser.add_option('-C', '--check',
                  dest="check_name", help='Check name to query')
parser.add_option('-M', '--metric',
                  dest="metric", help='Metric name to query')
parser.add_option('-w', '--warning',
                  dest="warning", help='Warning value for the requested value, in % distance from the trending value. Default: 50')
parser.add_option('-c', '--critical',
                  dest="critical", help='Critical value for requested value, in % distance from the trending value. Default: 100')
parser.add_option('--chunk-interval', dest='chunk_interval', help='Chunk interval')


def get_key(hname, sdesc, metric, wday, chunk_nb):
    _metric = metric.replace('/', 'SLASH')
    _hname = hname.decode('utf8', 'ignore')
    _sdesc = sdesc.decode('utf8', 'ignore')
    r = re.compile(r'[^\w-]')
    _hname = r.sub('_', _hname)
    _sdesc = r.sub('_', _sdesc)
    
    return _hname + '.' + _sdesc + '.' + _metric + '.' + 'week' + '.' + str(wday) + '.' + 'Vtrend' + '.' + str(chunk_nb)


if __name__ == '__main__':
    # Ok first job : parse args
    opts, args = parser.parse_args()
    if args:
        parser.error("Does not accept any argument.")
    uri = opts.uri or 'mongodb://localhost/?j=false&fsync=false'
    shinken_database = opts.database or 'shinken'
    
    hostname = opts.hostname
    if not hostname:
        print "Error : hostname parameter (-H) is mandatory"
        sys.exit(2)
    check_name = opts.check_name
    if not check_name:
        print "Error : Check name parameter (-C) is mandatory"
        sys.exit(2)
    
    metric = opts.metric
    if not metric:
        print "Error : Metric name parameter (-M) is mandatory"
        sys.exit(2)
    
    chunk_interval = opts.chunk_interval
    if not chunk_interval:
        chunk_interval = 900
    else:
        chunk_interval = int(chunk_interval)
    
    warning = opts.warning
    if not warning:
        warning = 50
    warning = int(warning)
    
    critical = opts.critical
    if not critical:
        critical = 100
    critical = int(critical)
    
    # Now do the job
    try:
        con = MongoClient(uri, safe=True)
    except Exception, exp:
        print "Error: cannot connect to database: %s" % exp
        sys.exit(2)
    
    shinken_database = getattr(con, shinken_database)
    col = getattr(shinken_database, 'trending')
    
    now = int(time.time())
    sec_from_morning = get_sec_from_morning(now)
    wday = get_wday(now)
    chunk_nb = sec_from_morning / chunk_interval
    
    # print chunk_nb, wday, sec_from_morning
    
    key = get_key(hostname, check_name, metric, wday, chunk_nb)
    # print key
    
    e = col.find_one({'_id': key})
    
    if e is None:
        print "UNKNOWN: there is currently no computed value for this chunk period"
        sys.exit(3)
    VcurrentSmooth = e.get('VcurrentSmooth')
    VtrendSmooth = e.get('VtrendSmooth')
    
    # print "Current Smooth", VcurrentSmooth
    # print "Trend Smooth", VtrendSmooth
    if VcurrentSmooth == None or VtrendSmooth == None or VtrendSmooth == 0:
        print "UNKNOWN: the current or trending values are missing"
        sys.exit(3)
    
    diff = VcurrentSmooth * 100 / VtrendSmooth
    # print diff
    abs_diff = abs(diff)
    
    perfdata = '%s_currentsmooth=%.4f   %s_trendingsmooth=%.4f' % (metric, VcurrentSmooth, metric, VtrendSmooth)
    
    # print abs_diff, critical
    if abs_diff > critical:
        print "Critical: the metric %s/%s/%s is getting too far (%.0f%% more/less than %s%%) from the reference value: current:%.4f reference:%.4f | %s" % (hostname, check_name, metric, diff, critical, VcurrentSmooth, VtrendSmooth, perfdata)
        sys.exit(2)
    
    if abs_diff > warning:
        print "Warning: the metric %s/%s/%s is getting too far (%.0f%% more/less than %s%%) from the reference value: current:%.4f reference:%.4f | %s" % (hostname, check_name, metric, diff, warning, VcurrentSmooth, VtrendSmooth, perfdata)
        sys.exit(1)
    
    print "OK: the metric %s/%s/%s is in the valid bounds (%.0f%% with limits %s%% and %s%%) from the reference value: current:%.4f reference:%.4f | %s" % (hostname, check_name, metric, diff, warning, critical, VcurrentSmooth, VtrendSmooth, perfdata)
    sys.exit(0)
