#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.
import io
import optparse
import subprocess
from collections import namedtuple
from sys import exit

from pymongo import MongoClient

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.localinstall import get_local_instances_for_type

try:
    from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
    from shinken.synchronizer.dao.def_items import ITEM_STATE, ITEM_TYPE, SERVICE_OVERRIDE
    from shinken.synchronizer.dao.helpers import get_name_from_type, safe_add_to_dict
    from shinken.synchronizer.dao.validators.validator import Validator
except ImportError:
    from synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
    from synchronizer.dao.def_items import ITEM_STATE, ITEM_TYPE, SERVICE_OVERRIDE
    from synchronizer.dao.helpers import get_name_from_type, safe_add_to_dict
    from synchronizer.dao.validators.validator import Validator

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Dict, Any, NoReturn, Tuple
    from typing import TextIO

CHECK_NOT_FOUND = u'check not found'
LOG_FILE = 'delete_invalid_service_overrides.log'


class SanatizeException(Exception):
    pass


def system_call(command):
    p = subprocess.Popen([command], stdout=subprocess.PIPE, shell=True)
    p.wait()
    return p.returncode


def print_line(title, color, s):
    lines = cut_line(s)
    title = '%-10s' % title
    print ' ' * 4 + '\033[%dm%s\033[0m %s' % (color, title, lines[0])
    if len(lines) > 1:
        for line in lines[1:]:
            print ' ' * 5 + ' ' * len(title) + line


def cut_line(s):
    size = 80
    lines = []
    words = s.split(' ')
    cline = ''
    for word in words:
        # Maybe this is too large already, put the line
        if len(cline) + len(word) > size:
            lines.append(cline)
            cline = ''
        cline += ' ' + word
    lines.append(cline)
    return lines


def get_synchronizer_db():
    con = MongoClient()
    db = con.synchronizer
    return db


def print_or_delete_invalid_service_overrides(only_summary):
    # type: (bool) -> bool
    
    if _does_synchronizer_needs_to_stop(only_summary):
        print_line(u'ERROR: Please stop the synchronizer before running this sanatize', 31, '')
        return False
    
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    override_forbidden_properties = Validator(None, None).OVERRIDE_FORBIDDEN_PROPERTIES
    data_provider_mongo = DataProviderMongo(db)
    links_to_remove = {}
    
    all_checks = []
    for service in ITEM_TYPE.ALL_SERVICES:
        all_checks.extend(data_provider_mongo.find_items(service, ITEM_STATE.STAGGING))
    all_checks = dict([(item[u'_id'], item) for item in all_checks])
    
    for item_state in [ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING]:
        for host_type in ITEM_TYPE.ALL_HOST_CLASS:
            all_hosts = data_provider_mongo.find_items(host_type, item_state)
            for host in all_hosts:
                host_name_and_id = u'%s - %s' % (host.get(u'host_name', None), host.get(u'_id', None))
                
                service_override = host.get(SERVICE_OVERRIDE, {})
                if not service_override:
                    continue
                
                links_to_remove[host_name_and_id] = {}
                service_override_check_name_and_prop = []
                service_override_entries = service_override.get(u'links', [])
                for service_override_entry_to_remove in service_override_entries:
                    check_link = service_override_entry_to_remove.get(u'check_link', {})
                    prop_name = service_override_entry_to_remove.get(u'key', u'')
                    
                    link_to_remove_entry = {
                        u'host'                            : host,
                        u'reason'                          : u'',
                        u'prop_name'                       : prop_name,
                        u'service_override_entry_to_remove': service_override_entry_to_remove,
                        u'item_state'                      : item_state,
                        u'item_type'                       : host_type
                    }
                    
                    try:
                        check_name = _find_check_name(host_name_and_id, check_link, service_override_entry_to_remove, all_checks, links_to_remove, link_to_remove_entry)
                    except SanatizeException:
                        continue
                    
                    service_override_check_name_and_prop.append((check_name, prop_name))
                    
                    if service_override_entry_to_remove:
                        if u'value' not in service_override_entry_to_remove:
                            link_to_remove_entry[u'reason'] = u'have no value'
                            safe_add_to_dict(links_to_remove[host_name_and_id], check_name, link_to_remove_entry)
                            continue
                        
                        if prop_name in override_forbidden_properties:
                            link_to_remove_entry[u'reason'] = u'have override on forbidden property'
                            safe_add_to_dict(links_to_remove[host_name_and_id], check_name, link_to_remove_entry)
                            continue
                        
                        value = service_override_entry_to_remove[u'value']
                        if prop_name == u'check_command_args' and value and isinstance(value, dict):
                            link_to_remove_entry[u'reason'] = u'have invalid override : %s' % value
                            safe_add_to_dict(links_to_remove[host_name_and_id], check_name, link_to_remove_entry)
                
                duplicates_indexes = _get_duplicates_indexes(service_override_check_name_and_prop)
                for duplicate, dup_indexes in duplicates_indexes.iteritems():
                    index_service_override_to_keep = dup_indexes.pop()
                    
                    new_value = service_override_entries[index_service_override_to_keep][u'value']
                    prop_name = duplicate[1]
                    check_name = duplicate[0]
                    
                    link_to_remove_entry = {
                        u'host'                              : host,
                        u'reason'                            : u'is duplicated',
                        u'reason_code'                       : u'duplicate',
                        u'prop_name'                         : prop_name,
                        u'service_override_entry_to_keep'    : service_override_entries[index_service_override_to_keep],
                        u'service_override_entries_to_remove': [],
                        u'is_same_value'                     : False,
                        u'item_state'                        : item_state,
                        u'item_type'                         : host_type
                    }
                    
                    for dup_index in dup_indexes:
                        service_override_entry_to_remove = service_override_entries[dup_index]
                        link_to_remove_entry[u'service_override_entries_to_remove'].append(service_override_entry_to_remove)
                        old_value = service_override_entry_to_remove[u'value']
                        if not new_value == old_value:
                            link_to_remove_entry[u'is_same_value'] = False
                    
                    safe_add_to_dict(links_to_remove[host_name_and_id], check_name, link_to_remove_entry)
    
    if not only_summary:
        _display_summary_and_ask_for_link_choice(links_to_remove, is_remove_mode=True)
        g_did_run = _delete_service_overrides_from_mongo(links_to_remove, data_provider_mongo)
    
    logfile = None
    try:
        logfile = io.open(LOG_FILE, u'w+', encoding=u'UTF-8')
        _display_summary_and_ask_for_link_choice(links_to_remove, logfile=logfile)
    finally:
        logfile.close()
    
    return g_did_run


def _does_synchronizer_needs_to_stop(only_summary):
    # type: (bool) -> bool
    if only_summary:
        return False
    _activated_daemons = [_id for _id, enable in get_local_instances_for_type(u'synchronizer') if enable]
    if _activated_daemons:
        code = system_call(u'/etc/init.d/shinken-%s status' % u'synchronizer')
        if code == 0:
            return True
    return False


def _find_check_name(host_name, check_link, service_override_entry_to_remove, all_checks, links_to_remove, link_to_remove_entry):
    if not check_link:
        link_to_remove_entry[u'reason'] = CHECK_NOT_FOUND
        safe_add_to_dict(links_to_remove[host_name], CHECK_NOT_FOUND, link_to_remove_entry)
        raise SanatizeException(link_to_remove_entry[u'reason'])
    
    if u'exists' not in check_link:
        link_to_remove_entry[u'reason'] = u'exists not in check_link'
        safe_add_to_dict(links_to_remove[host_name], CHECK_NOT_FOUND, link_to_remove_entry)
        raise SanatizeException(link_to_remove_entry[u'reason'])
    
    if check_link[u'exists']:
        check_id = check_link.get(u'_id', u'')
        if not check_id:
            link_to_remove_entry[u'reason'] = u'id not in check_link and exists is true'
            safe_add_to_dict(links_to_remove[host_name], CHECK_NOT_FOUND, link_to_remove_entry)
            raise SanatizeException(link_to_remove_entry[u'reason'])
        
        check = all_checks.get(check_id, {})
        if not check:
            link_to_remove_entry[u'reason'] = u'not check with uuid %s was found' % check_id
            safe_add_to_dict(links_to_remove[host_name], CHECK_NOT_FOUND, link_to_remove_entry)
            raise SanatizeException(link_to_remove_entry[u'reason'])
        
        check_name = get_name_from_type(check[u'@metadata'][u'type'], check)
        
        dfe_key = service_override_entry_to_remove.get(u'dfe_key', None)
        if check.get(u'duplicate_foreach', None) and not dfe_key:
            link_to_remove_entry[u'reason'] = u'is on dfe check but do not have dfe_key'
            safe_add_to_dict(links_to_remove[host_name], u'check [ %s ] with id [ %s ]' % (check_name, check_id), link_to_remove_entry)
            raise SanatizeException(link_to_remove_entry[u'reason'])
        
        if u'$KEY$' in check_name:
            check_name = check_name.replace(u'$KEY$', dfe_key)
    
    else:
        check_name = check_link.get(u'name', u'')
        if not check_name:
            link_to_remove_entry[u'reason'] = u'name not in check_link and exists is false'
            safe_add_to_dict(links_to_remove[host_name], CHECK_NOT_FOUND, link_to_remove_entry)
            raise SanatizeException(link_to_remove_entry[u'reason'])
    
    return check_name


def _display_summary_and_ask_for_link_choice(summary, is_remove_mode=False, logfile=None):
    # type: (Dict[Any], bool, TextIO) -> NoReturn
    i = 0
    for host_name, by_checks in summary.iteritems():
        if not by_checks:
            continue
        
        if is_remove_mode and not next((True for link_to_remove_entries in by_checks.itervalues() if not _has_only_duplicate_entries_with_same_value(link_to_remove_entries)), False):
            continue
        
        _print_and_write_to_file(u'┌' + u'─' * 30, not is_remove_mode, logfile)
        _print_and_write_to_file(u'│host : %s' % host_name, not is_remove_mode, logfile)
        
        for check_name, link_to_remove_entries in by_checks.iteritems():
            
            if is_remove_mode and _has_only_duplicate_entries_with_same_value(link_to_remove_entries):
                continue
            
            _print_and_write_to_file(u'│\tcheck_name : %s' % check_name, not is_remove_mode, logfile)
            
            for link_to_remove_entry in link_to_remove_entries:
                if is_remove_mode and link_to_remove_entry.get(u'is_same_value', False):
                    continue
                
                i += 1
                if is_remove_mode and link_to_remove_entry.get(u'reason_code', u'') == u'duplicate' and link_to_remove_entry.get(u'service_override_entry_to_keep', None):
                    prop_name = link_to_remove_entry[u'prop_name']
                    default_choice = len(link_to_remove_entry[u'service_override_entries_to_remove'])
                    user_choice = _ask_user_confirmation(prop_name, link_to_remove_entry) - 1
                    if not user_choice == default_choice:
                        link_to_remove_entry[u'service_override_entry_to_keep'], link_to_remove_entry[u'service_override_entries_to_remove'][user_choice] = \
                            link_to_remove_entry[u'service_override_entries_to_remove'][user_choice], link_to_remove_entry[u'service_override_entry_to_keep']
                
                else:
                    message = link_to_remove_entry[u'reason']
                    prop_name = link_to_remove_entry[u'prop_name']
                    _print_and_write_to_file(u'│\t\t• Service override to remove on [ %s ] %s' % (prop_name, message), not is_remove_mode, logfile)
                    
                    for service_override_entry_to_remove in link_to_remove_entry[u'service_override_entries_to_remove']:
                        value = service_override_entry_to_remove[u'value']
                        _print_and_write_to_file(u'│\t\t\t \033[31m• (Old value) : %s\033[0m' % value, not is_remove_mode, logfile)
                    
                    new_value = link_to_remove_entry[u'service_override_entry_to_keep'][u'value']
                    _print_and_write_to_file(u'│\t\t\t \033[32m• (last update) : %s\033[0m' % new_value, not is_remove_mode, logfile)
            
            _print_and_write_to_file(u'│', not is_remove_mode, logfile)
        
        _print_and_write_to_file(u'└' + u'─' * 30, not is_remove_mode, logfile)
    
    if i:
        _print_and_write_to_file(u'service override cleaned : %s' % i, not is_remove_mode, logfile)
    if not is_remove_mode:
        logfile.close()


def _print_and_write_to_file(message, should_write_to_file, file_stream):
    # type: (unicode, bool, TextIO) -> NoReturn
    print(message)
    if should_write_to_file:
        file_stream.write(u'%s\n' % message)


def _delete_service_overrides_from_mongo(modified_summary, data_provider_mongo):
    # type: (Dict, DataProviderMongo) -> bool
    MongoEntry = namedtuple(u'MongoEntry', [u'host_uuid', u'item_type', u'item_state'])
    to_remove_mongo_entries = {}  # type: Dict[MongoEntry, Dict]
    
    g_did_run = False
    for host_name, by_checks in modified_summary.iteritems():
        if not by_checks:
            continue
        
        for check_name, link_to_remove_entries in by_checks.iteritems():
            for link_to_remove_entry in link_to_remove_entries:
                host = link_to_remove_entry[u'host']
                for service_override_to_remove in link_to_remove_entry[u'service_override_entries_to_remove']:
                    if service_override_to_remove in host[SERVICE_OVERRIDE][u'links']:
                        host[SERVICE_OVERRIDE][u'links'].remove(service_override_to_remove)
                    if not host[SERVICE_OVERRIDE][u'links']:
                        del host[SERVICE_OVERRIDE]
                    to_remove_mongo_entries[MongoEntry(host[u'_id'], link_to_remove_entry[u'item_type'], link_to_remove_entry[u'item_state'])] = host
    
    for mongo_entry, host in to_remove_mongo_entries.iteritems():
        data_provider_mongo.save_item(host, mongo_entry.item_type, mongo_entry.item_state)
        g_did_run = True
    return g_did_run


def _has_only_duplicate_entries_with_same_value(link_to_remove_entries):
    for link_to_remove_entry in link_to_remove_entries:
        if not link_to_remove_entry.get(u'is_same_value', False):
            return False
    return True


def _ask_user_confirmation(prop_name, link_to_remove_entry):
    # type: (unicode, Dict) -> int
    answer = u'None'
    print(u'│\t\tService override on [ %s ] is duplicate' % prop_name)
    cpt = 1
    for service_override_to_remove in link_to_remove_entry[u'service_override_entries_to_remove']:
        value = service_override_to_remove[u'value']
        print(u'│\t\t\t %d ➤ (Old value) : %s' % (cpt, value))
        cpt += 1
    value = link_to_remove_entry[u'service_override_entry_to_keep'][u'value']
    print(u'│\t\t\t\033[33m %d ➤ (last update) : %s\033[0m' % (cpt, value))
    while not answer.lower() in unicode(range(1, cpt + 1)):
        print(u'│ Choose a value to keep [ 1 to %d ] :' % cpt)
        answer = raw_input()
    if answer == u'':
        return cpt
    return int(answer)


def _get_duplicates_indexes(service_override_check_name_and_prop):
    # type: (List[Tuple[unicode, unicode]]) -> Dict[Tuple[unicode, unicode], List[int]]
    duplicates = []
    for check_name_and_prop in service_override_check_name_and_prop:
        count = service_override_check_name_and_prop.count(check_name_and_prop)
        if count > 1:
            duplicates.append(check_name_and_prop)
    
    duplicate_indexes = {}
    for duplicate in duplicates:
        duplicate_indexes[duplicate] = [index for index, value in enumerate(service_override_check_name_and_prop) if value == duplicate]
    return duplicate_indexes


if __name__ == '__main__':
    parser = optparse.OptionParser("%prog ", description='This script is used to remove invalid service overrides or just display it.')
    parser.add_option('', '--only-summary', dest='only_summary', action='store_true', default=False, help="use by fix 'delete_invalid_service_overrides' to only display the summary")
    opts, _ = parser.parse_args()
    try:
        print_or_delete_invalid_service_overrides(opts.only_summary)
    except KeyboardInterrupt:
        print('\n\033[33mInterrupted by keyboard\033[0m')
        exit(130)
