#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.


import logging
import logging.handlers
import os

import sys

from shinkensolutions.cfg_formatter.cfg_formatter import format_config_file, ShinkenCfgFormatterException, FORMATTER_FILES
from .sanatize_utils import get_file_md5

try:
    import pwd
    import grp
except ImportError:
    # don't expect to have this on windows :)
    pwd = grp = None

try:
    import shinken
except ImportError:
    print 'Cannot import shinken lib, please install it before launching this tool'
    sys.exit(2)

from shinken.log import logger
from shinken.objects.config import Config
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Set, Dict, List


# UTILS

def __load_full_configuration():
    # Launch the configuration, but without logging
    old_level = logger.level
    logger.setLevel(logging.FATAL)
    cfg_path = '/etc/shinken/shinken.cfg'
    conf = Config()
    buf = conf.read_config([cfg_path])
    if not conf.conf_is_correct:
        logger.setLevel(old_level)
        err = 'The sanatize can\'t load default configuration at [ %s ].' % cfg_path
        raise Exception(err)
    
    logger.setLevel(old_level)
    
    raw_objects = conf.read_config_buf(buf, [])
    conf.load_packs()
    return raw_objects


def format_and_check_config_file_modification(module, updated_files_path, cfg_params_new_name_mapping, already_written_paths, cfg_file_name=u'', need_old_cfg=False, is_override=False):
    # type: (Dict, Dict[unicode,bool], Dict[unicode, unicode], Set[unicode], unicode, bool, bool) -> None
    
    if is_override:
        module_cfg_file_path = module.get(u'imported_from', u'')
    else:
        module_imported_from = module.get(u'imported_from', u'')[0]  # type: unicode
        module_cfg_file_path, _ = module_imported_from.strip().split(u':', 1)
    
    file_md5 = get_file_md5(module_cfg_file_path)
    format_config_file(module, cfg_params_new_name_mapping, already_written_paths, cfg_file_name=cfg_file_name, is_old_cfg=need_old_cfg)
    if not file_md5 == get_file_md5(module_cfg_file_path):
        updated_files_path[module_cfg_file_path] = True
    already_written_paths.add(module_cfg_file_path)


def _generate_new_cfg_from_old_one(modules, module_type, cfg_params_new_name_mapping, cfg_file_name=u'', need_old_cfg=False):
    # type: (List, unicode, Optional[Dict], unicode, bool)-> Dict[unicode, bool]
    modules_to_reformat = [m for m in modules if m.get(u'module_type', None) == [module_type]]
    # In case we already have written a path, then it means that multiple module are in the same file
    already_written_paths = set()  # type: Set[unicode]
    
    updated_files_path = {}
    for module in modules_to_reformat:
        overrides = module.get(u'overrides', [])
        
        format_and_check_config_file_modification(module, updated_files_path, cfg_params_new_name_mapping, already_written_paths, cfg_file_name=cfg_file_name, need_old_cfg=need_old_cfg)
        
        for override_config in overrides:
            format_and_check_config_file_modification(override_config, updated_files_path, cfg_params_new_name_mapping, already_written_paths, need_old_cfg=need_old_cfg, is_override=True)
    
    return updated_files_path


def revert_configuration_files_from_new_format(modules, module_type, cfg_params_new_name_mapping, cfg_file_name=u''):
    # type: (List, unicode, Optional[Dict], unicode)-> Dict[unicode, bool]
    return _generate_new_cfg_from_old_one(modules, module_type, cfg_params_new_name_mapping, cfg_file_name=cfg_file_name, need_old_cfg=True)


def convert_configuration_files_to_new_format(modules, module_type, cfg_params_new_name_mapping, cfg_file_name=u''):
    # type: (List, unicode, Optional[Dict], unicode)-> Dict[unicode, bool]
    return _generate_new_cfg_from_old_one(modules, module_type, cfg_params_new_name_mapping, cfg_file_name=cfg_file_name)


def set_default_values_in_cfg_files(daemon_type, key_to_check, daemon_cfg_dir, daemon_default_values):
    g_run = False
    for daemon_config_file in os.listdir(daemon_cfg_dir):
        cfg_file = os.path.join(daemon_cfg_dir, daemon_config_file)
        if os.path.isdir(cfg_file):
            continue
        if os.path.islink(cfg_file):
            target = os.readlink(cfg_file)
            if not os.path.exists(target):
                continue  # the target doesn't exist
            if os.path.realpath(cfg_file) == cfg_file:
                continue  # It's pointing to itself, abort!
        in_define = False
        with open(cfg_file) as sync:
            lines = sync.readlines()
            for line_no, line in enumerate(lines):
                orig_line = line.rstrip()
                line = line.strip()
                if line.startswith('#'):
                    continue
                
                if line.startswith('define') and daemon_type in line:
                    in_define = True
                
                if in_define and line.startswith(key_to_check):
                    values = set([src.strip() for src in line.replace(key_to_check, '').split(',')])
                    
                    for expected_value in daemon_default_values:
                        if expected_value not in values:
                            orig_line = "%s,%s" % (orig_line, expected_value)
                            g_run = True
                
                if line.startswith('}'):
                    in_define = False
                
                lines[line_no] = "%s\n" % orig_line
        with open(cfg_file, 'w') as sync:
            sync.writelines(lines)
    return g_run


def convert_broker_module_livedata_configuration_files_to_new_format(modules):
    # type:(List[unicode]) -> Dict[unicode, bool]
    
    broker_module_livedata_cfg_params_mapping = {
        u'broker__module_livedata__lang'                                                               : [u'lang', u'broker__module_livedata__lang'],
        u'broker__module_livedata__listening_address'                                                  : [u'host', u'broker__module_livedata__listening_address'],
        u'broker__module_livedata__listening_port'                                                     : [u'port', u'broker__module_livedata__listening_port'],
        u'broker__module_livedata__use_ssl'                                                            : [u'use_ssl', u'broker__module_livedata__use_ssl'],
        u'broker__module_livedata__ssl_cert'                                                           : [u'ssl_cert', u'broker__module_livedata__ssl_cert'],
        u'broker__module_livedata__ssl_key'                                                            : [u'ssl_key', u'broker__module_livedata__ssl_key'],
        u'broker__module_livedata__broks_getter__activate_late_set_catchup'                            : [u'broker_module_livedata__broks_getter__activate_late_set_catchup', u'broker__module_livedata__broks_getter__activate_late_set_catchup'],
        u'broker__module_livedata__broks_getter__nb_late_set_allowed_before_catchup'                   : [u'broker_module_livedata__broks_getter__nb_late_set_allowed_before_catchup',
                                                                                                          u'broker__module_livedata__broks_getter__nb_late_set_allowed_before_catchup'],
        u'broker__module_livedata__broks_getter__catchup_broks_managed_by_module_in_a_catchup_loop'    : [u'broker_module_livedata__broks_getter__catchup_broks_managed_by_module_in_a_catchup_loop',
                                                                                                          u'broker__module_livedata__broks_getter__catchup_broks_managed_by_module_in_a_catchup_loop'],
        u'broker__module_livedata__broks_getter__catchup_run_endless_until_nb_late_set_allowed_reached': [u'broker_module_livedata__broks_getter__catchup_run_endless_until_nb_late_set_allowed_reached',
                                                                                                          u'broker__module_livedata__broks_getter__catchup_run_endless_until_nb_late_set_allowed_reached'],
        u'broker__module_livedata__broks_getter__include_deserialisation_and_catchup_in_lock'          : [u'broker_module_livedata__broks_getter__include_deserialisation_and_catchup_in_lock',
                                                                                                          u'broker__module_livedata__broks_getter__include_deserialisation_and_catchup_in_lock'],
        u'broker__module_livedata__token'                                                              : [u'token', u'broker__module_livedata__token'],
    }
    
    return convert_configuration_files_to_new_format(modules, u'broker_module_livedata', broker_module_livedata_cfg_params_mapping, cfg_file_name=FORMATTER_FILES.BROKER_MODULE_LIVEDATA)


def convert_livedata_module_sla_provider_configuration_files_to_new_format(modules):
    # type:(List[unicode]) -> Dict[unicode, bool]
    
    livedata_module_sla_provider_cfg_params_mapping = {
        u'broker__module_livedata__module_sla_provider__database__uri'                                                 : [u'livedata_module_sla_provider__database__uri', u'broker__module_livedata__module_sla_provider__database__uri'],
        u'broker__module_livedata__module_sla_provider__database__name'                                                : [u'livedata_module_sla_provider__database__name', u'broker__module_livedata__module_sla_provider__database__name'],
        u'broker__module_livedata__module_sla_provider__use_ssh_tunnel'                                                : [u'livedata_module_sla_provider__use_ssh_tunnel', u'broker__module_livedata__module_sla_provider__use_ssh_tunnel'],
        u'broker__module_livedata__module_sla_provider__ssh_user'                                                      : [u'livedata_module_sla_provider__ssh_user', u'broker__module_livedata__module_sla_provider__ssh_user'],
        u'broker__module_livedata__module_sla_provider__ssh_keyfile'                                                   : [u'livedata_module_sla_provider__ssh_keyfile', u'broker__module_livedata__module_sla_provider__ssh_keyfile'],
        u'broker__module_livedata__module_sla_provider__ssh_tunnel_timeout'                                            : [u'livedata_module_sla_provider__ssh_tunnel_timeout', u'broker__module_livedata__module_sla_provider__ssh_tunnel_timeout'],
        u'broker__module_livedata__module_sla_provider__database__retry_connection_X_times_before_considering_an_error': [u'livedata_module_sla_provider__database__retry_connection_X_times_before_considering_an_error',
                                                                                                                          u'broker__module_livedata__module_sla_provider__database__retry_connection_X_times_before_considering_an_error'],
        u'broker__module_livedata__module_sla_provider__database__wait_X_seconds_before_reconnect'                     : [u'livedata_module_sla_provider__database__wait_X_seconds_before_reconnect',
                                                                                                                          u'broker__module_livedata__module_sla_provider__database__wait_X_seconds_before_reconnect'],
        u'broker__module_livedata__module_sla_provider__no_data_period'                                                : [u'livedata_module_sla_provider__no_data_period', u'broker__module_livedata__module_sla_provider__no_data_period'],
    }
    
    return convert_configuration_files_to_new_format(modules, u'livedata_module_sla_provider', livedata_module_sla_provider_cfg_params_mapping, cfg_file_name=FORMATTER_FILES.LIVEDATA_MODULE_SLA_PROVIDER)
