# !/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.
import hashlib
import json
import logging
import os
import re
import subprocess
import tarfile

import sys
import time
from cStringIO import StringIO
from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects import Module as ShinkenModuleDefinition
from shinken.objects.config import Config
from shinkensolutions.localinstall import POSSIBLE_DAEMONS, get_local_instances_for_type, get_local_daemons
from shinkensolutions.ssh_mongodb.mongo_client import MongoClient
from shinkensolutions.ssh_mongodb.mongo_conf import MongoConf
from shinkensolutions.ssh_mongodb.sshtunnelmongomgr import mongo_by_ssh_mgr

try:
    from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2
    from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
except ImportError:
    from synchronizer.dao.datamanagerV2 import DataManagerV2
    from synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo

if TYPE_CHECKING:
    from shinken.misc.type_hint import Union, Tuple, Optional, Dict
    from shinkensolutions.ssh_mongodb.mongo_collection import MongoCollection

success = {}
warnings = {}
errors = {}

parts = set()  # list of all parts possible
mongo_by_ssh_mgr.set_ssh_tunnel_manager_for_check()

licence_data = None


class SANATIZE_MONGO(object):
    SYNCHRONIZER_INFO_COLLECTION = u'synchronizer-info'
    DATABASE_INFO_DOCUMENT = u'database_info'


def connect_to_mongo(mongo_host, mongo_port, use_ssh, ssh_key, ssh_user):
    global sanitize_data_mongo_synchronizer_conf, sanitize_data_mongo_broker_conf
    sanitize_data_mongo_synchronizer_conf = SanatizeDataMongoConf(u'synchronizer', mongo_host=mongo_host, mongo_port=mongo_port, mongo_use_ssh=use_ssh, mongo_ssh_key=ssh_key, mongo_ssh_user=ssh_user)
    sanitize_data_mongo_broker_conf = SanatizeDataMongoConf(u'shinken', mongo_host=mongo_host, mongo_port=mongo_port, mongo_use_ssh=use_ssh, mongo_ssh_key=ssh_key, mongo_ssh_user=ssh_user)


def system_call(command):
    p = subprocess.Popen([command], stdout=subprocess.PIPE, shell=True)
    p.wait()
    return p.returncode


def is_shinken_running():
    for daemon in POSSIBLE_DAEMONS:
        _activated_daemons = [id for id, enable in get_local_instances_for_type(daemon) if enable]
        if _activated_daemons:
            code = system_call('/etc/init.d/shinken-%s status' % daemon)
            if code == 0:
                return True
    return False


def add_success(s, part='unknown'):
    parts.add(part)
    if part not in success:
        success[part] = []
    success[part].append(s)


def add_warning(s, part='unknown'):
    parts.add(part)
    if part not in warnings:
        warnings[part] = []
    warnings[part].append(s)


def is_daemon_node(dtype):
    local_daemons = get_local_daemons()
    if 'central' in local_daemons or dtype in local_daemons:
        return True
    return False


def cut_line(s):
    size = 80
    lines = []
    words = s.split(' ')
    cline = ''
    for word in words:
        # Maybe this is too large already, put the line
        if len(cline) + len(word) > size:
            lines.append(cline)
            cline = ''
        cline += ' ' + word
    lines.append(cline)
    return lines


def print_line(title, color, s):
    lines = cut_line(s)
    title = '%-10s' % title
    print ' ' * 4 + '\033[%dm%s\033[0m %s' % (color, title, lines[0])
    if len(lines) > 1:
        for line in lines[1:]:
            print ' ' * 5 + ' ' * len(title) + line


def show_warnings(part):
    for (p, l) in warnings.iteritems():
        if p != part:
            continue
        title = 'AT RISK: '
        color = 35
        for s in l:
            print_line(title, color, s)


def add_error(s, part='unknown'):
    parts.add(part)
    if part not in errors:
        errors[part] = []
    errors[part].append(s)


def show_errors(part):
    for (p, l) in errors.iteritems():
        if p != part:
            continue
        title = 'ERROR: '
        color = 31
        for s in l:
            print_line(title, color, s)


def show_success(part):
    for (p, l) in success.iteritems():
        if p != part:
            continue
        title = 'OK: '
        color = 32
        for s in l:
            print_line(title, color, s)


def compute_diff(obj1, obj2):
    NOT_TO_LOOK = ['__SYNC_IDX__', 'import_date', 'sources', 'source_order',
                   '_SYNC_KEYS', 'overwrited_protected', '_id', 'source_strong_overwrite',
                   'imported_from', '_SE_UUID', '_SE_UUID_HASH', 'last_modification']
    changed = []
    for k in set(obj1.iterkeys()).union(obj2.iterkeys()):
        if k in NOT_TO_LOOK:
            continue
        vobj1 = obj1.get(k, None)
        vobj2 = obj2.get(k, None)
        # catch the same alias for null==__DEFAULT_NO_TEMPLATE__
        if vobj1 in ('null', '__DEFAULT_NO_TEMPLATE__') and vobj2 in ('null', '__DEFAULT_NO_TEMPLATE__'):
            continue
        if vobj1 != vobj2:
            changed.append(k)
    return changed


def add_history_info(item, old_item):
    last_modif = dict()
    last_modif['contact'] = '-1'
    last_modif['contact_name'] = 'shinken-core'
    last_modif['action'] = 'ELEMENT_MODIFICATION'
    last_modif['date'] = time.time()
    last_modif['change'] = []
    if old_item:
        for k in compute_diff(item, old_item):
            if k != "last_modification":
                last_modif['change'].append({"prop": k, "old": old_item.get(k, ''), "new": item.get(k, '')})
    
    if last_modif['change']:
        item['last_modification'] = last_modif


def show_avance(pct):
    print '\r[',
    print '.' * pct, ' ' * (100 - pct),
    print '] %d%%' % pct,
    sys.stdout.flush()
    if pct == 100:
        print ''


def write_support_output(data, p):
    data = json.dumps(data)
    out = tarfile.open(p, mode='w:gz')
    try:
        info = tarfile.TarInfo('heath.json')
        info.size = len(data)
        out.addfile(info, StringIO(data))
    finally:
        out.close()
        print '\n\033[32mSupport ouput file is available: %s\033[0m' % (p)


def natural_key(string_):
    l = []
    for s in re.split(r'(\d+)', string_):
        if s.isdigit():
            l.append(int(s))
        else:
            l.append(s.lower())
    return l


def natural_version(f):
    return natural_key(f.version)


def do_match_daemons(f):
    needed_daemons = f.context_daemons
    if not needed_daemons:
        return True
    
    for daemon_type in needed_daemons:
        local_instances = [d for d in get_local_instances_for_type(daemon_type) if d[1] is True]
        if local_instances:
            return True
    return False


class SanatizeDataMongoConf(MongoConf):
    def __init__(self, database_name, mongo_host=None, mongo_port=None, mongo_use_ssh=None, mongo_ssh_key=None, mongo_ssh_user=None):
        conf = ShinkenModuleDefinition(
            {
                'mongodb_uri'                             : u'mongodb://%s%s/?fsync=false' % ((mongo_host if mongo_host else u'localhost'), (u'' if mongo_port is None else (u':%s' % mongo_port))),
                'mongodb_replica_set'                     : u'',
                'mongodb_use_ssh_tunnel'                  : u'1' if mongo_use_ssh else u'0',
                'mongodb_ssh_keyfile'                     : mongo_ssh_key if mongo_ssh_key else u'~shinken/.ssh/id_rsa',
                'mongodb_ssh_user'                        : mongo_ssh_user if mongo_ssh_user else u'shinken',
                'mongodb_use_ssh_retry_failure'           : u'0',
                'mongodb_ssh_tunnel_timeout'              : u'0',
                'mongodb_auto_reconnect_max_try'          : u'0',
                'mongodb_auto_reconnect_sleep_between_try': u'0'
            }
        )
        super(SanatizeDataMongoConf, self).__init__(conf, default_database=database_name)


sanitize_data_mongo_synchronizer_conf = SanatizeDataMongoConf(u'synchronizer')
sanitize_data_mongo_broker_conf = SanatizeDataMongoConf(u'shinken')


# TODO: look at real db path
def get_synchronizer_db():
    # type: () -> MongoClient
    db = MongoClient(sanitize_data_mongo_synchronizer_conf)
    db.init(u'sanatize-data')
    return db


# TODO: look at real db path
def get_broker_db():
    # type: () -> MongoClient
    db = MongoClient(sanitize_data_mongo_broker_conf)
    db.init(u'sanatize-data')
    return db


def __load_full_configuration():
    # Launch the configuration, but without logging
    old_level = logger.level
    logger.setLevel(logging.FATAL)
    cfg_path = '/etc/shinken/shinken.cfg'
    conf = Config()
    buf = conf.read_config([cfg_path])
    if not conf.conf_is_correct:
        logger.setLevel(old_level)
        err = 'The sanatize can\'t load default configuration at [ %s ].' % cfg_path
        raise Exception(err)
    
    logger.setLevel(old_level)
    
    raw_objects = conf.read_config_buf(buf, [])
    conf.load_packs()
    return raw_objects


CONTEXT_PATH = '/var/lib/shinken/context.json'
VERSION = '2.03.02'

if os.path.exists(CONTEXT_PATH):
    context = json.loads(open(CONTEXT_PATH, 'r').read())
    VERSION = context.get('current_version', VERSION).split('-')[0]

# ENV variable to force unset at all cost
unset_variables = ['http_proxy', 'https_proxy', 'ftp_proxy']
for v in os.environ:
    if v.lower() in unset_variables:
        os.environ[v] = ''


def get_datamanager_v2_from_data_provider_metadata(mongo_database):
    # type: (MongoClient) -> Union[None, DataManagerV2]
    try:
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2
        from shinken.synchronizer.dao.dataprovider.dataprovider_metadata import DataProviderMetadata
        from shinken.synchronizer.dao.trash_manager.trash_manager import trash_manager
    except ImportError:
        return None
    
    trash_manager.set_mongo(mongo_database)
    data_provider_mongo = DataProviderMongo(mongo_database)
    dataprovider = DataProviderMetadata(data_provider_mongo)
    dataprovider.load_from_data_provider()
    datamanager_v2 = DataManagerV2(dataprovider, compute_double_links=False, use_default_callbacks=False)
    return datamanager_v2


def get_file_md5(file_name):
    # type: (unicode) -> unicode
    with open(file_name, u'rb') as f:
        data = f.read()
        return hashlib.md5(data).hexdigest()


def get_output_and_color(g_did_run):
    # type:(bool) -> Tuple[int, unicode]
    if g_did_run:
        return 32, u'executed [OK]'
    else:
        return 36, u'skip (unnecessary)'


def _get_synchronizer_info_and_database_info(db):
    # type: (MongoClient) -> (MongoCollection, Optional[Dict])
    synchronizer_info_col = db.get_collection(SANATIZE_MONGO.SYNCHRONIZER_INFO_COLLECTION)
    database_info = synchronizer_info_col.find_one({u'_id': SANATIZE_MONGO.DATABASE_INFO_DOCUMENT})
    return synchronizer_info_col, database_info


def check_sanatize_database_info_flag(db, sanatize_database_flag, expected_value):
    # type: (MongoClient, unicode, int) -> bool
    synchronizer_info_col, database_info = _get_synchronizer_info_and_database_info(db)
    if database_info and database_info.get(sanatize_database_flag, 0) == expected_value:
        return True
    return False


def update_database_info_collection(db, sanatize_database_flag, wanted_value):
    # type: (MongoClient, unicode, int) -> None
    synchronizer_info_col, database_info = _get_synchronizer_info_and_database_info(db)
    if not database_info:
        database_info = {u'_id': SANATIZE_MONGO.DATABASE_INFO_DOCUMENT}
    database_info[sanatize_database_flag] = wanted_value
    synchronizer_info_col.save(database_info)
