#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import ConfigParser
import cStringIO
import json
import os
import shutil
import signal
import sys
import time
import traceback
from string import digits

from shinken.log import logger as internal_log, LoggerFactory
from shinken.misc.type_hint import NoReturn, Dict
from shinken.property import StringProp, BoolProp
from shinken.thread_helper import Thread
from shinken.util import set_process_name
from shinken.vmware_stats import VMWareStatsCompute
from shinkensolutions.lib_checks.graphite import GraphiteMetricsCounter

GATHERER_PID_FILE = '/var/run/shinken/gatherer.pid'
BYTES_PER_SECTOR = 512
IO_STATS_FILE = '/dev/shm/__check_graphite_iostats.tmp'
IO_STATS_COLLECTOR_PID_FILE = '/opt/graphite/storage/iostats_collector.pid'

DEFAULT_UMASK = 0022
# Set umask to avoid problems when creating files
os.umask(DEFAULT_UMASK)

if os.environ.get('GATHERER_DEBUG', '0') == '1':
    internal_log.setLevel('DEBUG')

BASE_PROCESS_NAME = 'shinken-gatherer     [ Main daemon ]'

logger = LoggerFactory.get_logger()
logger_daemon = logger.get_sub_part('DAEMON')
logger_io_stats = logger.get_sub_part('IO-STATS')


class IoStats(object):
    def __init__(self):
        # type: () -> NoReturn
        
        self.previous_raw = {}
        self.previous_time = 0
        
        # Columns for disk entry in /proc/diskstats
        self.columns_disk = ['major', 'minor', 'device', 'reads', 'reads_merged', 'read_sectors', 'read_ms', 'writes', 'writes_merged', 'write_sectors', 'write_ms', 'cur_ios', 'total_io_ms', 'total_io_weighted_ms']
        # We don't care about theses fields
        # NOTE: write_ms and read_ms are over the sleep time, not sure about what it means
        self.columns_to_del_in_raw = ('major', 'minor', 'cur_ios', 'total_io_weighted_ms', 'read_ms', 'write_ms')
        self.data = {}
        if os.path.exists(IO_STATS_FILE):
            try:
                with open(IO_STATS_FILE, 'r') as f:
                    self.data = json.loads(f.read())
            except (ValueError, IOError):
                pass
    
    
    def _get_disk_stats(self):
        # type: () -> NoReturn
        file_path = '/proc/diskstats'
        result = {}
        
        # ref: http://lxr.osuosl.org/source/Documentation/iostats.txt
        
        with open(file_path, 'r') as f:
            lines = f.readlines()
        for line in lines:
            if line == '':
                continue
            split = line.split()
            if len(split) == len(self.columns_disk):
                columns = self.columns_disk
            else:
                # No match, drop partitions too
                continue
            
            data = dict(zip(columns, split))
            
            device_name = data['device']
            
            # we only want real device, NOT partition, so check with the presence in /sys/block/
            if not os.path.exists('/sys/block/%s' % device_name):
                continue
            
            for key in data:
                if key != 'device':
                    data[key] = int(data[key])
            # We don't care about some raw fields
            for k in self.columns_to_del_in_raw:
                del data[k]
            
            result[device_name] = data
        
        return result
    
    
    def compute_linux_disk_stats(self, new_raw_stats, diff_time):
        # type: (Dict, float) -> Dict
        r = {}
        for (device, new_stats) in new_raw_stats.items():
            old_stats = self.previous_raw.get(device, None)
            # A new disk did spawn? wait a loop to compute it
            if old_stats is None:
                continue
            r[device] = {}
            for (k, new_v) in new_stats.items():
                old_v = old_stats[k]
                
                # String= device name, but we already have it in the key path
                if isinstance(old_v, basestring):
                    continue
                # Some columns are finally computed in /s (diff/time)
                elif k in ('reads', 'reads_merged', 'writes', 'writes_merged'):
                    this_type_consumed = int((new_v - old_v) / float(diff_time))
                    r[device][k + '/s'] = this_type_consumed
                # Sectors are transformed into bytes/s
                elif k == 'read_sectors':
                    computed_v = int(BYTES_PER_SECTOR * (new_v - old_v) / float(diff_time))
                    r[device]['read_bytes/s'] = computed_v
                elif k == 'write_sectors':
                    computed_v = int(BYTES_PER_SECTOR * (new_v - old_v) / float(diff_time))
                    r[device]['write_bytes/s'] = computed_v
                # Time are transformed into % activity
                # NOTE: ms=> s = *1000
                #       percent= *100
                elif k == 'total_io_ms':
                    computed_v = int(100 * (new_v - old_v) / float(diff_time * 1000))
                    r[device][r'util%'] = computed_v
        return r
    
    
    def launch(self):
        # type: () -> Dict
        if not sys.platform.startswith('linux'):  # linux2 on python2, linux on python3
            return {}
        
        logger_io_stats.debug('Starting the get all disks stats')
        
        new_stats = self._get_disk_stats()
        new_time = time.time()
        # First loop: do a 1s loop an compute it, to directly have results
        if self.previous_time == 0:
            self.previous_time = time.time()
            self.previous_raw = new_stats
            time.sleep(1)
            new_stats = self._get_disk_stats()
            new_time = time.time()
        
        # So compute the diff
        io_stats = self.compute_linux_disk_stats(new_stats, new_time - self.previous_time)
        self.previous_raw = new_stats
        self.previous_time = new_time
        
        return io_stats
    
    
    def dump(self, io_stats):
        # type: (Dict) -> NoReturn
        
        for device, value in io_stats.iteritems():
            if not self.data.get(device, None):
                self.data[device] = []
            self.data[device].append(value['util%'])
            
            if len(self.data[device]) > 60:
                try:
                    self.data[device].pop(0)
                except IndexError:
                    pass
        
        logger_io_stats.debug('Saving io stats file %s' % IO_STATS_FILE)
        try:
            with open('%s.new' % IO_STATS_FILE, 'w') as f:
                f.write(json.dumps(self.data))
            shutil.move('%s.new' % IO_STATS_FILE, IO_STATS_FILE)
            logger_io_stats.debug('The io stats file was saved to %s' % IO_STATS_FILE)
        except Exception as exp:
            logger_io_stats.error('Cannot save io stats file %s: %s' % (IO_STATS_FILE, exp))


class GraphiteStatGatherer(Thread):
    
    def __init__(self):
        # type: () -> NoReturn
        self.graphite_count_metrics_thread = None
        self.graphite_counter = GraphiteMetricsCounter()
        super(GraphiteStatGatherer, self).__init__(loop_speed=60, force_stop_with_application=True)
    
    
    def loop_turn(self):
        self._do_graphite_count_metrics_thread()
    
    
    def get_thread_name(self):
        return u'do_graphite_count_metrics_thread'
    
    
    # Note: we will compute each minute the number of files inside the graphite directory
    def _do_graphite_count_metrics_thread(self):
        # type: () -> NoReturn
        set_process_name('%s [currently updating metrology metric count...]' % BASE_PROCESS_NAME)
        logger_io_stats.debug('Start computing graphite counters.')
        self.graphite_counter.update_count()
        set_process_name('%s' % BASE_PROCESS_NAME)
    
    
    def assert_graphite_count_metrics_thread(self):
        # type: () -> NoReturn
        if self.graphite_count_metrics_thread is None or not self.graphite_count_metrics_thread.is_alive():
            self.start_thread()


class Gatherer(object):
    properties = {
        'vmware_statistics_compute_enable': BoolProp(default=True),
        'logdir'                          : StringProp(default='/var/log/shinken'),
        'use_local_log'                   : BoolProp(default=True),
        'local_log'                       : StringProp(default='%(logdir)s/gathererd.log'),
        'log_level'                       : StringProp(default='INFO'),
    }
    
    
    def __init__(self):
        # type: () -> NoReturn
        self.config_file = '/etc/shinken/daemons/gathererd.ini'
        self.local_log = '%(logdir)s/gathererd.log'
        self.log_level = 'INFO'
        self.use_local_log = True
        self.logdir = '/var/log/shinken'
        self.vmware_statistics_compute_enable = True
        self.parse_config_file()
        self.set_log()
        
        self.graphite_stat_gatherer = GraphiteStatGatherer()
        
        for sig in (signal.SIGTERM, signal.SIGINT, signal.SIGUSR1, signal.SIGUSR2):
            signal.signal(sig, self.sigint_handler)
    
    
    def set_log(self):
        # type: () -> NoReturn
        internal_log.register_local_log(self.local_log)
        internal_log.setLevel(self.log_level)
        internal_log.set_name('gatherer')
        internal_log.set_human_format()
    
    
    def parse_config_file(self):
        # type: () -> NoReturn
        properties = self.__class__.properties
        config = ConfigParser.ConfigParser()
        # Beware: ini file do not like space before comments and such things, so clean it before read
        data = cStringIO.StringIO('\n'.join(line.strip() for line in open(self.config_file)))
        config.readfp(data)
        
        if not config.sections():
            logger.error('Bad or missing config file: %s ' % self.config_file)
            sys.exit(2)
        try:
            for key, value in config.items('daemon'):
                if key in properties:
                    value = properties[key].pythonize(value)
                setattr(self, key, value)
        except ConfigParser.InterpolationMissingOptionError as exp:
            exp = str(exp)
            wrong_variable = exp.split('\n')[3].split(':')[1].strip()
            logger.error("Incorrect or missing variable '%s' in config file : %s" % (wrong_variable, self.config_file))
            sys.exit(2)
        
        # Now fill all defaults where missing parameters
        for prop, entry in properties.iteritems():
            if not hasattr(self, prop):
                value = entry.pythonize(entry.default)
                setattr(self, prop, value)
    
    
    def sigint_handler(self, _signal, _frame):
        logger_daemon.info('Receiving a stop (signal %s). Exiting.' % _signal)
        self.graphite_stat_gatherer.stop()
        Gatherer._clean_pid_file()
        sys.exit(0)
    
    
    # Before this version there was the IO_STATS_COLLECTOR_PID_FILE file
    # if so, kill the old process, and delete the pid file
    @staticmethod
    def _check_old_pid_file():
        if os.path.exists(IO_STATS_COLLECTOR_PID_FILE):
            try:
                with open(IO_STATS_COLLECTOR_PID_FILE) as f:
                    old_pid = int(f.read().strip())
                    logger_daemon.info(' We did detect the old script process as pid %s' % old_pid)
                os.unlink(IO_STATS_COLLECTOR_PID_FILE)
                os.kill(old_pid, 9)  # HEAD SHOOT
                logger_daemon.info(' The old script was stopped successful.')
            except Exception as exp:
                logger_daemon.error(' Cannot stop the old script process: %s' % exp)
    
    
    @staticmethod
    def _check_pid_file(pid_file):
        my_pid = str(os.getpid())
        if os.path.exists(pid_file):
            try:
                with open(pid_file, 'r') as f:
                    pid_in_file = int(f.read())
                    if pid_in_file and Gatherer.pid_is_running(pid_in_file):
                        logger_daemon.info('The gatherer is already running as pid %s. Bailing out.' % pid_in_file)
                        exit(0)
            except Exception:
                pass
        with open(pid_file, 'w') as f:
            logger_daemon.info('Starting the gatherer as the process pid %s.' % my_pid)
            f.write(my_pid)
    
    
    @staticmethod
    def _clean_pid_file():
        try:
            os.unlink(GATHERER_PID_FILE)
        except Exception:
            pass
    
    
    @staticmethod
    def pid_is_running(pid):
        try:
            os.kill(pid, 0)
        except OSError:
            return False
        else:
            return True
    
    
    def main(self):
        try:
            self._main()
        except KeyboardInterrupt:
            logger_daemon.info('Exiting gatherer.')
            self._clean_pid_file()
            sys.exit(0)
        except Exception:
            logger_daemon.error('Did have a unknown exception: %s. Exiting the gatherer.' % (traceback.format_exc()))
    
    
    def _main(self):
        self._check_old_pid_file()
        self._check_pid_file(GATHERER_PID_FILE)
        
        io_stats = IoStats()
        last_threads_check = 0
        
        # IMPORTANT: we CANNOT do VMWare stuff in a thread, it will just segfault
        #            ==> IN THE MAIN THREAD
        logger_daemon.debug('Creating VMWare stats object.')
        vmware_stats_writer = VMWareStatsCompute()
        vmware_stats_writer.set_enable(self.vmware_statistics_compute_enable)
        logger_daemon.debug('Starting requesting VMWare stats.')
        vmware_stats_writer.update_stats()
        
        while True:
            # if the graphite count did crash, do not hammer the start as it will count files in graphite directory
            now = time.time()
            if now > last_threads_check + 60:
                self.graphite_stat_gatherer.assert_graphite_count_metrics_thread()
            
            io_stats.dump(io_stats.launch())
            
            logger_daemon.debug('Starting requesting VMWare stats.')
            vmware_stats_writer.update_stats()
            
            logger_daemon.debug('Gatherer is running.')
            time.sleep(1)


if __name__ == '__main__':
    gatherer = Gatherer()
    gatherer.main()
