//__lib__ lib_common
'use strict';

SHINKEN.OBJECT.JsonEditor           = function ( dom_element, json, default_tag ) {
    this.__class_name__ = 'SHINKEN.OBJECT.JsonEditor';
    this.dom_element    = dom_element;
    this.json           = json;
    this.content        = null;
    this.default_tag    = default_tag;
    this.init();
};
SHINKEN.OBJECT.JsonEditor.prototype = {
    init: function () {
        this.content = this.parseToHtml( this.json );
        this.updateHtml();
        this.computeErrorContent();
    },
    
    //********************************************  GET JSON  **************************************************//
    getCurrentContent: function () {
        return this.dom_element.innerText;
    },
    //********************************************  COMPUTE   **************************************************//
    computeErrorContent: function () {
        this.data_with_error = {};
        var _error_contents  = this.dom_element.querySelectorAll( ".shinken-json-error" );
        for ( var i = 0, _size_i = _error_contents.length; i < _size_i; i++ ) {
            if ( _error_contents[ i ].dataset.idError ) {
                this.data_with_error[ _error_contents[ i ].dataset.idError ] = _error_contents[ i ].innerText;
            }
        }
    },
    //********************************************  COMPUTE HTML  **************************************************//
    parseToHtml            : function ( to_parse, shinken_json_tag ) {
        var _to_return = '';
        if ( typeof to_parse === 'string' ) {
            _to_return += this.parseStringToHtml( to_parse, shinken_json_tag );
        }
        else if ( typeof to_parse === 'number' ) {
            _to_return += this.parseNumberToHtml( to_parse, shinken_json_tag );
        }
        else if ( typeof to_parse === 'boolean' ) {
            _to_return += this.parseBooleanToHtml( to_parse, shinken_json_tag );
        }
        else if ( to_parse === null || to_parse === undefined ) {
            if ( this.default_tag ) {
                _to_return += this.parseStringToHtml( shinken_json_tag ? "" : this.default_tag, shinken_json_tag );
            }
        }
        else if ( to_parse instanceof Array ) {
            _to_return += this.parseArrayToHtml( to_parse, shinken_json_tag );
        }
        else if ( typeof to_parse === 'object' ) {
            _to_return += this.parseObjectToHtml( to_parse, shinken_json_tag );
        }
        else {
            _to_return += '{}';
        }
        return _to_return;
    },
    parseObjectToHtml      : function ( to_parse, shinken_json_tag ) {
        var _to_return;
        if ( to_parse instanceof SHINKEN.OBJECT.Param ) {
            return this.parseToHtml( to_parse.getValue(), to_parse.getJsonTag() );
        }
        if ( to_parse[ SHINKEN.OBJECT.JsonEditor.CONST.SHINKEN_JSON_UUID ] ) {
            if ( to_parse[ SHINKEN.OBJECT.JsonEditor.CONST.SHINKEN_JSON_UUID ] instanceof SHINKEN.OBJECT.Param ) {
                _to_return = '<span data-json-uuid-part="' + to_parse[ SHINKEN.OBJECT.JsonEditor.CONST.SHINKEN_JSON_UUID ].getValue() + '"><span class="shinken-json-object-start">{</span>';
            }
            else {
                _to_return = '<span data-json-uuid-part="' + to_parse[ SHINKEN.OBJECT.JsonEditor.CONST.SHINKEN_JSON_UUID ] + '"><span class="shinken-json-object-start">{</span>';
            }
        }
        else {
            _to_return = '<span><span class="shinken-json-object-start">{</span>';
        }
        if ( to_parse[ "__JSON_ERROR__UUID__" ] ) {
            _to_return += '<ul class="shinken-json-dict shinken-json-error" data-id-error="' + to_parse[ "__JSON_ERROR__UUID__" ] + '">';
            delete to_parse[ "__JSON_ERROR__UUID__" ];
        }
        else {
            _to_return += '<ul class="shinken-json-dict">';
        }
        var _keys = Object.keys( to_parse );
        var _current_key;
        for ( var i = 0, _size = _keys.length; i < _size; i++ ) {
            _current_key = _keys[ i ];
            _to_return += this.buildLineEntry( _current_key, to_parse[ _keys[ i ] ] );
            _to_return += '<span class="shinken-json-key">"' + _current_key + '"</span> : ';
            _to_return += this.parseToHtml( to_parse[ _keys[ i ] ] );
            if ( i < _keys.length - 1 ) {
                _to_return += ',';
            }
            _to_return += '</div></li>';
        }
        _to_return += '</ul>';
        _to_return += '<span class="shinken-json-object-end">}</span></span>';
        return _to_return;
    },
    parseArrayToHtml       : function ( to_parse, shinken_json_tag ) {
        if ( to_parse.length === 0 ) {
            return '[]';
        }
        
        var _to_return = '<span class="shinken-json-array-start">[</span>';
        _to_return += '<ol class="shinken-json-array">';
        
        for ( var i = 0, _size_i = to_parse.length; i < _size_i; i++ ) {
            _to_return += '<li>';
            _to_return += this.parseToHtml( to_parse[ i ] );
            if ( i < to_parse.length - 1 ) {
                _to_return += ',';
            }
            _to_return += '</li>';
        }
        _to_return += '</ol>';
        _to_return += '<span class="shinken-json-array-end">]</span>';
        return _to_return;
    },
    parseStringToHtml      : function ( to_parse, shinken_json_tag ) {
        if ( to_parse === this.default_tag && !shinken_json_tag ) {
            return '"<span class="shinken-default-value" data-default-value="' + this.default_tag + '">' + this.default_tag + '</span>"';
        }
        var to_parse_escape = SHINKEN.TOOLS.STRING.cleanXss( to_parse );
        return '<span class="shinken-json-string ' + this.parseJsonTagToClassCss( shinken_json_tag ) + '" ' + this.parseJsonTagToAttribute( shinken_json_tag ) + '>"' + to_parse_escape + '"</span>';
    },
    parseBooleanToHtml     : function ( to_parse, shinken_json_tag ) {
        return '<span class="shinken-json-boolean ' + this.parseJsonTagToClassCss( shinken_json_tag ) + '" ' + this.parseJsonTagToAttribute( shinken_json_tag ) + '>' + to_parse + '</span>';
    },
    parseNumberToHtml      : function ( to_parse, shinken_json_tag ) {
        return '<span class="shinken-json-number ' + this.parseJsonTagToClassCss( shinken_json_tag ) + '" ' + this.parseJsonTagToAttribute( shinken_json_tag ) + '>' + to_parse + '</span>';
    },
    parseJsonTagToClassCss : function ( to_parse ) {
        var _to_return = "";
        switch ( to_parse ) {
            case SHINKEN.OBJECT.JsonEditor.CONST.JSON_INVALID_VALUE:
                _to_return = "shinken-json-error";
                break;
        }
        return _to_return;
    },
    parseJsonTagToAttribute: function ( to_parse ) {
        var _to_return = "";
        switch ( to_parse ) {
            case SHINKEN.OBJECT.JsonEditor.CONST.JSON_INVALID_VALUE:
                _to_return = 'data-id-error="' + SHINKEN.TOOLS.STRING.buildUUID() + '"';
                break;
        }
        return _to_return;
    },
    buildLineEntry         : function ( key, value ) {
        var li_class_to_add  = "";
        var div_class_to_add = "";
        var div_attribute    = "";
        if ( value !== undefined ) {
            switch ( value.json_tag ) {
                case SHINKEN.OBJECT.JsonEditor.CONST.JSON_UNKNOWN_PROPERTY:
                    div_class_to_add = "shinken-json-error";
                    div_attribute    = 'onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" shi-tip-type="shinken-warning" shi-tip-placement="left" shi-tip-html="' + SHINKEN.TOOLS.STRING.format( _( "json_editor.unknown_value", [DICTIONARY_COMMON_UI] ), key ) + '"';
                    div_attribute += ' data-json-uuid-part="' + value.getUUID() + '"';
                    break;
                case SHINKEN.OBJECT.JsonEditor.CONST.JSON_BAD_FORMAT_IS_NOT_AN_OBJECT:
                    div_class_to_add = "shinken-json-error";
                    div_attribute    = 'onmouseenter="SHINKEN_TOOLTIP.showTooltip(this)" onmouseleave="SHINKEN_TOOLTIP.hideTooltip()" shi-tip-type="shinken-warning" shi-tip-placement="left" shi-tip-html="' + SHINKEN.TOOLS.STRING.format( _( "json_editor.not_valid_format_value", [DICTIONARY_COMMON_UI] ), key ) + '"';
                    div_attribute += ' data-json-uuid-part="' + value.getUUID() + '"';
                    break;
            }
        }
        if ( key === SHINKEN.OBJECT.JsonEditor.CONST.SHINKEN_JSON_UUID ) {
            li_class_to_add = "shinken-json-dissimulate";
        }
        return '<li class="' + li_class_to_add + '"><div class="shinken-div-inline-block ' + div_class_to_add + '" ' + div_attribute + '>';
    },
    //********************************************  VIEW   **************************************************//
    updateHtml            : function () {
        this.dom_element.innerHTML = this.content;
        this.dom_element.classList.add( "shinken-json-editor-container" );
        this.dom_element.setAttribute( "contenteditable", "true" );
    },
    cleanHtmlAfterEdition : function () {
        this.updateDefaultValueHtml();
        this.removeErrorAfterUpdate();
    },
    updateDefaultValueHtml: function () {
        var _default_value_doms = this.dom_element.querySelectorAll( ".shinken-default-value" );
        for ( var i = 0, _size_i = _default_value_doms.length; i < _size_i; i++ ) {
            if ( _default_value_doms[ i ].innerText !== _default_value_doms[ i ].dataset.defaultValue ) {
                _default_value_doms[ i ].classList.remove( "shinken-default-value" );
            }
        }
    },
    removeErrorAfterUpdate: function () {
        var _keys = Object.keys( this.data_with_error );
        var _current_key;
        var _current_dom;
        for ( var i = 0, _size = _keys.length; i < _size; i++ ) {
            _current_key = _keys[ i ];
            _current_dom = this.dom_element.querySelector( '.shinken-json-error[data-id-error="' + _current_key + '"]' );
            if ( _current_dom && !this._is_same_text( _current_dom.innerText, this.data_with_error[ _current_key ] ) ) {
                _current_dom.classList.remove( "shinken-json-error" );
                delete this.data_with_error[ _current_key ];
            }
        }
        
    },
    _is_same_text         : function ( text_1, text_2 ) {
        return SHINKEN.TOOLS.STRING.replaceAll( text_1, "\n", "" ) === SHINKEN.TOOLS.STRING.replaceAll( text_2, "\n", "" );
    }
};

SHINKEN.OBJECT.JsonEditor.CONST = {
    SHINKEN_JSON_UUID                : "__SHINKEN_JSON_UUID__",
    JSON_UNKNOWN_PROPERTY            : "__JSON_UNKNOWN_PROPERTY__",
    JSON_BAD_FORMAT_IS_NOT_AN_OBJECT : "__JSON_BAD_FORMAT_IS_NOT_AN_OBJECT__",
    JSON_INVALID_VALUE               : "__JSON_INVALID_VALUE__",
    JSON_MODIFIED_VALUE_TO_FIT_FORMAT: "__JSON_MODIFIED_VALUE_TO_FIT_FORMAT__",
    JSON_DEFAULT_VALUE_IS_OK         : "__JSON_DEFAULT_VALUE_IS_OK__"
};
SHINKEN.TOOLS.CLASS.addPrototype( SHINKEN.OBJECT.JsonEditor, SHINKEN.OBJECT.ShinkenObject );

SHINKEN.OBJECT.JsonEditorService = (function ( self ) {
    "use strict";
    var __LINE_TAG__       = "__SHINKEN_LINE_TAG__";
    self.computeValidation = function ( dom_element ) {
        var _text_parsed = _parseJsonDomElementToText( dom_element );
        _text_parsed     = SHINKEN.TOOLS.STRING.replaceAll( _text_parsed, __LINE_TAG__, "\n" );
        return [_text_parsed, SHINKEN.TOOLS.STRING.JSON.isFormatJsonInvalid( _text_parsed )];
    };
    self.computeLineNumber = function ( dom_element ) {
        var _padding_bottom = DOM.Service.getComputedStyle( dom_element, 'padding-bottom' );
        var _padding_top    = DOM.Service.getComputedStyle( dom_element, 'padding-top' );
        
        var number_line = (dom_element.scrollHeight - _padding_bottom - _padding_top) / 16;
        var _html       = '';
        for ( var i = 0, _size = number_line; i < _size; i++ ) {
            _html += '<div class="shinken-line-number" data-number="' + (i + 1) + '">' + (i + 1) + '</div>';
        }
        var _container       = document.querySelector( '.shinken-textarea-line-nums' );
        _container.innerHTML = _html;
    };
    
    var _parseJsonDomElementToText = function ( dom_element ) {
        var clone       = DOM.Service.cloneElement( dom_element );
        var dissimulate = clone.querySelectorAll( ".shinken-json-dissimulate" );
        for ( var i = 0, _size_i = dissimulate.length; i < _size_i; i++ ) {
            DOM.Service.removeElement( dissimulate[ i ] );
        }
        
        var _text    = clone.innerHTML;
        _text        = SHINKEN.TOOLS.STRING.replaceAll( _text, "<li class>", __LINE_TAG__ + "<li>" );
        _text        = SHINKEN.TOOLS.STRING.replaceAll( _text, "</ol>", __LINE_TAG__ + "</ol>" );
        _text        = SHINKEN.TOOLS.STRING.replaceAll( _text, "</ul>", __LINE_TAG__ + "</ul>" );
        _text        = SHINKEN.TOOLS.STRING.replaceAll( _text, "<ul class=\"shinken-json-dict\">", __LINE_TAG__ + "<ul class=\"shinken-json-dict\">" );
        var _new_dom = DOM.Service.createElement( "div", {}, _text );
        var _text_2  = _new_dom.innerText;
        return SHINKEN.TOOLS.STRING.replaceAll( _text_2, __LINE_TAG__, "\n" );
    };
    
    return self;
})( SHINKEN.OBJECT.JsonEditorService || {} );