#!/bin/bash

STEP_NUMBER="8"

function display_help() {
  printf "revert-patch [options]\n"
  printf "This script will revert the patch PATCH${PATCH_VERSION} for the shinken version v${SHINKEN_VERSION}-release.${PATCH_LANG,,}\n"
  printf "Options:\n"
  printf "  --help          -h                     : will display this help\n"
  printf "  --force                                : removes the need to hit a key when required during revert. All daemons will be restarted\n"
  printf "  --ignore-pre-setup-non-blocking-errors : PLEASE USE THIS OPTION ONLY WITH YOUR DEDICATED SUPPORT\n"
  printf "                                           Ignore errors of the shinken-backup before the patch revert\n"
  printf "\n"
}

force=""

while [ $# -ge 1 ]; do
  case "$1" in
  -h | --help)
    display_help # Call your function
    # no shifting needed here, we're done.
    exit 0
    ;;
  --force)
    #  It's better to assign a string, than a number like "verbose=1"
    #  because if you're debugging script with "bash -x" code like this:
    #    if [ "$verbose" ] ...
    #  You will see:
    #    if [ "verbose" ] ...
    force="--force"
    shift
    ;;
  --ignore-pre-setup-non-blocking-errors)
    BYPASS_NON_BLOCKING_ERRORS="bypass"
    shift
    ;;
  --) # End of all options
    shift
    break
    ;;
  -*)
    echo "Error: Unknown option: $1" >&2
    display_help
    exit 1
    ;;
  *) # No more options
    break
    ;;
  esac
done

if [[ ! -f lib/install_lib/common.sh ]]; then
  printf "\033[31m Error: please launch the patch from its directory\033[0m\n"
  exit 2
fi

. lib/install_lib/common.sh

if [[ "$force" != "" ]]; then
  printf -- "\n${YELLOW}NOTE: FORCE MODE ENABLED${RESET}\n"
fi

run_python --check-version

restore_all_patch_valid_rights

print_step "Patch information" "1" $STEP_NUMBER
init_patch_name_and_location "$PATCH_ID"
display_uninstall_acceptance_banner "$force"
# If the patch was not apply, cannot revert
run_python --patch-is-installed

print_step "Shinken make a configuration backup" "2" $STEP_NUMBER
run_python_non_blocking --backup-config --backup-path "${OUTPUT_FOLDER}/BACKUP_CONFIGURATION_PRE_UNPATCH---${PATCH_ID}---/"

print_step "Stopping Shinken daemons" "3" $STEP_NUMBER
run_python --stop "${force}"

print_step "Restore selinux rules" "4" $STEP_NUMBER
run_python --remove-selinux-rules

print_step "Revert sanatize" "5" $STEP_NUMBER
run_python_and_return_code --revert-sanatize --log-path "${LOG_PATH}"
ERROR_ON_SANATIZE=$?

print_step "Restore files from backup" "6" $STEP_NUMBER
run_python --revert-patch --id "$PATCH_ID"
clean_shinken_module

print_step "Restart tools used by Shinken : ${TOOLS_USED_BY_SHINKEN}" "7" $STEP_NUMBER
# Checking if the command "systemctl" is in the system path, we cannot reload daemons without it
if command -v systemctl 2&> /dev/null ; then
  systemctl daemon-reload
fi

if [[ ${TOOLS_USED_BY_SHINKEN} == "" ]]; then
  printf -- "\n${YELLOW}There is no tools used by Shinken to restart${RESET}\n\n"
fi

for tool_used_by_shinken in ${TOOLS_USED_BY_SHINKEN}; do
  run_python --restart --service-list "${tool_used_by_shinken}" "${force}"
done

print_step "Restart Shinken daemons" "8" $STEP_NUMBER
run_python --start "${force}"

if [[ ${ERROR_ON_SANATIZE} -eq 1 ]]; then
  printf "\n\033[31mSome errors occurred while running the fixes.\n"
  printf "Please check the log file for more information (\033[32m${SANATIZE_LOGFILE}\033[31m).\n"
  printf "Send this file to your Shinken support if needed.\033[0m\n"
fi

printf -- "\n$YELLOW Your shinken installation was restored into version v${SHINKEN_VERSION}-release.${PATCH_LANG,,} without any patch.$RESET\n\n"
