#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import errno
import fnmatch
import os
import shutil
import subprocess


class COLOR:
    GREY = '\033[90m'
    BLUE = '\033[94m'
    MAGENTA = '\033[94m'
    YELLOW = '\033[93m'
    RED = '\033[31m'
    RESET = '\033[0m'


def make_unicode(change_input):
    change_output = change_input
    if isinstance(change_input, unicode):
        return change_output
    elif hasattr(change_input, 'decode') and type(change_input) != unicode:
        change_output = change_input.decode('utf8', 'ignore')
    elif isinstance(change_input, dict):
        change_output = {}
        for prop, value in change_input.iteritems():
            change_output[make_unicode(prop)] = make_unicode(value)
    elif isinstance(change_input, list):
        change_output = [make_unicode(i) for i in change_input]
    elif isinstance(change_input, tuple):
        change_output = tuple([make_unicode(i) for i in change_input])
    return change_output


def create_tree(_path):
    abs_path = os.path.abspath(_path)
    parent = os.path.dirname(abs_path)
    
    if os.path.exists(abs_path):
        return
    elif os.path.exists(parent):
        os.makedirs(abs_path)
        return
    else:
        create_tree(parent)
        create_tree(_path)


def touch_file(filename):
    with open(filename, 'a') as fd:
        os.fsync(fd.fileno())
        os.utime(filename, None)


def find_files(directory, pattern):
    for root, dirs, files in os.walk(directory):
        for basename in files:
            if fnmatch.fnmatch(basename, pattern):
                filename = os.path.join(root, basename)
                yield filename


def find_folder(directory, pattern):
    for root, dirs, files in os.walk(directory):
        for basename in dirs:
            if fnmatch.fnmatch(basename, pattern):
                filename = os.path.join(root, basename)
                yield filename


def copy_recursively(_path, dest, debug=False):
    files_to_copy = os.walk(_path)
    for _root, _dirs, _files in files_to_copy:
        
        sub_folder_list = _root.split('%s%s' % (_path, os.sep))
        sub_folder = sub_folder_list[1] if len(sub_folder_list) > 1 else ''
        
        for _dir in _dirs:
            dir_path = protect_long_path_for_windows(os.path.join(dest, sub_folder, _dir))
            create_tree(dir_path)
        for _file in _files:
            copy_file(os.path.join(_root, _file), os.path.join(dest, sub_folder, _file), debug=debug)


def copy_file(_source, _dest, deleted_files=None, debug=False):
    _dest = protect_long_path_for_windows(_dest)
    
    if os.path.exists(_dest):
        print_info('File [%s] has been overwritten.' % _dest.strip(u'\\\\?\\'))
    
    if debug:
        print_info('Copy file [%s] \t==>\t [%s]' % (_source, _dest.strip(u'\\\\?\\')))
    
    file_path = os.path.dirname(_dest)
    create_tree(file_path)
    try:
        shutil.copyfile(_source, _dest)
    except IOError:
        if deleted_files is not None:
            deleted_files.append(_source)
        else:
            print_warning('The file %s can\'t be copied.' % _source)


def protect_long_path_for_windows(path):
    if os.name == 'nt':
        return u'\\\\?\\%s' % os.path.abspath(path)
    return os.path.abspath(path)


def mkdir_p(path):
    try:
        os.makedirs(path)
    except OSError as exc:  # Python >2.5
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


def ask_user(question, possible_choice=['y', 'o', 'n'], printable_choice='Yes/No', default_choice='Yes'):
    valid_choice = False
    user_choice = default_choice
    while not valid_choice:
        print_info(question)
        user_choice = raw_input('(%s) by default \'%s\' : ' % (printable_choice, default_choice.capitalize())).strip().lower()
        if not user_choice:
            user_choice = default_choice[0].lower()
            valid_choice = True
        else:
            user_choice = user_choice[0]
            
            if user_choice in possible_choice:
                valid_choice = True
            
            else:
                print_error('Your answer is not valid, please choose one this answer %s' % printable_choice)
    
    print '\n'
    return user_choice


def print_info(msg, color=COLOR.BLUE):
    print '%s%s%s' % (color, msg, COLOR.RESET)


def print_warning(msg):
    print '\n%sWARNING : %s%s\n' % (COLOR.YELLOW, msg, COLOR.RESET)


def print_error(msg):
    print '\n%sERROR : %s%s\n' % (COLOR.RED, msg, COLOR.RESET)


def run_command_with_return_code(command, shell=True, env=None):
    if env is None:
        _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shell)
    else:
        _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shell, env=env)
    stdout, stderr = _process.communicate()
    return _process.returncode, make_unicode(stdout), make_unicode(stderr)


