#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2019:
#
# This file is part of Shinken Enterprise, all rights reserved.
import sys

from pymongo import Connection

class DataException(Exception):
    pass


class ITEM_STATE:
    NEW = 'new'
    STAGGING = 'stagging'
    PRODUCTION = 'production'
    MERGE_SOURCES = 'merge_from_sources'
    CHANGES = 'changes'
    SUBMIT_STAGGING = 'submit_stagging'
    WORKING_AREA = 'working_area'
    DELETED = 'deleted'
    RAW_SOURCES = 'raw_source'


class ITEM_TYPE:
    ELEMENTS = 'elements'
    
    CLUSTERS = 'clusters'
    CLUSTERTPLS = 'clustertpls'
    
    HOSTS = 'hosts'
    HOSTTPLS = 'hosttpls'
    HOSTGROUPS = 'hostgroups'
    
    SERVICESHOSTS = 'serviceshosts'
    SERVICESHOSTTPLS = 'serviceshosttpls'
    SERVICESCLUSTERS = 'servicesclusters'
    SERVICESCLUSTERTPLS = 'servicesclustertpls'
    SERVICETPLS = 'servicetpls'
    
    CONTACTS = 'contacts'
    CONTACTTPLS = 'contacttpls'
    CONTACTGROUPS = 'contactgroups'
    
    ESCALATIONS = 'escalations'
    NOTIFICATIONWAYS = 'notificationways'
    COMMANDS = 'commands'
    BUSINESSIMPACTMODULATIONS = 'businessimpactmodulations'
    MACROMODULATIONS = 'macromodulations'
    RESULTMODULATIONS = 'resultmodulations'
    TIMEPERIODS = 'timeperiods'
    PACK = 'packs'
    PLUGIN = 'plugins'


COLLECTIONS = {
    ITEM_STATE.NEW            : 'newelements-%s',
    ITEM_STATE.STAGGING       : 'configuration-stagging-%s',
    ITEM_STATE.PRODUCTION     : 'configuration-production-%s',
    ITEM_STATE.MERGE_SOURCES  : 'merge_from_sources-%s',
    ITEM_STATE.RAW_SOURCES    : 'data-%s-%s',
    ITEM_STATE.CHANGES        : 'changeelements-%s',
    ITEM_STATE.DELETED        : 'deleted-stagging-%s',
    ITEM_STATE.WORKING_AREA   : 'configuration-working-area-%s',
    ITEM_STATE.SUBMIT_STAGGING: 'configuration-working-area-%s',
}

DEF_ITEMS = {
    # Source element : table: where to look up, key:value=> which table to query, property to clean, and where=match only theses elements
    ITEM_TYPE.CLUSTERS                 : {
        'key_name': 'host_name',
        'table'   : 'host',
        'where'   : {'register': {'$ne': '0'}, 'is_cluster': '1'},
    },
    ITEM_TYPE.CLUSTERTPLS              : {
        'key_name': 'name',
        'table'   : 'host',
        'where'   : {'register': '0', 'is_cluster': '1'},
    },
    ITEM_TYPE.HOSTS                    : {
        'key_name': 'host_name',
        'table'   : 'host',
        'where'   : {'register': {'$ne': '0'}, 'is_cluster': {'$ne': '1'}},
    },
    ITEM_TYPE.HOSTTPLS                 : {
        'key_name': 'name',
        'table'   : 'host',
        'where'   : {'register': '0', 'is_cluster': {'$ne': '1'}},
    },
    ITEM_TYPE.HOSTGROUPS               : {
        'key_name': 'hostgroup_name',
        'table'   : 'hostgroup',
    },
    ITEM_TYPE.SERVICESHOSTS            : {
        'key_name': 'service_description',
        'table'   : 'service',
        'where'   : {'register': {'$ne': '0'}, 'apply_on_type': {'$nin': [ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS, ITEM_TYPE.HOSTTPLS]}},
    },
    ITEM_TYPE.SERVICESHOSTTPLS         : {
        'key_name': 'service_description',
        'table'   : 'service',
        'where'   : {'register': '0', 'apply_on_type': {'$nin': [ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS, ITEM_TYPE.HOSTS]}, 'service_description': {'$exists': 1}},
    },
    ITEM_TYPE.SERVICESCLUSTERS         : {
        'key_name': 'service_description',
        'table'   : 'service',
        'where'   : {'register': {'$ne': '0'}, 'apply_on_type': ITEM_TYPE.CLUSTERS},
    },
    ITEM_TYPE.SERVICESCLUSTERTPLS      : {
        'key_name': 'service_description',
        'table'   : 'service',
        'where'   : {'register': '0', 'apply_on_type': ITEM_TYPE.CLUSTERTPLS, 'service_description': {'$exists': 1}},
    },
    ITEM_TYPE.SERVICETPLS              : {
        'key_name': 'name',
        'table'   : 'service',
        'where'   : {'register': '0', 'name': {"$exists": 1}},
    },
    ITEM_TYPE.CONTACTS                 : {
        'key_name': 'contact_name',
        'table'   : 'contact',
        'where'   : {'register': {'$ne': '0'}},
    },
    ITEM_TYPE.CONTACTTPLS              : {
        'key_name': 'name',
        'table'   : 'contact',
        'where'   : {'register': '0'},
    },
    ITEM_TYPE.CONTACTGROUPS            : {
        'key_name': 'contactgroup_name',
        'table'   : 'contactgroup',
    },
    ITEM_TYPE.ESCALATIONS              : {
        'key_name': 'escalation_name',
        'table'   : 'escalation',
    },
    ITEM_TYPE.NOTIFICATIONWAYS         : {
        'key_name': 'notificationway_name',
        'table'   : 'notificationway',
    },
    ITEM_TYPE.COMMANDS                 : {
        'key_name': 'command_name',
        'table'   : 'command',
    },
    ITEM_TYPE.BUSINESSIMPACTMODULATIONS: {
        'key_name': 'business_impact_modulation_name',
        'table'   : 'businessimpactmodulation',
    },
    ITEM_TYPE.MACROMODULATIONS         : {
        'key_name': 'macromodulation_name',
        'table'   : 'macromodulation',
    },
    ITEM_TYPE.RESULTMODULATIONS        : {
        'key_name': 'resultmodulation_name',
        'table'   : 'resultmodulation',
    },
    ITEM_TYPE.TIMEPERIODS              : {
        'key_name': 'timeperiod_name',
        'table'   : 'timeperiod',
    },
}

TRAD = {
    "state": {
        ITEM_STATE.STAGGING    : 'Staging',
        ITEM_STATE.NEW         : 'New',
        ITEM_STATE.WORKING_AREA: 'Working Area',
        ITEM_STATE.PRODUCTION  : 'Production',
    },
    "type" : {
        ITEM_TYPE.CLUSTERS                 : "Cluster",
        ITEM_TYPE.CLUSTERTPLS              : "Cluster template",
        ITEM_TYPE.HOSTS                    : "Host",
        ITEM_TYPE.HOSTTPLS                 : "Host template",
        ITEM_TYPE.HOSTGROUPS               : "Host group",
        ITEM_TYPE.SERVICESHOSTS            : "Check applied on Host",
        ITEM_TYPE.SERVICESHOSTTPLS         : "Check applied on Host template",
        ITEM_TYPE.SERVICESCLUSTERS         : "Check applied on Cluster",
        ITEM_TYPE.SERVICESCLUSTERTPLS      : "Check applied on Cluster template",
        ITEM_TYPE.SERVICETPLS              : "Check template",
        ITEM_TYPE.CONTACTS                 : "User",
        ITEM_TYPE.CONTACTTPLS              : "User template",
        ITEM_TYPE.CONTACTGROUPS            : "User group",
        ITEM_TYPE.ESCALATIONS              : "Escalation",
        ITEM_TYPE.NOTIFICATIONWAYS         : "Notification Way",
        ITEM_TYPE.COMMANDS                 : "Command",
        ITEM_TYPE.BUSINESSIMPACTMODULATIONS: "Business impact modulation",
        ITEM_TYPE.MACROMODULATIONS         : "Data modulation",
        ITEM_TYPE.RESULTMODULATIONS        : "Result modulation",
        ITEM_TYPE.TIMEPERIODS              : "Timeperiod",
    },
}


# TODO implement Lookup parameter for get functions
class DataProviderMongo(object):
    def __init__(self, mongo, database_cipher=None):
        self.mongo = mongo
        self.database_cipher = database_cipher
    
    
    @staticmethod
    def get_collection_name(item_type, item_state, item_source=''):
        try:
            table = DEF_ITEMS[item_type]['table']
            if item_state == ITEM_STATE.RAW_SOURCES:
                collection_name = COLLECTIONS.get(item_state, item_state) % (item_source, table)
            else:
                collection_name = COLLECTIONS.get(item_state, item_state) % table
        except Exception as e:
            raise DataException('[%s] Collections type[%s] - state[%s] not found %s' % (DataProviderMongo.__name__, item_type, item_state, e))
        
        return collection_name
    
    
    def get_collection(self, item_type, item_state, item_source=''):
        if not (isinstance(item_state, basestring)):
            raise DataException('[%s] get_collection : item_state [%s] must be a string' % (self.__class__.__name__, item_state))
        if not (isinstance(item_type, basestring)):
            raise DataException('[%s] get_collection : item_type [%s] must be a string' % (self.__class__.__name__, item_type))
        
        try:
            col_name = DataProviderMongo.get_collection_name(item_type, item_state, item_source)
            col = getattr(self.mongo, col_name)
        except Exception as e:
            raise DataException('[%s] Collections (type:[%s]-state:[%s]) not found. [%s]' % (self.__class__.__name__, item_type, item_state, e))
        
        return col
    
    
    def find_items(self, item_type, item_state='', item_source='', where=None, lookup=None):
        if not item_type:
            raise DataException('[%s] find_items : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] find_items : Please set item_state' % self.__class__.__name__)
        
        collection = self.get_collection(item_type, item_state, item_source)
        
        if not where:
            where = {}
        # In change collections we don't store item just change.
        if item_state != ITEM_STATE.CHANGES:
            where_for_type = self._get_where(item_type, item_state)
            where_for_type.update(where)
            where = where_for_type
        items = list(collection.find(where, lookup))
        if self.database_cipher:
            items = [self.database_cipher.uncipher(item, item_type, item_state) for item in items]
        return items
    
    
    def _get_where(self, item_type, item_state):
        if item_state == ITEM_STATE.CHANGES:
            return {}
        target_def_item = DEF_ITEMS[item_type]
        where = target_def_item.get('where', {}).copy()
        
        return where


def get_name_from_type(item_type, item):
    return item.get(DEF_ITEMS[item_type]['key_name'], '')


def main():
    errors = []
    try:
        con = Connection()
        db = con.synchronizer
    except Exception as e:
        errors.append("Unable to connect to the Mongo database : %s" % e)
    else:
        data_provider_mongo = DataProviderMongo(db)
        managed_states = (ITEM_STATE.STAGGING, ITEM_STATE.NEW, ITEM_STATE.WORKING_AREA, ITEM_STATE.PRODUCTION)
        for item_state in managed_states:
            for item_type in DEF_ITEMS.iterkeys():
                if item_type in (ITEM_TYPE.SERVICESCLUSTERTPLS, ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESCLUSTERS, ITEM_TYPE.SERVICESHOSTS):
                    continue
                known_names = set()
                list_item = data_provider_mongo.find_items(item_type, item_state)
                for item in list_item:
                    item_name = get_name_from_type(item_type, item)
                    if item_name in known_names:
                        errors.append("   * In area [ %s ], for item type \" %s \", several items have the name [%s]" % (TRAD['state'].get(item_state, item_state), TRAD['type'].get(item_type, item_type), item_name))
                    else:
                        known_names.add(item_name)
    
    
    for error in errors:
        print "\033[31m%s\033[0m\n" % error
    if errors:
        sys.exit(1)


if __name__ == "__main__":
    main()
