#!/bin/bash

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

# load common functions  and variables
. lib/common.sh

function display_help {
  printf "./install.sh [daemon options]\n" 
  printf "This tool will install all the Shinken Enterprise tool into the server."
  printf "By default it will enable all daemons. If you specify one or more daemon to enable, it will only enable theses ones.\n"
  printf "Options:\n"
  printf "  # Daemon activation at startup:\n"
  printf "    --pollernode:        enable the poller daemon (launch checks)\n"
  printf "    --reactionnernode:   enable the reactionner daemon (launch notifications)\n"
  printf "    --schedulernode:     enable the scheduler daemon (schedule checks)\n"
  printf "    --arbiternode:       enable the arbiter daemon (central distribution role)\n"
  printf "    --receivernode:      enable the receiver daemon (receive checks)\n"
  printf "    --synchronizernode:  enable the synchronizer daemon (manage configuration)\n"
  printf "    --brokernode:        enable the broker daemon (export data)\n"
  printf " Installation options:\n"
  printf "    --disable-add-public-epel:               if you have your own EPEL repository you must disable the installation of the EPEL repository that came with Shinken Enterprise\n"
  printf "    --disable-important-notices-user-input:  removes the need to hit a key when required during installation\n"
  printf "    --activate-encryption [<keyname>]:       creates a new encryption key with the name you give, and activates encryption.\n"
  printf "                                             If no name is given, you will be prompted for it.\n"
  printf "                                             Note that this can be done at a later stage.\n"
  printf "    --package-update-only-on-conflict:       Will not try to update package by default if already installed, and will only update\n"
  printf "                                             them if need by a conflict on a new package installation.\n"
  printf "    --skip-redhat-subscription-check:        Will skip to check if the RedHat subscription is enabled on this server.\n"
  exit 0
}  


# Flag that say we install everything
export centralnode="centralnode"
# display list of installed roles
export default_install_roles="arbiter synchronizer scheduler poller reactionner receiver broker"
export specific_install_roles=""
export installepel="installepel"
# By default we will try to update packages only on conflict for new packages
export PACKAGE_UPDATE_ONLY_ON_CONFLICT="false"
# By default, on readhat, we are looking if subscription is open, so we can look if repo are ok, but
# some server have local repos and no subscription, so skip the check in this case
export SKIP_REDHAT_SUBSCRIPTION_CHECK="false"

SANATIZE_LOGFILE="${OUTPUT_FOLDER}/sanitize.log"
activate_database_encryption=0
encryption_key_name=

while [ $# -ge 1 ] ; do
    case "$1" in
	-h | --help)
            display_help  # Call your function
	  # no shifting needed here, we're done.
	    exit 0
	    ;;
#      -u | --user)
#        username="$2" # You may want to check validity of $2
#	  shift 2
#	    ;;
	--pollernode)
          #  It's better to assign a string, than a number like "verbose=1"
	      #  because if you're debugging script with "bash -x" code like this:
	      #    if [ "$verbose" ] ...
	      #  You will see:
	      #    if [ "verbose" ] ...
	    pollernode="pollernode"
	    centralnode=""
	    specific_install_roles="$specific_install_roles poller"
	    shift
	    ;;
	--reactionnernode)
	    reactionnernode="reactionner"
	    centralnode=""
	    specific_install_roles="$specific_install_roles reactionner"
	    shift
	    ;;
	--schedulernode)
	    schedulernode="schedulernode"
	    centralnode=""
	    specific_install_roles="$specific_install_roles scheduler"
	    shift
	    ;;
	--receivernode)
	    receivernode="receivernode"
	    centralnode=""
	    shift
	    ;;
	--synchronizernode)
	    synchronizernode="synchronizernode"
	    centralnode=""
	    specific_install_roles="$specific_install_roles synchronizer"
	    shift
	    ;;
	--arbiternode)
	    arbiternode="arbiternode"
	    centralnode=""
	    specific_install_roles="$specific_install_roles arbiter"
	    shift
	    ;;
	--brokernode)
	    brokernode="brokernode"
	    centralnode=""
	    specific_install_roles="$specific_install_roles broker"
	    shift
	    ;;
	--disable-epel | --disable-add-public-epel)
	    installepel="donotinstallepel"
	    shift
	    ;;
	--disable-important-notices-user-input)
	    disable_user_input="disableuserinput"
	    shift
	    ;;
	--package-update-only-on-conflict)
	    export PACKAGE_UPDATE_ONLY_ON_CONFLICT="true"
	    shift
        ;;
   --skip-redhat-subscription-check)
       export SKIP_REDHAT_SUBSCRIPTION_CHECK="true"
       shift
       ;;
	--activate-encryption)
        activate_database_encryption=1
        shift
	    if [ $# -ge 1 ]; then
            encryption_key_name="$1"
        else
            encryption_key_name=
        fi
        shift
	    ;;
	--) # End of all options
            shift
	    break
	    ;;
	-*)
       echo "Error: Unknown option: $1" >&2
	    exit 1
	    ;;
	*)  # No more options
	    break
	    ;;
    esac
done


# Must be run as root
assert_is_root


# Be sure the tarball rights are valid
restore_all_tarball_valid_rights


# Check if not already installed. if so exit!
assert_is_not_installed


# Check if we are with a redhat or a centos OS. If not, exit
assert_redhat_or_centos

# During installation, display a Warning if Mongodb is already installed. Any existing Mongodb installation will be updated and options can be added by the installer (and sanatize-data script)
display_warning_if_mongo_already_installed

# Check that if there is a mongo installed, the version is known and valid
assert_only_valid_mongo


print_header "Welcome to Shinken Enterprise installer"

print_std_line "This installer is designed to easy install Shinken Enterprise $DISPLAY_VERSION"

if [ $centralnode ]; then
  print_std_line "It will install on this server the master server (All Shinken & storage)"
else
  print_std_line "It will install only the $specific_install_roles daemons"
fi

print_managed_distro

if [ $activate_database_encryption -eq 1 ]; then
    printf "\n\033[33mThis Shinken installation will enable protected fields encryption in the Synchronizer\n"
    if [ -n "$encryption_key_name" ]; then
        printf "The encryption key will be named \033[35m${encryption_key_name}\033[33m.\n"
    else
        printf "You will be asked for the encryption key name during the installation process.\033[33m\n"
    fi
fi

printf "\033[0m"

# Check global system state (dnsname, not running shinken, etc etc)
system_check

# No more iptables, good bye
disable_iptables


printf "\n\n"
print_header "Installing Shinken Enterprise dependencies"


# Install epel unless the user did ask to not install it
if [ $installepel == "installepel" ] ; then
    install_epel
    yum clean all >/dev/null
fi

# always install graphite, but enable it only on the broker node
install_only_graphite

if [ $centralnode ] || [ $brokernode ] ; then
    assume_graphite
fi

# in all case, be sure to setup the clean of graphite logs once a day
assume_graphite_clean

# always install mongo, but enable it only on the arbiter, synchronizer and broker nodes
install_or_update_mongodb_rpm

# then install mongo but only if not a simple poller mode
if [ $centralnode ] || [ $arbiternode ] || [ $synchronizernode ] ; then
    assume_mongodb
fi


printf "\n\n"
print_header "Installing Shinken Enterprise $DISPLAY_VERSION"


# MAIN DEPS="python-cherrypy python-simplejson nagios-plugins-all python-pymongo nmap python-pymongo-gridfs perl-Config-IniFiles perl-DateTime fping perl-Crypt-DES perl-Net-SNMP perl-Socket6 perl-Digest-HMAC perl-Digest-SHA1 perl-IO-stringy perl-DBD-MySQL python-gevent perl-Class-MethodMaker perl-Archive-Zip perl-Data-Dump perl-SOAP-Lite perl-XML-SAX perl-XML-NamespaceSupport perl-XML-LibXML perl-libxml-perl libuuid-devel perl-Nagios-Plugin redhat-lsb-core perl-Time-HiRes python-paramiko python-ldap pycairo"


print_std_line "Installing Shinken Enterprise dependencies ..."
install_enterprise_deps
print_ok_result_line

# Now we have all dependencies, we can check for selinux rules now (need selinux package)
assert_selinux

print_std_line "Installing Shinken Enterprise package ..."
dorpminstall "packages/shinken-enterprise-$RPM_VERSION.el$CENTOSRELEASE.x86_64.rpm" "Shinken Enterprise package"
# copy the 27 edition init.d script (shinken+shinken-gatherer). Theses line MUST BE EXECUTED just after the rpminstall
cp -rp dependencies/enterprise/init.d/* /etc/init.d/
chmod a+x /etc/init.d/shinken*
print_ok_result_line

# Remove comment on the pidfile line in mongo configuration files (bug in mongo init scripts on CentOS 6)
# Because mongo configuration files are initially set up by rpm, we need to fix this right after Mongo install
fix_02_07_03_remove_pidfile_comment_in_mongodb_config_files


# We overwrite shinken-packs file
overwrite_shinken_packs

# we need to change the nmap-mac-prefixes file
assume_nmap_mac_prefixes_file

# we should create user directories
assume_user_directories

# we should add discovery_rules default user file
assume_discovery_rules_file

# we should add nmap_mac_prefixes default user file
assume_nmap_mac_prefixes_user_file

#### Also install the server analyzer
install_server_analyzer

# Wait until packages are installed to set access rights for whisper (needs selinux tools to be installed)
set_whisper_rights

# Be sure it start on the boot
chkconfig --add shinken

setup_shinken_ssh_key

# Now fix default security parameters if need
update_security_parameters

# Install NagVis and restart httpd
assume_nagvis "$centralnode" "$arbiternode" "$brokernode"
NAGVIS_INSTALL_FAILED=$?

# Change NagVis default configuration files and enable Livestatus in broker
set_nagvis_configuration $NAGVIS_SHINKEN_PATH
set_nagvis_architecture_export_configuration $NAGVIS_SHINKEN_PATH

set_nagvis_configuration $NAGVIS_USER_PATH

printf "\n\n"
print_header "Starting Shinken Enterprise daemons"


# Create the context file with :
# * the valid daemons enabled
# * the installation path (will find it itselve)
# * the version we are installing
lib/set-install-context.py --code-version="$VERSION" --installer-version="$INSTALLER_VERSION" --pollernode=$pollernode --schedulernode=$schedulernode --centralnode=$centralnode --reactionnernode=$reactionnernode --arbiternode=$arbiternode --synchronizernode=$synchronizernode --receivernode=$receivernode
if [ $? != 0 ];then
    echo "ERROR: cannot create the context file (that track installation, updates, active daemons,...). As it is mandatory for Shinken Enterprise, we cannot finish the installation."
    exit 2
fi

if [ -n "$centralnode" ] || [ -n "$synchronizernode" ] ; then
    # Protected fields encryption setup and activation
    fix_02_05_00_add_crypted_passwords_install
fi


# Force enable of nagvis addons when installing
if [ $centralnode ] || [ $brokernode ] ; then
    shinken-addons-enable --force nagvis > /dev/null 2>&1
fi
if [ $centralnode ] || [ $arbiternode ] ; then
    shinken-addons-enable --force nagvis-shinken-architecture > /dev/null 2>&1
fi


# Disable poller if not central or pollernode
if [ ! $centralnode ] && [ ! $pollernode ]; then
    /usr/sbin/shinken-daemons-disable   poller
fi
if [ ! $centralnode ] && [ ! $schedulernode ]; then
    /usr/sbin/shinken-daemons-disable   scheduler
fi
if [ ! $centralnode ] && [ ! $brokernode ]; then
    /usr/sbin/shinken-daemons-disable   broker
fi
if [ ! $centralnode ] && [ ! $reactionnernode ]; then
    /usr/sbin/shinken-daemons-disable   reactionner
fi
if [ ! $centralnode ] && [ ! $receivernode ]; then
    /usr/sbin/shinken-daemons-disable   receiver
fi
if [ ! $centralnode ] && [ ! $synchronizernode ]; then
    /usr/sbin/shinken-daemons-disable   synchronizer
fi
if [ ! $centralnode ] && [ ! $arbiternode ]; then
    /usr/sbin/shinken-daemons-disable    arbiter
fi

# be sure to reenable it after we set the final init.d script
chkconfig --add shinken

# Fix if need writes in the /etc/shinken
chown -R shinken:shinken /etc/shinken/ 2>/dev/null


# The libXrender install for shinken can break http/graphite, so force apache to restart
# and so load the new lib version
/etc/init.d/httpd stop >/dev/null 2>/dev/null
# Only restart it if we need it
if [ $centralnode ] || [ $brokernode ] ; then
    /etc/init.d/httpd start >/dev/null 2>/dev/null
fi

# Fix specific cherrypy bug
fix_02_03_03_r76_cherrypyssl

fix_02_06_00_create_default_synchronizer_backup_directory


# Now all is OK, we can start the daemons
_start_shinken_daemons


if [ $centralnode ] || [ $synchronizernode ] ; then
    # Sleep some seconds so the synchronizer do the first load

    if service shinken-synchronizer status; then
        print_std_line "Checking and validating default configuration objects..."

        sleep 15
        # wait for the synchronizer to respond at raw_stat
        until $(curl --output /dev/null --silent --head --fail http://localhost:7765/get_raw_stats); do
            sleep 1
            printf "-"
        done
        while [ $(curl -s http://localhost:7765/get_api_source_controller_importing) = "true" ]; do
            sleep 1
            printf "."
        done
        sleep 1
        curl -s http://localhost:7765/validate_all_elements  > /dev/null
        sleep 5
        curl -s http://localhost:7765/commit_all > /dev/null
        print_ok_result_line
        /etc/init.d/shinken stop >/dev/null 2>/dev/null
        /etc/init.d/shinken start >/dev/null 2>/dev/null
    else
        printf "\033[31mFAIL :\033[30m The synchronizer is not running\n\n"
        printf "Please run \033[35mshinken-healthcheck\033[30m and check the logs for more information.\n"
        exit 1
    fi
fi


printf "\n\n"
print_header "Checking installation"

print_std_line "Note that you can check a Shinken Enterprise installation with the command shinken-healthcheck"


# warn that the vmware sdk is NOT installed by default
check_vmware_sdk


# Save this update/install path so we know after the install where to ask more install things if need
save_intall_path

service shinken status > /dev/null 2>/dev/null
if [ $? -ne 0 ]; then
    printf "\n\n"
    printf "\033[31mSome Shinken daemons could not start.\033[0m\n"
    printf "Please run the command \033[93m/usr/sbin/shinken-healthcheck\033[0m and check the logs to get more informations.\n"
    exit 1
fi



printf "\n\n"
print_header "End"

if [ $centralnode ] || [ $brokernode ] || [ $synchronizernode ] ; then
    if [ $centralnode ] || [ $synchronizernode ] ; then
        display_protected_fields_warning
    fi
    printf "\nShinken installation status\n"
    printf "    You can now connect to Shinken Enterprise with the account admin/admin:\n"
    if [ $centralnode ] || [ $brokernode ] ; then
        printf "    Visualization:  \033[92mhttp://$EXTERNAL_IP:7767\033[0m\n"
    fi

    if [ $centralnode ] || [ $synchronizernode ] ; then
        printf "    Administration: \033[92mhttp://$EXTERNAL_IP:7766\033[0m\n"
    fi
fi

if [ $NAGVIS_INSTALL_FAILED -eq 0 ]; then
    printf "\nNagVis installation status\n"

    printf "  - The addon 'nagvis-shinken-architecture' has been installed during the installation process.\n"
    if [ $centralnode ] || [ $arbiternode ] ; then
        printf "    Shinken architecture can be visualized at the following address:\n"
    else
        printf "    \033[33mThis addon is currently disabled because no Arbiter is enabled on this server.\n"
        printf "    With an enabled Arbiter on this server, it can be enabled with the 'shinken-addons-enable nagvis-shinken-architecture' command.\033[0m\n"
        printf "    Once enabled, it will be accessible with the following address.\n"
    fi

    printf "    NagVis Shinken architecture:  \033[92mhttp://$EXTERNAL_IP$NAGVIS_SHINKEN_WEB_PATH\033[0m\n"

    if [ $centralnode ] || [ $arbiternode ] ; then
        printf "    You can now connect to this NagVis installation with any Shinken administrator account credentials\n\n"
    else
        printf "\n\n"
    fi


    printf "  - The addon 'nagvis' has been installed during the installation process.\n"
    if [ $centralnode ] || [ $brokernode ] ; then
        printf "    Custom maps can be viewed and created at the following address:\n"
    else
        printf "    \033[33mThis addon is currently disabled because no Broker is enabled on this server.\n"
        printf "    It can be enabled manually with the 'shinken-addons-enable nagvis' command.\033[0m\n"
        printf "    Once enabled, it will be accessible with the following address.\n"
    fi

    printf "    NagVis:                       \033[92mhttp://$EXTERNAL_IP$NAGVIS_USER_WEB_PATH\033[0m\n"

    if [ $centralnode ] || [ $brokernode ] ; then
        printf "    You can now connect to this NagVis installation with any Shinken account credentials\n\n\n"
    else
        printf "\n\n"
    fi
fi

# If on CentOS 6, restart terminal to refresh groups of user (needed for architecture-export)
if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
    printf "\033[33mDue to technical limitations on CentOS 6, the shell session needs to be restarted to account for user rights changed by Shinken installation before continuing.\n"
    printf "It can be done with the command \"su -\", or by opening a new terminal window.\033[0m\n\n"
fi