#!/usr/bin/env bash

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

# Common lib, variables and functions for update & install.sh

export DISPLAY_VERSION=02.08.01-OPTIONS_local_repository_added.fr
# And real package version
export VERSION=02.08.01-OPTIONS_local_repository_added.fr
# NOTE: rpm file have a special name, as it do not have - for release but . instead
export RPM_VERSION=V02.08.01.OPTIONS_local_repository_added.fr
# if the build was special, will be put in this value (void by default)
export INSTALLER_VERSION="PACKAGE-008"

export pollernode=""

# Force English LANG so yum reports errors in English, and not french or such silly things
# NOTE: both LANGUAGE and LANG even if only LANGUAGE is need, because newcomers know only about the LANG=
export LANGUAGE=en_US.UTF8
export LANG=en_US.UTF-8

export MAINTENANCE_FILE="/var/lib/shinken/MAINTENANCE_MODE_ENABLED"


# don't be impacted by user alias for classic commands
unalias rm cp mv grep python2 >/dev/null 2>/dev/null

# Set umask to avoid problems on files created by the installation script
umask 0022

# Manage different centos/redhat versions
OSRELEASE=$(cat /etc/redhat-release)
OSRE="^(.*)release ([6,7])\.([0-9])(.*)$"
if [[ "$OSRELEASE" =~ $OSRE ]]
then
    export CENTOSRELEASE="${BASH_REMATCH[2]}"
    CENTOSSUBRELEASE="${BASH_REMATCH[3]}"
fi

# Which distro are we using
grep -i "red hat" /etc/redhat-release > /dev/null
if [ $? == 0 ]; then
   DISTRO="REDHAT"
fi

grep -i "centos" /etc/redhat-release > /dev/null
if [ $? == 0 ]; then
   DISTRO="CENTOS"
fi

# Final value with the linux version
LINUX_VERSION="$DISTRO-$CENTOSRELEASE"

# The Synchronizer config file
SYNCHRONIZER_CFG=/etc/shinken/synchronizer.cfg

# The file in which the protected key will be stored
KEYFILE=/etc/shinken/secrets/protected_fields_key

# Retrieve external IP address
## Get network interface used by the default gateway
EXTERNAL_NETWORK_INTERFACE=$(ip route | grep -e "^default" | head -n 1 | cut -d' ' -f 5)
## Get ip address of said interface
EXTERNAL_IP_WITH_MASK=$(ip -4 -o addr show dev $EXTERNAL_NETWORK_INTERFACE | cut -d' ' -f 7)
EXTERNAL_IP=${EXTERNAL_IP_WITH_MASK%%/[0-9]*}

sestatus | grep "SELinux status.*enabled" > /dev/null 2>&1
SELINUX_ENABLED=$?

# Now look at python version
PYTHONVERSION=$(python2 --version 2>&1)
PYTHONRE="^Python\s([2])\.([0-9])\.([0-9]+)$"
if [[ "$PYTHONVERSION" =~ $PYTHONRE ]]
then
    PYTHONMAJOR="${BASH_REMATCH[1]}"
    PYTHONMINOR="${BASH_REMATCH[2]}"
fi


CONTEXT_FILE="/var/lib/shinken/context.json"
SANATIZE_LOGFILE="~/sanatize.update.log"

# Gets currently installed version from context file, but only if exist
# WARNING: so should be used only in the fix_** calls as it is not available in the install path
PREVVERSION=`sed -ne "s/.*\?\"current_version\" *: *\"\([^\"]*\)\".*/\1/p" $CONTEXT_FILE 2>/dev/null`

# An error? just bailout with a red error message
function doexit {
  printf "\033[91m $1 \033[0m\n"
  exit 2
}

function print_header {
   python lib/color_print.py "header" "$1"
}

function print_red_header {
   python lib/color_print.py "red-header" "$1"
}

function print_std_line {
   python lib/color_print.py "std-line" "$1"
}


function print_std_line_no_end {
   python lib/color_print.py "std-line-no-end" "$1"
}

function print_ok_result_line {
   python lib/color_print.py "ok-result-line"
}

function print_note_line {
   python lib/color_print.py "note-line-grey" "$1"
}


function print_OK {
   python lib/color_print.py "OK"
}


function print_WARNING {
   python lib/color_print.py "WARNING"
}

function print_ERROR {
   python lib/color_print.py "ERROR"
}

function print_magenta_no_end {
   python lib/color_print.py "magenta-no-end" "$1"
}


# Functions for checking if installed/not installed state for install/update
# INSTALL: should not be installed, so should not have a CONTEXT
function assert_is_not_installed {
    if [ -f $CONTEXT_FILE ]; then
        doexit "There is already an installed version on this server, please use the update.sh command instead"
    fi
}
# UPDATE: should be installed, so with a context file
function assert_is_installed {
    if [ -f $CONTEXT_FILE ]; then
        return
    fi
    doexit "There is no Shinken Enterprise installed on this server, please use the install.sh command instead"
}


function assert_redhat_or_centos {
   if [ "X$DISTRO" == "XCENTOS" ]; then
	  if [ $CENTOSRELEASE -eq 7 ]; then
	     if [ $CENTOSSUBRELEASE -lt 2 ] || [ $CENTOSSUBRELEASE -gt 9 ];  then  # in the branch 7, only the 7.2 to 7.9 are managed
	         doexit "Your system $OSRELEASE is not managed by this installer. Only the CentOS 7.2 to 7.9 are managed."
	     fi
	  fi
	  # if 6, always OK
      return
   fi

   if [ "X$DISTRO" == "XREDHAT" ]; then
	  if [ $CENTOSRELEASE -eq 7 ]; then
	     if [ $CENTOSSUBRELEASE -lt 2 ] || [ $CENTOSSUBRELEASE -gt 9 ];  then  # in the branch 7, only the 7.2 to 7.9 are managed
	         doexit "Your system $OSRELEASE is not managed by this installer. Only the RedHat 7.2 to 7.9 are managed."
	     fi
	  fi
	  if [ "X$SKIP_REDHAT_SUBSCRIPTION_CHECK" == "Xfalse" ]; then  # By default we are checking the redhat subscription
        # Maybe this RedHat is not registred at redhat, if so yum call will fail.
        sub=$(LANG=C subscription-manager status)
         if [ $? -ne 0 ]; then
             # Ok not a registered system
            doexit "Your RedHat system does have subscription issues and it will cause yum installation to fail. Please fix it before install Shinken Enterprise.\nHere is the error reported by the subscription-manager command:\n\n$sub."
         fi
     else
         print_std_line "RedHat: the subscription check is disabled, please be sure the server have acces to valid local repository"
         echo ""
         echo ""
     fi
	  # if 6, always OK
      return
   fi

   doexit "Your system $OSRELEASE is not managed by this installer."
}


function assert_no_duplicates {
  print_std_line "Checking Synchronizer database consistency:"
  lib/preupdate-checks/check-db-for-duplicates.py
  RC=$?
  if [[ $RC -ne 0 ]]; then
    doexit "To prevent database corruption, the update is aborted. Please fix the issue before running update.sh again."
  else
    print_ok_result_line
  fi
}

function assert_protected_fields_conf_not_corrupted {
    local cfg_file=$SYNCHRONIZER_CFG

    local nb_definitions
    local parameter_to_check
    local has_errors=0
    local errors=""

   for parameter_to_check in protect_fields__activate_encryption protect_fields__encryption_keyfile protect_fields__substrings_matching_fields; do
       nb_definitions=$(grep -c "${parameter_to_check}" ${cfg_file})
       if [ $nb_definitions -gt 1 ]; then
            if [ "$has_errors" == "0" ] ; then
                print_red_header "PROTECTED FIELDS CONFIGURATION CORRUPTED"
            fi
            print_std_line_no_end "* The parameter [${parameter_to_check}] is defined multiple times in $cfg_file"
            print_ERROR
            has_errors=1
       fi
   done
    if [ "$has_errors" == "1" ]; then
        doexit "To prevent database corruption, the update is aborted. Please fix the issue before running update.sh again."
    fi


}
function print_managed_distro {
  print_std_line "Managed linux distributions: "
  print_std_line_no_end "Centos 6 "
  if [[ "X$LINUX_VERSION" == "XCENTOS-6" ]]; then
     print_magenta_no_end "Detected"
  fi
  printf '\n'

  print_std_line_no_end "Centos 7 "
  if [[ "X$LINUX_VERSION" == "XCENTOS-7" ]]; then
     print_magenta_no_end "Detected"
  fi
  printf '\n'

  print_std_line_no_end "RedHat 6 "
  if [[ "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
     print_magenta_no_end "Detected"
  fi
  printf '\n'

  print_std_line_no_end "RedHat 7 "
  if [[ "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
     print_magenta_no_end "Detected"
  fi
  printf '\n'
}

# Compare version strings
#0) op='=
#1) op='>
#2) op='<
function vercomp () {
    if [[ $1 == $2 ]]
    then
        return 0
    fi
    local IFS=.
    local i ver1=($1) ver2=($2)
    # fill empty fields in ver1 with zeros
    for ((i=${#ver1[@]}; i<${#ver2[@]}; i++))
    do
        ver1[i]=0
    done
    for ((i=0; i<${#ver1[@]}; i++))
    do
        if [[ -z ${ver2[i]} ]]
        then
            # fill empty fields in ver2 with zeros
            ver2[i]=0
        fi
        if ((10#${ver1[i]} > 10#${ver2[i]})) 2>/dev/null
        then
            return 1
        fi
        if ((10#${ver1[i]} < 10#${ver2[i]})) 2>/dev/null
        then
            return 2
        fi
    done
    return 0
}


# We try to install bunh of rpm, but only not installed one, and if failed, skip with yum instead of rpm only
# $1: list of rpm
# $2: text about what we are installing
# $3: forceupdate or void, should we force the install, even if already installed
function dorpminstall {
    # First loop over all packages, if already installed, skip them
    export ALL_INSTALLED=1
    # keep a list of not installed packages
    export NOT_INSTALLED=''

    # Loop for knowing whoch packages are not installed, or need an update
    for ii in `ls -1 $1`;do
        # If we did force the
        if [ $3 ]; then
            export ALL_INSTALLED=0
            export NOT_INSTALLED="$NOT_INSTALLED $ii"
            continue
        fi
        # warning: do not look at deps here, we will manage it after, means that a new version is already installed, not a problem
        export PNAME=`rpm -qp --queryformat '%{NAME}\n'  $ii 2>/dev/null`
        export PVERSION=`rpm -qp --queryformat '%{VERSION}-%{RELEASE}\n'  $ii 2>/dev/null`
        rpm -qi $PNAME > /dev/null 2>/dev/null
        # Look if already isntalled, or not :)
        if [ $? -ne 0 ]; then
            export ALL_INSTALLED=0
            export NOT_INSTALLED="$NOT_INSTALLED $ii"
        else
            # We can look at package version, but only if the install/update allow it (some distro with fic repo will be
            # a problem if try to update packages too high)
            if [ "$PACKAGE_UPDATE_ONLY_ON_CONFLICT" != "true" ];then
                # get local version because maybe we need to install it
                export LVERSION=`rpm -qi --queryformat '%{VERSION}-%{RELEASE}\n' $PNAME | tail -n 1`
                # If version do not match, install it
                vercomp  $PVERSION  $LVERSION
                SAME=$?
                # If PVERSION > LocalVERSION
                if [ $SAME -eq 2 ] || [ $SAME -eq 1 ]; then
                            export ALL_INSTALLED=0
                            export NOT_INSTALLED="$NOT_INSTALLED $ii"
                fi
            fi
        fi
    done

    #echo "NOT INSTALLAED" $NOT_INSTALLED
    # If not all installed, ask for all install :)
    if [ $ALL_INSTALLED -ne 1 ]; then
	    # ok not alreadt installed, try to install it
	    # if force, set allow oldpackage to be installed
        export FORCE_OPTIONS=""
	if [ $3 ]; then
        export FORCE_OPTIONS=" --oldpackage --force "
    fi
	rpm -Uh --nosignature --quiet $FORCE_OPTIONS $NOT_INSTALLED >/tmp/install.txt 2>&1
	if [ $? -ne 0 ]; then
	    cat /tmp/install.txt >> "${OUTPUT_FOLDER}/shinken.enterprise.install.detail.log"
	    # We should be in a no end line, so force to get back before print something
	    printf '\n'
	    print_note_line "Local packages installation uncomplete, switching to yum to get missing packages."
	    yum --nogpgcheck -y --rpmverbosity=error --errorlevel=1 --color=auto install $NOT_INSTALLED > /tmp/install.txt  2>&1
	    export RES=$?
	    if [ $RES -ne 0 ]; then
            # Maybe it's just that all depsare already installed
		    grep "Error: Nothing to do" /tmp/install.txt > /dev/null
            if [ $? -ne 0 ]; then

                # Maybe there are some old deps, if so try to install them and then relaunch the install
                cat /tmp/install.txt | grep 'Error: Package' > /dev/null
                # If cannot find them, that's the end guy
                if [ $? -ne 0 ]; then
                    cat /tmp/install.txt
                    doexit "Cannot install $2 packages, please check your YUM installation."
                fi
                cp /tmp/install.txt /tmp/install.txt.beforebogus
                # ok we did catch some, at least try to manage them
                cat /tmp/install.txt | grep 'Error: Package' | cut -d'/' -f2 | cut -d'(' -f1 > /tmp/bogus_install.txt
                export LAST_TRY_INSTALL=`cat /tmp/bogus_install.txt | tr '\n' ' '`
                LAST_TRY_INSTALL=`python lib/cut-package-name.py $LAST_TRY_INSTALL`
                print_note_line "Trying to install some old packages from repositories instead: $LAST_TRY_INSTALL"
                # Give yum only the package name WITHOUT the version to allow it to upgrade it
                yum --nogpgcheck -y --rpmverbosity=error --errorlevel=1 --color=auto install $LAST_TRY_INSTALL > /tmp/install_bogus.txt 2>&1
                export RESBOGUS=$?
                if [ $RESBOGUS -ne 0 ]; then
                    grep "Error: Nothing to do" /tmp/install_bogus.txt > /dev/null
                    if [ $? -ne 0 ]; then
                        cat /tmp/install.txt
                        doexit "Cannot install $2 packages, please check your YUM installation."
                    fi
                fi
                print_note_line "Restart the part installation after fixing the old packages"
                # If ok, restart the whole install with theses packages now :)
                dorpminstall "$1" "$2"
                return;
            else
                return;
            fi
	    else
		# Installation valid thanks to yum
		return;
	    fi
	    cat /tmp/install.txt
	    doexit "Cannot install $2 packages, please check your YUM installation."
	fi
    else
	    #print_note_line "All packages are already installed, skip it."
	    # print nothing, we did nothing
	    printf ""
    fi
}


function __system_start_daemon {
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        /etc/init.d/$1 start >/dev/null 2>/dev/null
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        systemctl start $1 >/dev/null 2>/dev/null
    fi
}


function __system_enable_daemon {
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        chkconfig $1 on
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        systemctl enable $1  2>/dev/null
    fi
}


function __system_disable_daemon {
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        chkconfig $1 off
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        systemctl disable $1  2>/dev/null
    fi
}


function __system_stop_daemon {
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        /etc/init.d/$1 stop >/dev/null 2>/dev/null
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        systemctl stop $1 >/dev/null 2>/dev/null
    fi
}


function _start_shinken_daemons {
   print_std_line "Starting Shinken ... "

   # Since user groups need to be refreshed before starting Shinken (NagVis installation adds root and shinken users to apache group), we need to start Shinken using a new shell
   # This fix is only needed for CentOS 6, the problem was not observed on CentOS 7
   # NOTE: #SEF-9823: NOT with "su -" as maybe the root user have /sbin/nologin shell. But sudo is always OK
   if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
       sudo bash /etc/init.d/shinken start
   else  # centos 7 and more
       /etc/init.d/shinken start
   fi

   print_ok_result_line
}


# FOR UPDATE ONLY: when updating, we are backuping the configuration if we are a synchronizer
function mongod_check {
    IS_SYNCHRONIZER=$(shinken-daemons-has synchronizer 2>/dev/null)
    # Only look at the mongodb is we are the synchronizer (backuping configuration)
    if [[ "$IS_SYNCHRONIZER" == "NO" ]];
    then
        return
    fi
    print_std_line "Checking mongod for backup during update process ..."
    service mongod status > /dev/null
    if [ $? -ne 0 ] ;
        then printf "\033[93mMongo database is not available, please check your Mongo service and try the update again.\033[0m\n";
        exit 1
    else
        print_ok_result_line
    fi
}

function assert_jemalloc_memory {
    # For jemalloc, we set thread preallocated arena to 0, save LOT
    # of memory for small env
    if [ ! -f "/etc/malloc.conf" ];then
        echo "narenas:1" > /etc/malloc.conf
    fi
}


# Install and update must be run as root
function assert_is_root {
    if [[ "$(id -u)" != "0" ]]; then
       doexit "The install and update must be run as root. Exiting."
    fi
}


# The user can have a bad rights after the tar (like a not root with a umask 0077 for example). If so, fix all
# rights on the directory
function restore_all_tarball_valid_rights {
    RESTO_LOG=/tmp/tarball_restauration.log
    > $RESTO_LOG
    LANG=C setfacl --restore=tarball.file_rights >>$RESTO_LOG 2>>$RESTO_LOG
    if [ $? != 0 ];then
        doexit "Cannot restore all installation file rights. Please look at the log $RESTO_LOG for errors."
    fi
}


function system_check {
# First some checks, so we can cleanly exit before go on a box that is not correctly installed
# mainly valid for apache startup, but can also bea problem for daemons
    printf "\n\n"
    #printf "*****************************************************************************\n"
    #printf "****       \033[92mSystem checks\033[0m                                             ********\n"
    #printf "*****************************************************************************\n"
    print_header "System checks"

    print_std_line "Checking network configuration:"

    ping `hostname` -n -c 1 > /dev/null 2>&1
    if [ $? -ne 0 ] ; then
         print_WARNING
         print_std_line "Your hostname `hostname` cannot be resolved in DNS. It can raise an error in apache. Please add it to your DNS server or update your /etc/hosts file with it" ;
    else
         print_ok_result_line
    fi

    # Disabling IPV6 localhost so only ipv4 localhost is matched
    sed -i 's/^::1/#::1/'  /etc/hosts

    # Assert that our limit file is present, to increase process/threads limits
    cp -f dependencies/limits.d/*.conf  /etc/security/limits.d/ 2>/dev/null

    # If the ssh is allowed to do DNS lookup, disable it, because it can cause ssh tunnel to be freaking long
    grep -E '^UseDNS' /etc/ssh/sshd_config > /dev/null
    if [ $? != 0 ]; then
        print_std_line "Disabling the DNS lookup in the SSH configuration"
        printf "\nUseDNS  no\n\n" >> /etc/ssh/sshd_config
        service sshd reload
    fi

    # assert than jemalloc tuning is done
    assert_jemalloc_memory

    print_std_line "Stopping Shinken daemons ..."
    /etc/init.d/shinken stop 2>/dev/null >/dev/null
    print_ok_result_line

}


function assert_selinux {
    print_header "Checking SeLinux"
    # If there is selinux, disable it! but only under centos6
    if [[ "X$LINUX_VERSION" == "XCENTOS-6"  ]]; then
        print_std_line_no_end "Checking SELinux configuration:"
        grep 1 /selinux/enforce 2>/dev/null >/dev/null
        if [ $? -ne 1 ]; then
            print_WARNING
            print_std_line "SELinux enforce mode detected."
            print_std_line "Disabling SELinux."
            if [ -f /selinux/enforce ]; then
                echo 0 >/selinux/enforce 2>/dev/null
            fi
            cp -f dependencies/selinux /etc/selinux/config
            chmod 644 /etc/selinux/config
        fi
    fi

    # 7 version need to bypass selinux with specific nmap rules
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7"  ]]; then
        print_std_line "Updating SELinux configuration (this can take some minutes) ..."
        # force all selinux rules, whatever we do (install or update)
        for ii in `ls -1 dependencies/selinux-rules/*pp`; do
            rule_name=$(basename "$ii" | cut -d. -f1)
            print_std_line "  rule $rule_name"
            checkmodule -M -m -o /tmp/$rule_name.mod dependencies/selinux-rules/$rule_name.te 2>/dev/null >/dev/null
            semodule_package -o  /tmp/$rule_name.pp -m /tmp/$rule_name.mod
            semodule -i /tmp/$rule_name.pp
        done

        # For allowing graphite/apache to spawn a ssh if mongo+ssh is used, we need to add this
        # selinux auth
        setsebool -P nis_enabled 1
    fi
    print_ok_result_line
}


function set_maintenance_mode {
    # Create the maintenance file to prevent shinken start by another user/script/tool/...
    mkdir -p /var/lib/shinken/ 2>/dev/null >/dev/null
    chown shinken:shinken /var/lib/shinken/
    chmod 755 /var/lib/shinken/
    touch ${MAINTENANCE_FILE}
}



function disable_iptables {
    # no iptables service on vanilla centos7, but there is on redhat
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6"  ||  "X$LINUX_VERSION" == "XREDHAT-7" || "X$LINUX_VERSION" == "XCENTOS-7" ]]; then
        print_std_line_no_end "Disabling iptables rules."
        print_OK
        print_note_line "This is need to be sure you can easily access to the UIs from outside this server."
        print_note_line "Feel free to reenable them after adding the ports to the INPUT rules:"
        PORTS="80 443 7766 7767 7768 7769 7770 7771 7772 7773"
        for port in $PORTS; do
           print_note_line "$port"
        done

        # Note: iptables for 6 version, firewalld for 7
        if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
           __system_stop_daemon iptables
           __system_disable_daemon iptables
        else
           __system_stop_daemon firewalld
           __system_disable_daemon firewalld
        fi
    fi
}



function check_vmware_sdk {
    if [ ! -f /usr/bin/vmware-cmd ]; then
        printf "\n\n"
        #printf "*****************************************************************************\n"
        #printf "***************           \033[92m   Installing VMWare SDK\033[0m                    ******************\n"
        #printf "*****************************************************************************\n"
        print_header "Installing VMWare SDK"
        print_std_line "In order to execute VMware checks, we need to install the VMware SDK. "
        print_std_line_no_end " * Installation in progress... "
        tools/install-vmware-sdk.sh 2>/dev/null >/dev/null
        if [ $? != 0 ]; then
           print_WARNING
           print_std_line "The VMware SDK installation did fail, please launch the script `readlink -e tools/install-vmware-sdk.sh`"
        else
           print_OK
        fi
    fi
}

function check_encryption_status {
    # IF the lib is not there, we are upgrading from a pre-2.5 version so Encryption can't be enabled
    if [ ! -f /var/lib/shinken/libexec/libs/shinken_protected_fields.sh ] ; then
        return
    fi

    source /var/lib/shinken/libexec/libs/shinken_protected_fields.sh
    source ../../../sources/shinken-patched/libexec/libs/shinken_protected_fields.sh 2>/dev/null # For Pycharm

    encryption_activated=$(spf_get_activated_from_synchronizer_cfg)
    if [ $encryption_activated = "true" ]; then
        key_file=$(spf_get_keyfile_name_from_synchronizer_cfg)
        key_name=$(spf_get_key_name_from_key_file "$key_file")
        printf "\n"
        printf "$(show_critical_info "Encryption is already activated  with key ") $(show_data "$key_name")\n\n"
        printf "$(show_important_info "There is no need to use the") $(show_command_parameter  --activate-encryption) $(show_important_info option.)\n\n"
        printf "$(show_important_info "If you want to change encryption key, this must be done separately from the update process.")\n"
        printf "$(show_important_info "Please see the documentation for more information about the encryption key change procedure.")\n\n"
        exit 1
    fi
}


# If the /etc/shinken inner repertories are not present, create them so the update will be clean
function assume_etc_shinken_directories {
    # note: list as of 2.03.03 version
    export SUB_DIRECTORIES="arbiters brokers certs daemons hosts modules objects packs pollers reactionners realms receivers resource.d schedulers services sources synchronizers taggers"
    for ii in $SUB_DIRECTORIES; do
        export DIR_PATH=/etc/shinken/$ii
        if [ ! -d $DIR_PATH ]; then
            mkdir -p $DIR_PATH 2>/dev/null
            chown -R shinken:shinken $DIR_PATH 2>/dev/null
        fi
    done
}

function install_enterprise_deps {
    # We are giving mainly redhat6 packages, but centos are a bit different for some, so manage this case
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" ]]; then
       dorpminstall "dependencies/enterprise/packages/redhat-centos/6.X/centos/*rpm" "Shinken Enterprise dependencies for centos 6.X"
       dorpminstall "dependencies/enterprise/packages/redhat-centos/6.X/common/*rpm" "Shinken Enterprise dependencies"
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" ]]; then
            dorpminstall "dependencies/enterprise/packages/redhat-centos/7.X/centos/*rpm" "Shinken Enterprise dependencies for centos 7.2"
	        # currently all is in common until we manage redhat 7.2
	        #dorpminstall "dependencies/enterprise/packages/redhat-centos/7.X/common/*rpm" "Shinken Enterprise dependencies"
    fi
    if [[ "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
	    dorpminstall "dependencies/enterprise/packages/redhat-centos/6.X/redhat/*rpm" "Shinken Enterprise dependencies for redhat"
	    dorpminstall "dependencies/enterprise/packages/redhat-centos/6.X/common/*rpm" "Shinken Enterprise dependencies"
    fi

    if [[ "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
            dorpminstall "dependencies/enterprise/packages/redhat-centos/7.X/centos/*rpm" "Shinken Enterprise dependencies for centos 7.2"
	        # currently all is in common until we manage redhat 7.2
	        #dorpminstall "dependencies/enterprise/packages/redhat-centos/7.X/common/*rpm" "Shinken Enterprise dependencies"
    fi

    # install pymongo part
    assume_pymongo

    # install pyexcel part
    assume_pyexcel

    # Fix suid for check_icmp and check_dhcp
    chmod o+x /usr/lib64/nagios/plugins/check_icmp /usr/lib64/nagios/plugins/check_dhcp

    # Fix execution for ping command
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
            chmod u+s /usr/bin/ping
    elif [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6 " ]]; then
            chmod u+s /bin/ping
    fi

    # VMware part
    dorpminstall "dependencies/vmware/packages/*rpm" "VMWare"
    # Specific to REDHAT7/CENTOS7: use a specific lib-www in /opt
    # It is used in the VICommon.pm specific version of the VMWare SDK
    grep -i "centos" /etc/redhat-release > /dev/null
    if [ $? -eq 0 ]; then
        if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
            tar xfz "dependencies/vmware/libs/redhat-centos/7X/vmware_perl_libwww.tar.gz"  --directory=/opt
        fi
    fi

    # Oracle part
    dorpminstall "dependencies/oracle/redhat-centos/common/*rpm" "Oracle common forredhat/centos 6.X and 7.X"
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6 " ]]; then
            dorpminstall "dependencies/oracle/redhat-centos/6.X/common/*rpm" "Oracle dependencies for redhat/centos 6.X"
        fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
	    dorpminstall "dependencies/oracle/redhat-centos/7.X/common/*rpm" "Oracle dependencies for redhat /centos 7.X"
    fi

    # Mssql part
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        dorpminstall "dependencies/mssql/6.X/common/*rpm" "Sqlserver dependencies for redhat/centos 6.X"
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        dorpminstall "dependencies/mssql/7.X/common/*rpm" "Sqlserver dependencies for redhat/centos 7.X"
    fi
    # overwrite the freetds so we are ok on the protocolversion for everyone
    cp -f dependencies/mssql/freetds.conf /etc/freetds.conf

}

function setup_shinken_ssh_key {
    # Create a sshkey for the shinken user, but only if not already exists
    if [ ! -f ~shinken/.ssh/id_rsa ]; then
        su - shinken -c "ssh-keygen -q -b 2048 -f ~/.ssh/id_rsa -P ''"
    fi

    hostname=$(hostname)
    authkeys_file=~shinken/.ssh/authorized_keys
    if [ -f $authkeys_file ] ; then
        # Add the key only if it is not yet in the file
        grep -q "shinken@${hostname}" $authkeys_file || cat ~shinken/.ssh/id_rsa.pub >> $authkeys_file
    else
        cat ~shinken/.ssh/id_rsa.pub > $authkeys_file
        chown shinken:shinken $authkeys_file
        chmod 600 $authkeys_file
    fi
}

function update_security_parameters  {
    # Updating security parameters
    print_std_line "Updating security parameters ... "
    # NOTE: 256random cahr auth based
    export AUTHSECRET=AUTH-SECRET-$( python -c "import string;import random;print ''.join([random.choice(string.ascii_letters + string.digits+'_-') for i in range(256)])" )
    sed -i "s/THIS SHOULD BE CHANGED BEFORE PRODUCTION/$AUTHSECRET/" /etc/shinken/synchronizer.cfg
    sed -i "s/THIS SHOULD BE CHANGED BEFORE PRODUCTION/$AUTHSECRET/" /etc/shinken/modules/webui.cfg

    if grep -q 'MASTERKEY-SHOULDBECHANGED' /etc/shinken/synchronizer.cfg ; then
        export MASTERKEY=MASTERKEY-$( python -c "import string;import random;print ''.join([random.choice(string.ascii_letters + string.digits+'_-') for i in range(256)])" )
    else
        export MASTERKEY=$(grep master_key= /etc/shinken/synchronizer.cfg |cut -f2 -d=)
    fi

    sed -i "s/MASTERKEY-SHOULDBECHANGED/$MASTERKEY/" /etc/shinken/synchronizer.cfg
    if grep -q 'master_key' /etc/shinken/modules/synchronizer-import.cfg ; then
        sed -i "s/master_key.*/master_key      $MASTERKEY/" /etc/shinken/modules/synchronizer-import.cfg
    else
        sed -i "/^\s\+url/a\    master_key     $MASTERKEY" /etc/shinken/modules/synchronizer-import.cfg
    fi


    print_ok_result_line
}


# Lot of dependencies can only be founded into the EPEL repos, so we add it
function install_epel {
    print_std_line "Installing EPEL repository"

    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        dorpminstall "dependencies/epel-release-6-8.noarch.rpm" "EPEL"
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        dorpminstall "dependencies/epel-release-latest-7.noarch.rpm" "EPEL"
    fi
    # check if the ssl is disabled, because new centos release bug here with too recent ssl certificates
    grep sslverify /etc/yum.repos.d/epel.repo >/dev/null
    if [ $? -ne 0 ]; then
        sed -i "s/\[epel\]/\[epel\]\nsslverify=false\n/" /etc/yum.repos.d/epel.repo
    fi
    print_std_line_no_end "EPEL is installed:"
    print_OK
    #printf "EPEL:  \033[92mOK\033[0m\n"
}


#note: the banner are generated with the command: figlet -f slant "Mongodb"
################################################################################################
#               ______                 __    _ __
#              / ____/________ _____  / /_  (_) /____
#             / / __/ ___/ __ `/ __ \/ __ \/ / __/ _ \
#            / /_/ / /  / /_/ / /_/ / / / / / /_/  __/
#            \____/_/   \__,_/ .___/_/ /_/_/\__/\___/
#                           /_/
################################################################################################

# Sub function for graphite, that will force the whisper lib update.
# at install AND at update, to be sure pre-2.03.03-U01 versions are uptodate with whisper read boost
function install_only_graphite_whisper {
    /bin/cp -rp dependencies/whisper/bin/*py /usr/bin/
    /bin/cp -rp dependencies/whisper/lib/whisper* /usr/lib/python2.${PYTHONMINOR}/site-packages/

    # We are giving mainly redhat6 packages, but centos are a bit different for some, so manage this case
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" ]]; then
       dorpminstall "dependencies/graphite/packages/redhat-centos/6.X/centos/*rpm" "Graphite dependencies for centos 6.X"
       dorpminstall "dependencies/graphite/packages/redhat-centos/6.X/common/*rpm" "Graphite dependencies for centos 6.X"

    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" ]]; then
       dorpminstall "dependencies/graphite/packages/redhat-centos/7.X/centos/*rpm" "Graphite dependencies for centos 7.X"
    fi
    if [[ "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        dorpminstall "dependencies/graphite/packages/redhat-centos/6.X/redhat/*rpm" "Graphite dependencies for centos 6.X"
        dorpminstall "dependencies/graphite/packages/redhat-centos/6.X/common/*rpm" "Graphite dependencies for centos 6.X"
    fi
    if [[ "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
       dorpminstall "dependencies/graphite/packages/redhat-centos/7.X/centos/*rpm" "Graphite dependencies for centos 7.X"
    fi

    # Make sure Carbon and Graphite data directories so we can give them the right permissions and owners
    mkdir -p /opt/graphite/storage/whisper 2> /dev/null
    mkdir -p /var/log/graphite 2> /dev/null
}


# We must set the grapite log & data rights to match Selinux.
# cd /var/log/graphite => ls -lZa
#    => drwxr-xr-x. apache apache system_u:object_r:httpd_log_t:s0 .
# cd /opt/graphite/storage/
#    => drwxr-xr-x. apache apache system_u:object_r:httpd_sys_rw_content_t:s0 whisper
function set_whisper_rights {
    # Whisper directories can be cp with a bad user name, so force good rights
    chmod -R 0755 /opt/graphite/storage/ /var/log/graphite 2>/dev/null
    chown -R apache:apache /opt/graphite/storage/ /var/log/graphite 2>/dev/null
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7"  ]]; then
        if [ $SELINUX_ENABLED -eq 0 ]; then
            # set /var/log/graphite as httpd_log_t so selinux will allow log rotation and co
            semanage fcontext -a -t httpd_log_t -s system_u "/var/log/graphite(/.*)?"
            # NOTE: restorecon with -F to force set the new user
            restorecon -RF /var/log/graphite
            semanage fcontext -a -t httpd_sys_rw_content_t -s system_u "/opt/graphite/storage(/.*)?"
            restorecon -RF /opt/graphite/storage
        fi
    fi
}


function update_graphite_webapp {
    /bin/cp -rp dependencies/graphite/opt/graphite/webapp/* /opt/graphite/webapp/

    # Also check that the mongodb.conf that link graphite & mongo is in place
    if [ ! -f /opt/graphite/conf/mongodb.conf ]; then
       cp -rp dependencies/graphite/opt/graphite/conf/mongodb.conf  /opt/graphite/conf/mongodb.conf
       dos2unix /opt/graphite/conf/mongodb.conf 2>/dev/null
    fi
}


# Only enable the graphite part, but don't enable it nor start it
function install_only_graphite {
    print_std_line_no_end "Installing Graphite (metrology):"

    # First of all, and to avoid some segfault from python, stop carbon
    /etc/init.d/carbon-cache stop >/dev/null 2>/dev/null

    # First whisper
    install_only_graphite_whisper


    # carbon need python-zope and python-zope-interface
    # Main deps here: "python-zope-interface mod_wsgi httpd"

    # Twisted: beware of the lib64 here. A nightmare for packages
    /bin/cp -rp dependencies/twisted/ /usr/lib64/python2.${PYTHONMINOR}/site-packages
    # Django: for the web part of graphite
    /bin/cp -rp dependencies/django/* /usr/lib64/python2.${PYTHONMINOR}/site-packages
    # Carbon too
    /bin/cp -rp dependencies/graphite/init.d/carbon-cache  /etc/init.d/

    # Let's declare it, but do not enable it at startup
    chkconfig --add carbon-cache
    chkconfig carbon-cache off

    # Now the real part, with graphite
    /bin/cp -rp dependencies/graphite/opt/graphite /opt

    # Apache part
    # do not enable it at startup
    __system_disable_daemon "httpd"
    __system_stop_daemon "httpd"


    # now link graphite conf into apache
    # but beware, auth are not the same for centos6 and 7 (http 2.2 and 2.4)
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        ln -sf /opt/graphite/conf/graphite.6.X.conf /etc/httpd/conf.d/graphite.conf
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        ln -sf /opt/graphite/conf/graphite.7.X.conf /etc/httpd/conf.d/graphite.conf
    fi

    # Create log path too for graphite, but AFTER installing apache
    mkdir /var/log/graphite 2>/dev/null
    chown apache:apache /var/log/graphite

    print_OK
}


function restart_gatherer {
    print_std_line_no_end "The shinken-gatherer must be restarted after update:"
    /etc/init.d/shinken-gatherer stop  >/dev/null  2>/dev/null
    # just be sure the gathere stop is done before start
    sleep 3
    /etc/init.d/shinken-gatherer start >/dev/null  2>/dev/null
    print_OK
}


function assume_graphite {
    print_std_line_no_end "Enabling Graphite (metrology):"
    
    # Let's enable it
    # carbon is with a init.d script, http can be a systemd daemon
    chkconfig carbon-cache on    
    __system_enable_daemon "httpd"

    
    # start carbon
    # First of all, and to avoid some segfault from python, stop carbon
    /etc/init.d/carbon-cache stop >/dev/null 2>/dev/null
    /etc/init.d/carbon-cache start >/dev/null 2>/dev/null
    
    # then apache
    __system_stop_daemon "httpd"
    __system_start_daemon "httpd"

    print_OK
}


# be sure the graphite logs will be setup in crontab
function assume_graphite_clean {
   /bin/cp -rp  dependencies/graphite/cron.daily/graphite-clean  /etc/cron.daily/graphite-clean
   chmod a+x    /etc/cron.daily/graphite-clean
   # be sure it is run at least once, so it's already ok after the install/update
   /etc/cron.daily/graphite-clean
}

# end of graphite

# We will check and create if need some directories for the users
# * /var/lib/shinken-user/libexec/ -> for plugins
#   -> must be referenced in /etc/shinken/resource.d/paths.cfg as USERPLUGINSDIR
# * /var/lib/shinken-user/external-resources/css -> for the user check css to inline in conf/webui
# * /etc/shinken-user/source-data/ -> for user source configuration import, that will match the skeleton propose in the
#   installer
# * /etc/shinken-user/configuration/ -> for user configuration part (default values, daemons, ...), that will match the skeleton propose in the
#   installer
function assume_user_directories {
    # Look at father directory
    export SHINKEN_USER_DIRECTORY="/var/lib/shinken-user/"
    if [ ! -d $SHINKEN_USER_DIRECTORY ]; then
        mkdir -p $SHINKEN_USER_DIRECTORY 2>/dev/null
        chown -R shinken:shinken $SHINKEN_USER_DIRECTORY
    fi

    # check if directory /var/lib/shinken-user/libexec/ exists
    export LIBEXEC_USER="/var/lib/shinken-user/libexec"
    if [ ! -d $LIBEXEC_USER ]; then
        mkdir -p $LIBEXEC_USER 2>/dev/null
        chown -R shinken:shinken $LIBEXEC_USER
    fi

    # and if declared into the paths file
    grep 'USERPLUGINSDIR' /etc/shinken/resource.d/paths.cfg > /dev/null
    if [ $? == 1 ]; then
        printf "\n\$USERPLUGINSDIR\$=$LIBEXEC_USER\n" >> /etc/shinken/resource.d/paths.cfg
    fi

    # check if directory /var/lib/shinken-user/external-resources/css exists
    export CHECK_PLUGIN_STYLES_CSS="/var/lib/shinken-user/external-resources/css"
    if [ ! -d $CHECK_PLUGIN_STYLES_CSS ]; then
        mkdir -p $CHECK_PLUGIN_STYLES_CSS 2>/dev/null
        chown -R shinken:shinken $CHECK_PLUGIN_STYLES_CSS
    fi

    # check if user configuration directory is existing
    export SHINKEN_USER_EXAMPLE_DIR="/etc/shinken-user-example"
    export SHINKEN_USER_DIR="/etc/shinken-user"
    export SHINKEN_USER_DIR_CONFIGURATION_DAEMONS="/etc/shinken-user/configuration/daemons/"

    # SOURCE DATA
    mkdir -p $SHINKEN_USER_DIR/source-data 2>/dev/null
    # List of directories to assume/create
    # Create if necessary skeletons directories from skeletons/source-data into the user source-data directory
    # Note: it get only the directories name here with a find
    export SKELETONS_USER_DIRS=`find skeletons/source-data -maxdepth 1 -type d | sed "s/skeletons\/source-data//" | sed "s/^\///" | grep -vE '^$'`
    for ii in $SKELETONS_USER_DIRS; do
        export P="$SHINKEN_USER_DIR/source-data/$ii"
        cp -TRnp skeletons/source-data/$ii $P
    done

    # CONFIGURATION
    mkdir -p $SHINKEN_USER_DIR/configuration 2>/dev/null
    mkdir -p $SHINKEN_USER_EXAMPLE_DIR 2>/dev/null
    # Create if necessary skeletons directories from skeletons/configuration into the user configuration directory
    # Note: it get only the directories name here with a find
    export SKELETONS_USER_DIRS=`find skeletons/configuration -maxdepth 1 -type d | sed "s/skeletons\/configuration//" | sed "s/^\///" | grep -vE '^$'`
    for ii in $SKELETONS_USER_DIRS; do
        export P="$SHINKEN_USER_DIR/configuration/$ii"
        cp -TRnp skeletons/configuration/$ii $P
    done

    # Arbiter overload configuration file
    if [ ! -f $SHINKEN_USER_DIR_CONFIGURATION_DAEMONS/arbiters/arbiter_cfg_overload.cfg ]; then
       cp -p skeletons/configuration/daemons/arbiters/arbiter_cfg_overload.cfg   $SHINKEN_USER_DIR_CONFIGURATION_DAEMONS/arbiters/arbiter_cfg_overload.cfg
    fi

    # Synchronizer overload configuration file
    if [ ! -f $SHINKEN_USER_DIR_CONFIGURATION_DAEMONS/synchronizers/synchronizer_cfg_overload.cfg ]; then
       cp -p skeletons/configuration/daemons/synchronizers/synchronizer_cfg_overload.cfg   $SHINKEN_USER_DIR_CONFIGURATION_DAEMONS/synchronizers/synchronizer_cfg_overload.cfg
    fi

    # Be sure the shinken user have all rights on this
    chown -R shinken:shinken $SHINKEN_USER_DIR
    chown -R shinken:shinken $SHINKEN_USER_EXAMPLE_DIR

    # Example directory.
    cp -rp skeletons/examples/* $SHINKEN_USER_EXAMPLE_DIR

    # After copying examples, make it obvious this directory shouldn't be written in
    chmod -R a-w $SHINKEN_USER_EXAMPLE_DIR

}

function assume_nmap_mac_prefixes_file {
    cp -f dependencies/nmap/nmap-mac-prefixes /usr/share/nmap/nmap-mac-prefixes
    chown -f shinken:shinken /usr/share/nmap/nmap-mac-prefixes
    chmod 664 /usr/share/nmap/nmap-mac-prefixes
}

function assume_discovery_rules_file {
    export DISCOVERY_RULES_USER="/etc/shinken-user/configuration/daemons/synchronizers/sources/discovery/discovery_rules.json"
    if [[ ! -f ${DISCOVERY_RULES_USER} ]]; then
        cp -f skeletons/configuration/daemons/synchronizers/sources/discovery/discovery_rules.json ${DISCOVERY_RULES_USER}
    fi
}

function assume_nmap_mac_prefixes_user_file {
    export NMAP_MAC_PREFIXES_USER="/etc/shinken-user/configuration/daemons/synchronizers/sources/discovery/nmap/nmap-mac-prefixes"
    if [[ ! -f ${NMAP_MAC_PREFIXES_USER} ]]; then
        cp -f skeletons/configuration/daemons/synchronizers/sources/discovery/nmap/nmap-mac-prefixes ${NMAP_MAC_PREFIXES_USER}
    fi
}

################################################################################################
#                __  ___                            ____
#               /  |/  /___  ____  ____  ____  ____/ / /_
#              / /|_/ / __ \/ __ \/ __ \/ __ \/ __  / __ \
#             / /  / / /_/ / / / / /_/ / /_/ / /_/ / /_/ /
#            /_/  /_/\____/_/ /_/\__, /\____/\__,_/_.___/
#                               /____/
################################################################################################

# Only during initial Shinken installation, if a Mongodb install is already present, display a warning saying that the Mongodb configuration will be handled by Shinken
function display_warning_if_mongo_already_installed {
    MONGODB_VERSION=$(mongod --version 2>/dev/null | grep 'db version')

    # Maybe there is no mongo, so we are OK
    if [ $? != 0 ]; then
        return
    fi

    printf "\n"
    printf "\n"
    print_red_header "IMPORTANT NOTICE : MONGODB INSTALLATION"

    printf "\033[31m\n"
    printf "An existing Mongodb installation has been detected. The Shinken installer installs and handles Mongodb.\n"
    printf "The current local Mongodb installation will be updated and overwritten by Shinken from now on.\n"
    printf "\033[0m\n\n"

    if [ -z "$disable_user_input" ]; then
        printf "\033[31mHit ENTER to continue\033[0m\n"
        read
    fi
}

# We only support some mongodb versions. So if at INSTALL we already have a mongo, it MUST
# be the valid one or no one at all
function assert_only_valid_mongo {
    # 2.6.9 => 2.6.0 and before
    # 3.0.15 => 2.6.1
    VALID_VERSIONS="2.6.9 3.0.15"
    MONGODB_VERSION=$(mongod --version 2>/dev/null | grep 'db version')

    # Maybe there is no mongo, so we are OK
    if [ $? != 0 ]; then
        return
    fi

    for valid_version in $VALID_VERSIONS; do
        echo "$MONGODB_VERSION"  | grep "$valid_version" > /dev/null
        # Valid version founded
        if [ $? == 0 ]; then
            return
        fi
    done
    echo "ERROR: Mongodb is already installed but your Mongodb version ($MONGODB_VERSION) is not supported for install/update"
    exit 2
}


function install_or_update_mongodb_rpm {
   # as the update/cluster case can be tricky, the install/update is done by another script
   # (and it's easier to ask for a admin to relaunch it)
   lib/update-mongodb.sh
   if [ $? != 0 ];then
      echo "ERROR"
      exit 2
   fi
}


function install_only_mongodb {
    # First look if already mongodb server installed
    rpm -qa | grep mongodb-org-server > /dev/null
    if [ $? -ne 0 ]; then
        install_or_update_mongodb_rpm
    fi

    # Maybe it was installed, but the old version
    rpm -qa | grep mongodb-org-server | grep '2.6.9'  > /dev/null
    if [ $? -eq 0 ]; then
        install_or_update_mongodb_rpm
    fi

    # Disable the mongodb startup at boot, only the assume can
    # enable it if need by daemons
    /etc/init.d/mongod stop >/dev/null 2>/dev/null
    chkconfig --add mongod
    chkconfig mongod off
# end of mongodb
}


function assume_mongodb {
    # First look if already mongodb server installed
    install_only_mongodb
    
    # Start the mongodb now it's installed
    /etc/init.d/mongod stop >/dev/null 2>/dev/null
    /etc/init.d/mongod start >/dev/null 2>/dev/null
    if [ $? -ne 0 ]; then
	      doexit "Mongod didn't want to start. Please check the log \"/var/log/mongodb/mongod.log\" and fix the problem before continue."
      fi
    
    # And enable it at startup
    chkconfig mongod on
    print_std_line_no_end "Checking if mongodb is enabled:"
    print_OK
# end of mongodb
}


# pymongo was in rpm before, and now we should give it as direct
# gridfs and bson are now included in pymongo, so be sure to remove old packages
function assume_pymongo {
    # be sure to clean previous pymongo from system
    rpm -e --nodeps python-pymongo        2>/dev/null >/dev/null
    rpm -e --nodeps python-pymongo-gridfs 2>/dev/null >/dev/null
    rpm -e --nodeps python-bson           2>/dev/null >/dev/null

    rm -fr /usr/lib64/python2.${PYTHONMINOR}/site-packages/gridfs 2>/dev/null >/dev/null
    rm -fr /usr/lib64/python2.${PYTHONMINOR}/site-packages/pymongo 2>/dev/null >/dev/null
    rm -fr /usr/lib64/python2.${PYTHONMINOR}/site-packages/bson 2>/dev/null >/dev/null

    mkdir /usr/lib64/python2.${PYTHONMINOR}/site-packages/pymongo 2>/dev/null >/dev/null
    mkdir /usr/lib64/python2.${PYTHONMINOR}/site-packages/bson 2>/dev/null >/dev/null
    mkdir /usr/lib64/python2.${PYTHONMINOR}/site-packages/gridfs 2>/dev/null >/dev/null

    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        /bin/cp -rfp dependencies/pymongo/packages/redhat-centos/6.X/*    -t /usr/lib64/python2.${PYTHONMINOR}/site-packages/pymongo 2>/dev/null >/dev/null
        /bin/cp -rfp dependencies/bson/packages/redhat-centos/6.X/*    /usr/lib64/python2.${PYTHONMINOR}/site-packages/bson 2>/dev/null >/dev/null
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        /bin/cp -rfp dependencies/pymongo/packages/redhat-centos/7.X/*    /usr/lib64/python2.${PYTHONMINOR}/site-packages/pymongo 2>/dev/null >/dev/null
        /bin/cp -rfp dependencies/bson/packages/redhat-centos/7.X/*    /usr/lib64/python2.${PYTHONMINOR}/site-packages/bson 2>/dev/null >/dev/null
    fi

    /bin/cp -rfp dependencies/gridfs     /usr/lib64/python2.${PYTHONMINOR}/site-packages/ 2>/dev/null >/dev/null
}

function assume_pyexcel {
    rm -fr /usr/lib64/python2.${PYTHONMINOR}/site-packages/pyexcel       2>/dev/null >/dev/null
    rm -fr /usr/lib64/python2.${PYTHONMINOR}/site-packages/pyexcel_io    2>/dev/null >/dev/null
    rm -fr /usr/lib64/python2.${PYTHONMINOR}/site-packages/pyexcel_xls    2>/dev/null >/dev/null
    rm -fr /usr/lib64/python2.${PYTHONMINOR}/site-packages/texttable     2>/dev/null >/dev/null
    rm -fr /usr/lib64/python2.${PYTHONMINOR}/site-packages/texttable.py  2>/dev/null >/dev/null

    mkdir /usr/lib64/python2.${PYTHONMINOR}/site-packages/pyexcel        2>/dev/null >/dev/null
    mkdir /usr/lib64/python2.${PYTHONMINOR}/site-packages/pyexcel_io     2>/dev/null >/dev/null
    mkdir /usr/lib64/python2.${PYTHONMINOR}/site-packages/pyexcel_xls     2>/dev/null >/dev/null
    mkdir /usr/lib64/python2.${PYTHONMINOR}/site-packages/lml     2>/dev/null >/dev/null

    /bin/cp -rfp dependencies/pyexcel/packages/redhat-centos/common/pyexcel/*               /usr/lib64/python2.${PYTHONMINOR}/site-packages/pyexcel    2>/dev/null >/dev/null
    /bin/cp -rfp dependencies/pyexcel/packages/redhat-centos/common/pyexcel_io/*            /usr/lib64/python2.${PYTHONMINOR}/site-packages/pyexcel_io 2>/dev/null >/dev/null
    /bin/cp -rfp dependencies/pyexcel/packages/redhat-centos/common/pyexcel_xls/*            /usr/lib64/python2.${PYTHONMINOR}/site-packages/pyexcel_xls 2>/dev/null >/dev/null
    /bin/cp -rfp dependencies/pyexcel/packages/redhat-centos/common/lml/*            /usr/lib64/python2.${PYTHONMINOR}/site-packages/lml 2>/dev/null >/dev/null

    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        dorpminstall "dependencies/pyexcel/packages/redhat-centos/6.X/*.rpm" "pyexcel"
        /bin/cp -rfp dependencies/pyexcel/packages/redhat-centos/6.X/texttable/*  /usr/lib/python2.${PYTHONMINOR}/site-packages/ 2>/dev/null >/dev/null
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        dorpminstall "dependencies/pyexcel/packages/redhat-centos/7.X/*.rpm" "pyexcel"
    fi
}


function save_intall_path {
    SCRIPT=$(readlink -f $0)
    SCRIPTPATH=`dirname $SCRIPT`
    SCRIPTPATH=`dirname $SCRIPTPATH`
    echo $SCRIPTPATH > /var/lib/shinken/install.path
    chmod a+r /var/lib/shinken/install.path
}

function remove_metric  {
    find /opt/graphite/storage/whisper/ | grep "$1" | xargs rm -f
}

################################################################################################
#                 _   __           _    ___
#                / | / /___ _____ | |  / (_)____
#               /  |/ / __ `/ __ `/ | / / / ___/
#              / /|  / /_/ / /_/ /| |/ / (__  )
#             /_/ |_/\__,_/\__, / |___/_/____/
#                         /____/
################################################################################################

NAGVIS_SHINKEN_PATH="/var/lib/shinken-nagvis/"
NAGVIS_SHINKEN_WEB_PATH="/shinken-core-map"

NAGVIS_USER_PATH="/opt/nagvis/"
NAGVIS_USER_WEB_PATH="/shinken-map"

function set_nagvis_configuration {
    INSTALL_PATH=$1

    sed -i "s/^;hosturl.*$/hosturl=\"http:\/\/$EXTERNAL_IP:7767\/detail-by-name\/\[host_name\]\"/" $INSTALL_PATH/etc/nagvis.ini.php
    sed -i "s/^;serviceurl.*$/serviceurl=\"http:\/\/$EXTERNAL_IP:7767\/detail-by-name\/\[host_name\]\/checks\/\[service_description\]\"/" $INSTALL_PATH/etc/nagvis.ini.php

    sed -i "s/^;urltarget.*$/urltarget=\"_blank\"/" $INSTALL_PATH/etc/nagvis.ini.php
    sed -i "s/live_1/shinken_livestatus/" $INSTALL_PATH/etc/nagvis.ini.php
    sed -i "s/;backend=\(.*\)/backend=\1/" $INSTALL_PATH/etc/nagvis.ini.php
    sed -i "s/;shinken_features=0/shinken_features=1/" $INSTALL_PATH/etc/nagvis.ini.php
    sed -i 's/;authmodule=.*/authmodule="CoreAuthModShinken"/' $INSTALL_PATH/etc/nagvis.ini.php
    sed -i 's/;authorisationmodule=.*/authorisationmodule="CoreAuthorisationModShinkenGroups"/' $INSTALL_PATH/etc/nagvis.ini.php
    sed -i 's/;logonmodule="LogonMixed"/logonmodule="LogonShinkenMixed"/' $INSTALL_PATH/etc/nagvis.ini.php
    sed -i 's/;eventsound=.*/eventsound=0/' $INSTALL_PATH/etc/nagvis.ini.php

    # Copy perms.db file if not already created by the user
    if [ ! -f $INSTALL_PATH/etc/perms.db ]; then
        cp dependencies/nagvis/installation/etc/perms.db $INSTALL_PATH/etc/perms.db
    fi
}

function set_nagvis_architecture_export_configuration {
    INSTALL_PATH=$1

    sed -i 's/authorisationmodule=.*/authorisationmodule="CoreAuthorisationModShinken"/' $INSTALL_PATH/etc/nagvis.ini.php
    sed -i 's/;shinken_auth_restrict_to_shinken_admin=.*/shinken_auth_restrict_to_shinken_admin="1"/' $INSTALL_PATH/etc/nagvis.ini.php

    # Change default architecture name in architecture-export configuration file
    architecture_name="Shinken-$(hostname)"
    sed -i "s/\(architecture_name\s*\).*/\1$architecture_name/" /etc/shinken/modules/architecture-export.cfg
}

function write_apache_configuration_for_nagvis {
    local file_name=$1
    local install_path=$2
    local web_path=$3
    local addon_enabled=$4

    file_ext=""
    if [ $addon_enabled -ne 0 ]; then
        file_ext=".disabled"
    fi

    # The user might want to change apache configuration for NagVis. Do not overwrite it if the file already exists
    if [ ! -e "/etc/httpd/conf.d/$file_name$file_ext" ]; then
        sed -e "s#@NAGVIS_PATH@#$install_path/share#g;s#@NAGVIS_WEB@#$web_path#g" etc/apache2-nagvis.conf-sample > /etc/httpd/conf.d/$file_name$file_ext
    fi
}

function install_nagvis_deps {
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" ]]; then
        dorpminstall "dependencies/nagvis/packages/redhat-centos/6.X/centos/*rpm" "NagVis dependencies for CentOS 6"
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" ]]; then
        dorpminstall "dependencies/nagvis/packages/redhat-centos/7.X/centos/*rpm" "NagVis dependencies for CentOS 7"
    fi
    if [[ "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
	    dorpminstall "dependencies/nagvis/packages/redhat-centos/6.X/centos/*rpm" "NagVis dependencies for RedHat 6"
    fi
    if [[ "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        dorpminstall "dependencies/nagvis/packages/redhat-centos/7.X/centos/*rpm" "NagVis dependencies for RedHat 7"
    fi
}

function perform_single_nagvis_install {
    local install_path=$1
    local web_path=$2
    local apache_file_name=$3
    local addon_enabled=$4
    local fail=0

    ./install.sh -n /var/lib/shinken -p $install_path -l "tcp:127.0.0.1:50000" -i mklivestatus -W $web_path -u apache -g apache -w /etc/httpd/conf.d -a n -o -q -r
    if [ ! $? -eq 0 ]; then
        echo "NagVis installation/update failed ($web_path) for an unknown reason"
        fail=1
    fi
    write_apache_configuration_for_nagvis $apache_file_name $install_path $web_path $addon_enabled

    # Add Apache SELinux rules for CentOS/RedHat 7
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        if [ $SELINUX_ENABLED -eq 0 ]; then
            chcon -t httpd_sys_rw_content_t $install_path -R
            setsebool -P httpd_can_network_connect 1
        fi
    fi

    # Custom shinken css is not copied at update because it is in "userfiles" directory. We copy it manually on each install/update
    cp -r share/userfiles/styles/* $install_path/share/userfiles/styles/

    return $fail
}

# Install NagVis and dependencies
function install_only_nagvis {
    local iscentral=$1
    local isarbiter=$2
    local isbroker=$3

    local install_failed=0
    shinken-addons-has nagvis > /dev/null 2>&1
    local nagvis_addon_enabled=$?
    shinken-addons-has nagvis-shinken-architecture > /dev/null 2>&1
    local nagvis_shinken_architecture_addon_enabled=$?

    print_header "Installing NagVis (cartography)"
    install_nagvis_deps

    # Contains uncompressed archive of nagvis .tar.gz installed package
    cd dependencies/nagvis/installation
        # Install NagVis used for automatic map generation by Shinken
        perform_single_nagvis_install $NAGVIS_SHINKEN_PATH $NAGVIS_SHINKEN_WEB_PATH "nagvis_etc.conf" $nagvis_shinken_architecture_addon_enabled
        install_failed=$?
        if [ ! -d /etc/shinken/external ]; then
            mkdir -p /etc/shinken/external
        fi
        if [ ! -e /etc/shinken/external/nagvis ]; then
            ln -s $NAGVIS_SHINKEN_PATH /etc/shinken/external/nagvis
        fi

        # Shinken user must be able to browse and write configuration to NagVis folders (only needed for the NagVis install where maps are generated)
        usermod -aG apache shinken
        usermod -aG apache root
        chmod 774 ${NAGVIS_SHINKEN_PATH}/etc/conf.d/
        chmod 774 ${NAGVIS_SHINKEN_PATH}/etc/maps/
        chmod 774 ${NAGVIS_SHINKEN_PATH}/share/userfiles/images/maps/
        chon shinken:apache ${NAGVIS_SHINKEN_PATH}/etc/conf.d/* > /dev/null 2>&1
        chown shinken:apache ${NAGVIS_SHINKEN_PATH}/etc/maps/* > /dev/null 2>&1
        chown shinken:apache ${NAGVIS_SHINKEN_PATH}/share/userfiles/images/maps/* > /dev/null 2>&1


        # Install NagVis usable by the user to create and handle his maps only on central nodes and broker nodes
        perform_single_nagvis_install $NAGVIS_USER_PATH $NAGVIS_USER_WEB_PATH "nagvis_opt.conf" $nagvis_addon_enabled
        install_failed=$install_failed || $?


        # Enable NagVis for automap generation only on central nodes and arbiter nodes
        if [ -z "$iscentral" ] && [ -z "$isarbiter" ]; then
            # If addon was manually enabled, do not disable it
            if [ $nagvis_shinken_architecture_addon_enabled -ne 0 ]; then
                shinken-addons-disable nagvis-shinken-architecture > /dev/null 2>&1
            fi
        fi
        # Enable NagVis for user only on central nodes and broker nodes
        if [ -z "$iscentral" ] && [ -z "$isbroker" ]; then
            # If addon was manually enabled, do not disable it
            if [ $nagvis_addon_enabled -ne 0 ]; then
                shinken-addons-disable nagvis > /dev/null 2>&1
            fi
        fi
    cd ../../..


    # NagVis requires more than 128M on some big Shinken architectures to display maps. PHP default 128M limit is modified to 256M
    cp /etc/php.ini /etc/php.ini.shinkenbackup
    sed -i 's/^memory_limit\s*=.*/memory_limit = 256M/' /etc/php.ini

    return $install_failed
}

# Install NagVis and restart Apache server
function assume_nagvis {
    install_only_nagvis "$1" "$2" "$3"
    local install_failed=$?

    __system_enable_daemon "httpd"
    __system_stop_daemon "httpd"
    __system_start_daemon "httpd"

    if [ $install_failed -eq 0 ]; then
        printf "NagVis installed:  \033[92mOK\033[0m\n"
        printf "\n\n"
    fi
    return $install_failed
}

################################################################################################
#             Server Analyzer
################################################################################################

# Be sure that we are always uptodate for the server analyzer code, erase mode
function install_server_analyzer {
    mkdir -p /var/lib/shinken/analyzer/ 2>/dev/null
    /bin/cp -rpf dependencies/analyzer/*   /var/lib/shinken/analyzer/

    # Centos6: need both winexe & the Cryptodome lib (it's a package for centos7)
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        /bin/cp -rfp dependencies/analyzer/redhat-centos/6.X/winexe    /var/lib/shinken/libexec/analyzer/
        /bin/cp -rfp dependencies/analyzer/redhat-centos/6.X/Cryptodome /usr/lib64/python2.6/site-packages/Cryptodome
    fi
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        /bin/cp -rfp dependencies/analyzer/redhat-centos/7.X/*    /var/lib/shinken/libexec/analyzer/
    fi

    chmod a+x /var/lib/shinken/libexec/analyzer/*
}


################################################################################################
#                _______
#               / ____(_)  _____  _____
#              / /_  / / |/_/ _ \/ ___/
#             / __/ / />  </  __(__  )
#            /_/   /_/_/|_|\___/____/
################################################################################################

# In the /etc/shinken/sources/*cfg we want the enabled parameter to be set to 1 if unset
function fix_02_03_02_r43_sources_enabled_string {
    for ii in `ls -1 /etc/shinken/sources/*cfg`; do
        grep 'enabled' $ii > /dev/null
        if [ $? == 1 ]; then
            sed -i "s/modules/enabled             1\n    modules/" $ii
        fi
    done
}


# In the /etc/shinken/sources/*cfg we want the enabled parameter to be set to 1 if unset
function fix_02_03_03_forcewebui_lang {
    grep 'lang' /etc/shinken/modules/webui.cfg > /dev/null
    if [ $? == 1 ]; then
	    sed -i "s/}/     lang fr\n}/" /etc/shinken/modules/webui.cfg
    fi
}


# In the /etc/shinken/sources/*cfg we want the enabled parameter to be set to 1 if unset
function fix_02_03_03_forcehttp_use_ssl_in_synchronizer {
    grep 'http_use_ssl' /etc/shinken/synchronizer.cfg > /dev/null
    if [ $? == 1 ]; then
	    printf "\nhttp_use_ssl=0\n" >> /etc/shinken/synchronizer.cfg
	    printf "http_ssl_cert=/etc/shinken/certs/server.cert\n" >> /etc/shinken/synchronizer.cfg
	    printf "http_ssl_key=/etc/shinken/certs/server.key\n" >> /etc/shinken/synchronizer.cfg
    fi

    # Remove the line http_ca_cert from synchronizer.cfg if present, no more used
    grep -v "http_ca_cert" /etc/shinken/synchronizer.cfg > /tmp/synchronizer.cfg.tmp
    mv -f /tmp/synchronizer.cfg.tmp /etc/shinken/synchronizer.cfg
    chown shinken:shinken /etc/shinken/synchronizer.cfg
}


# In the /etc/shinken/sources/*cfg we want the enabled parameter to be set to 1 if unset
function fix_02_03_03_forcehttp_use_ssl_in_webui {
    grep 'use_ssl' /etc/shinken/modules/webui.cfg > /dev/null
    if [ $? == 1 ]; then
	    sed -i "s/}/ use_ssl   0\n ssl_cert   \/etc\/shinken\/certs\/server.cert\n ssl_key   \/etc\/shinken\/certs\/server.key\n \n}/" /etc/shinken/modules/webui.cfg
    fi
}


# Look at all daemon cfg on the arbiter side only, and add enabled boolean.
# then look if we are in the arbiter server role, and if so set 0/1 to
# the enabled boolean to the local daemons. (done by shinken-daemons-* --force)
function __fix_02_03_03_r44_satellite_cfg_enabled_enabled_string_helper {
    DID_EDIT="0"
    for ii in `ls -1 /etc/shinken/$1s/*cfg`; do
        grep 'enabled' $ii > /dev/null
        if [ $? == 1 ]; then
            DID_EDIT="1"
            sed -i "s/}/   enabled             1\n}\n/" $ii
        fi
    done
    # do not force enable/disable if we didn't touch to a file, like in the next update
    if [ $DID_EDIT == "1" ]; then
        # will update the valid cfg file based on our state if need
        RES=$(shinken-daemons-has $1)
        if [ "$RES" != "NO" ]; then
            for daemon_id in $RES; do
                shinken-daemons-enable  --id="${daemon_id}" $1   --force > /dev/null
            done
        else
            shinken-daemons-disable $1   --force > /dev/null
        fi
    fi
}


function fix_02_03_03_r44_satellite_cfg_enabled_enabled_string {
   __fix_02_03_03_r44_satellite_cfg_enabled_enabled_string_helper "arbiter"
   __fix_02_03_03_r44_satellite_cfg_enabled_enabled_string_helper "scheduler"
   __fix_02_03_03_r44_satellite_cfg_enabled_enabled_string_helper "broker"
   __fix_02_03_03_r44_satellite_cfg_enabled_enabled_string_helper "poller"
   __fix_02_03_03_r44_satellite_cfg_enabled_enabled_string_helper "reactionner"
   __fix_02_03_03_r44_satellite_cfg_enabled_enabled_string_helper "receiver"
   __fix_02_03_03_r44_satellite_cfg_enabled_enabled_string_helper "synchronizer"
}


# In the /etc/shinken/sources/*cfg we want the enabled parameter to be set to 1 if unset
function fix_02_03_03_r51_remove_packs_from_synchronizer {
    grep -v 'packs' /etc/shinken/synchronizer.cfg > /etc/shinken/synchronizer.cfg.clean
    mv -f /etc/shinken/synchronizer.cfg.clean /etc/shinken/synchronizer.cfg
    chown shinken:shinken /etc/shinken/synchronizer.cfg
}


function fix_02_03_03_r76_cherrypyssl {
  # Fix cherrypy bug
  /bin/cp -f dependencies/ssl_pyopenssl.py  /usr/lib/python2.${PYTHONMINOR}/site-packages/cherrypy/wsgiserver/ssl_pyopenssl.py
}


# In the /etc/shinken/sources/*cfg we want the enabled parameter to be set to 1 if unset
function fix_02_03_03_u01_rc116_move_discovery_rules_in_discovery_rules_dir  {
    grep 'discovery_rules' /etc/shinken/synchronizer.cfg > /dev/null
    if [ $? == 1 ]; then
	    printf "\ncfg_dir=discovery_rules\n" >> /etc/shinken/synchronizer.cfg
    fi
}

# We empty contactgroup file in /etc/shinken/contactgroups.cfg this file wasn't load by arbiter and
# we want our object in shinken-core, is load.
function fix_02_03_03_u01_rc116_empty_contactgroup  {
    echo "" > /etc/shinken/contactgroups.cfg
}

# We change the check SLA so we update graphite metric
function fix_02_03_03_u01_rc128_rename_metric_sla  {
    find /opt/graphite/storage/whisper/ | grep "sla_archive_sla_by_sec" | xargs rm -f
    find /opt/graphite/storage/whisper/ | grep "sla_sla_update_sla_by_sec" | xargs rm -f
    for ii in `find /opt/graphite/storage/whisper/ | grep "sla_sla_update_sla_load"`; do /bin/mv -f $ii `echo $ii | sed "s/sla_sla_update_sla_load/sla_update_load_percent/"`; done
    for ii in `find /opt/graphite/storage/whisper/ | grep "sla_sla_archive_sla_load"`; do /bin/mv -f $ii `echo $ii | sed "s/sla_sla_archive_sla_load/sla_archive_load_percent/"`; done
}

# The metric archive_sla_load isn't give anymore so we clean file in graphite
function fix_02_05_00_RC001_remove_metric_archive_sla_load  {
    find /opt/graphite/storage/whisper/ | grep "archive_sla_load" | xargs rm -f
    find /opt/graphite/storage/whisper/ | grep "update_sla_load" | xargs rm -f
}


# Add remote_user_auth on uis of missing in the cfg files
function fix_02_03_03_u01_add_remote_auth_options {
    # Check into the webui
    grep 'remote_user_enable' /etc/shinken/modules/webui.cfg > /dev/null
    if [ $? == 1 ]; then
        sed -i "s/}/    remote_user_enable      0\n    remote_user_variable    X-Remote-User\n\n}/" /etc/shinken/modules/webui.cfg
    fi

    grep 'remote_user_case_sensitive' /etc/shinken/modules/webui.cfg > /dev/null
    if [ $? == 1 ]; then
        sed -i "s/}/    #remote_user_case_sensitive      1\n\n}/" /etc/shinken/modules/webui.cfg
    fi

    # Check into the synchronizer
    grep 'http_remote_user_enable' /etc/shinken/synchronizer.cfg > /dev/null
    if [ $? == 1 ]; then
        printf "\nhttp_remote_user_enable=0\nhttp_remote_user_variable=X-Remote-User\n\n" >> /etc/shinken/synchronizer.cfg
    fi

    grep 'http_remote_user_case_sensitive' /etc/shinken/synchronizer.cfg > /dev/null
    if [ $? == 1 ]; then
        printf "\n#http_remote_user_case_sensitive=1\n\n" >> /etc/shinken/synchronizer.cfg
    fi

}


# Destroy "__for_git" in "/source-data-cfg-sample" folder.
function fix_02_03_03_u01_destroy_file_for_git {
    # Clean the git hack for keep empty folder.
    find "/etc/shinken-user/source-data/source-data-cfg-sample/" -type f -name "__for_git" -exec rm -f {} \;
}

# Add remote_user_auth on uis of missing in the cfg files
function fix_02_03_03_u01_fix_default_reactionner_sub_realms {
    # If the default option si set on the reactionner-master, then change it, and nly in the file we give at install
    sed -i "s/manage_sub_realms   0/manage_sub_realms   1/" /etc/shinken/reactionners/reactionner-master.cfg
}

# Merge sources and modules configuration files
function fix_02_05_00_merge_sources_modules {
    python lib/merge-sources-and-modules.py --sources-dir /etc/shinken/sources --modules-dir /etc/shinken/modules
}

# Add new configuration key for auth-active-dir module
function fix_02_03_03_u01_auth_active_dir {
    list=`grep -rl "module_type\s\+ad_webui" "/etc/shinken/modules" 2>/dev/null`
    while read -r path; do
        occ=$(grep -c "^\s*\#\?\s\+mapping_file" "$path")
        if [ "$occ" = "0" ]; then
            sed -i '/\}/ i\    # File for additional configuration of the module behavior\n    # mapping_file    \/etc\/shinken-user\/configuration\/modules\/auth-active-directory\/mapping.json' "$path"
        fi
    done <<< "$list"
}

# Add daemon_thread_pool_size to daemon if missing in the ini files
function fix_02_03_03_u01_fix_missing_daemon_thread_pool_size {
    for ii in `ls -1 /etc/shinken/daemons/*ini`; do
         grep 'daemon_thread_pool_size' $ii > /dev/null
         if [ $? == 1 ]; then
             printf "\ndaemon_thread_pool_size=64\n" >> $ii
         fi
    done
}

# Add contact mail to ldap import module mapping configuration
function add_contact_mail {
	list=`grep -rh mapping_configuration_file $1 | awk '{ print $2 }'`
	while read -r path; do
	    occ=`grep -c contact\.mail "$path"`
	    if [ "$occ" = "0" ]; then
		    sed -i '/contact\.name/ i\    "contact.mail": "mail",' "$path"
	    fi
	done <<< "$list"
}
function fix_02_03_03_u01_fix_ldap_mail_mapping {
    add_contact_mail /etc/shinken/sources 2>/dev/null
    add_contact_mail /etc/shinken/modules 2>/dev/null
}

# Add a commentary for language export of the arbiter
function fix_02_05_00_fix_language_arbiter {
	occ=`grep -c "language=" /etc/shinken/shinken.cfg`
	if [ "$occ" = "0" ]; then
	    echo "" >> /etc/shinken/shinken.cfg
        echo "# If you need to set a specific language for deamons output (supported: fr; anything else is english), uncomment it" >> /etc/shinken/shinken.cfg
        echo "#language=fr" >> /etc/shinken/shinken.cfg
    fi
}

# Add warning_threshold_cpu_usage option in shinken.cfg
function fix_02_04_04_fix_add_warning_threshold_cpu_usage_option {
	occ=`grep -c "warning_threshold_cpu_usage=" /etc/shinken/shinken.cfg`
	if [ "$occ" = "0" ]; then
	    echo "" >> /etc/shinken/shinken.cfg
        echo "# How many seconds a command check (for hosts, clusters and checks) is allowed to consume cpu" >> /etc/shinken/shinken.cfg
        echo "# before raise a warning in check scheduler performance" >> /etc/shinken/shinken.cfg
        echo "# by default: 5" >> /etc/shinken/shinken.cfg
        echo "#warning_threshold_cpu_usage=5" >> /etc/shinken/shinken.cfg
    fi
}

# Remap active directory configuration to the new format, split in 3 files
function fix_02_05_00_remap_active_directory {
    python lib/remap-active-dir.py --sources-dir /etc/shinken/sources --sources-data /etc/shinken-user/source-data/
}

# Make sure the synchronizer config have only existing sources defined in its list
function fix_02_05_00_fix_synchronizer_sources_list {
    list_sources=`grep -rh '^\s*source_name' /etc/shinken/sources/*.cfg | awk '{ print $2 }'`
    list_res=""
    prefix=""
    if [ "${PREVVERSION#*2.03.03}" != "$PREVVERSION" ] && [ "${PREVVERSION#*U02}" == "$PREVVERSION" ]; then
        # Shove all sources in the list
    	while read -r src; do
	        list_res="$list_res$prefix$src"
	        prefix=","
	    done <<< "$list_sources"
    else
        list_sync=`grep -rh '^\s*sources' /etc/shinken/synchronizers/*.cfg | awk '{ print $2 }' | sed -n 1'p' | tr ',' '\n'`
    	while read -r src; do
	    # Only add a source if it is a substring of already defined sources
	    if [ "${list_sources#*$src}" != "$list_sources" ]; then
	        list_res="$list_res$prefix$src"
	        prefix=","
	    fi
	    done <<< "$list_sync"
    fi

	# Search for the sources declaration and replace with our filtered list
	sed -i "s/^\(\s*sources\s\+\).*/\1$list_res/" /etc/shinken/synchronizers/*.cfg
}


# In the 02.03.03-U01 we did change how the active directory import module is defined, with json files now
# so we must look if the user did edit the default configuration from before, and propose a update of it
# and merge it's parameters into specific file that will be load from the module
function fix_02_03_03_u01_fix_active_dir_import_format {
    # If there is still parameters active in the original active directory file, transfert them to the
    # destinations
    # First find all active-dir-import modules files
    export MODULES_FILES=`grep -l active-dir-import  /etc/shinken/modules/*cfg`
    #set -x
    for ii in $MODULES_FILES; do
        #echo "FILE $ii"
        # Get the filename
        export FNAME_FULL=`basename $ii`
        export FNAME=`echo $FNAME_FULL | cut -d"." -f1`
        # Now we have the module name as it's filename, we can create the desc in the /etc/shinken-user
        export DEST=/etc/shinken-user/source-data/source-data-migrated-$FNAME
        #echo "DEST $DEST"
        if [ -d $DEST ]; then
            # we already did this, skip it
            continue
        fi
        # Prepare the skeleton
        cp -rp skeletons/source-data/source-data-active-directory-sample $DEST
        printf "########################\nNOTE: This source-data is generated for the migration of the configuration of the source/module $FNAME.\nYou can edit the _configuration files to update your configuration that was migrated.\n" > $DEST/README_MIGRATION.txt
        # Ok go migrate the data
        export INPUT_FILE=$ii
        # First rules
        export OUTPUT_FILE=$DEST/_configuration/active-directory-rules.json
        export KEYS="hosts_match,contacts_match,hosts_tag_,contacts_filters,contacts_tag,hosts_tag"
        python lib/migrate-cfg-file-to-json.py -i $INPUT_FILE -o $OUTPUT_FILE -k $KEYS -p rules_configuration_file
        if [ $? -ne 0 ]; then
            echo "ERROR: cannot migrate your Active Directory Import module '$INPUT_FILE' for the rules configuration. Please take contact with the support."
        fi
        # Then connection
        export OUTPUT_FILE=$DEST/_configuration/active-directory-connection.json
        export KEYS="url,base,hosts_base,contacts_base,username,password"
        python lib/migrate-cfg-file-to-json.py -i $INPUT_FILE -o $OUTPUT_FILE -k $KEYS -p connection_configuration_file
        if [ $? -ne 0 ]; then
            echo "ERROR: cannot migrate your Active Directory Import module '$INPUT_FILE' for the connection configuration. Please take contact with the support."
        fi

        printf "* Migration of your active directory module \033[35m$FNAME\033[0m configuration into the new format: \033[92mOK\033[0m\n"
    done
}

function fix_02_05_00_service_check_timeout_renaming {
    grep service_check_timeout /etc/shinken/shinken.cfg >> /dev/null
    MUST_UPDATE_SERVICE_CHECK_TIMEOUT=$?

    if [ $MUST_UPDATE_SERVICE_CHECK_TIMEOUT -eq 0 ]; then
        sed -i s/service_check_timeout/check_running_timeout/ /etc/shinken/shinken.cfg
    fi
}

function launch_sanatize_fix {
     # SESUP-396 / SEF-2531: hard code the python2 call to bypass the customer security limitation on the /var mount (noexec)
     if ! python2 /var/lib/shinken/libexec/tools/sanatize-data.py --no-hint -r $1 -L "${SANATIZE_LOGFILE}" ; then
        SANATIZE_ERRORS=1
     fi
}

# update double link in prod and stagging see SEF-1184
function fix_02_03_03_u01_fix_double_link_for_all_items {
    # do not apply if not synchronizer
    IS_SYNCHRONIZER=$(shinken-daemons-has synchronizer 2>/dev/null)
    # Only look at the mongodb is we are the synchronizer (backuping configuration)
    if [[ "$IS_SYNCHRONIZER" == "NO" ]];
    then
        return
    fi
    has_make_double_link=$(mongo synchronizer  --quiet --eval "db.getCollection('synchronizer-info').find({'_id': 'database_info'}).forEach( function(x) { print(x.all_double_link); } );")

    if [ "W${has_make_double_link}" != "W1" ]; then
        launch_sanatize_fix fix_double_link
    fi
}

# cleanup changes and news persistent data
function fix_02_05_00_remove_persistent_changes_and_news_data {
    rm -f /var/lib/shinken/news.dat
    rm -f /var/lib/shinken/changes.dat
}
# update sync_keys in stagging see SEF-1199
# 3 round : see SEF-5690
function fix_02_03_03_u01_fix_double_sync_keys {
    # do not apply if not synchronizer
    IS_SYNCHRONIZER=$(shinken-daemons-has synchronizer 2>/dev/null)
    # Only look at the mongodb is we are the synchronizer (backuping configuration)
    if [[ "$IS_SYNCHRONIZER" == "NO" ]];
    then
        return
    fi
    has_fix_double_sync_keys=$(mongo synchronizer  --quiet --eval "db.getCollection('synchronizer-info').find({'_id': 'database_info'}).forEach( function(x) { print(x.fix_double_sync); } );")

    if [ "W${has_fix_double_sync_keys}" != "W3" ]; then
        launch_sanatize_fix fix_double_sync_keys
    fi
}

# cleanup changes and news persistent data
function fix_02_05_00_remove_persistent_changes_and_news_data {
    rm -f /var/lib/shinken/news.dat
    rm -f /var/lib/shinken/changes.dat
}

# update item id see SEF-1131
function fix_02_03_03_u01_fix_item_uuid {
    # do not apply if not synchronizer
    IS_SYNCHRONIZER=$(shinken-daemons-has synchronizer 2>/dev/null)
    # Only look at the mongodb is we are the synchronizer (backuping configuration)
    if [[ "$IS_SYNCHRONIZER" == "NO" ]];
    then
        return
    fi
    has_fix_item_uuid=$(mongo synchronizer  --quiet --eval "db.getCollection('synchronizer-info').find({'_id': 'database_info'}).forEach( function(x) { print(x.fix_fix_default_uuid_v2); } );")

    if [ "W${has_fix_item_uuid}" != "W1" ]; then
        launch_sanatize_fix fix_default_item_se_uuid
    fi
}

# Remove shinken-core user
function fix_02_03_03_u01_fix_remove_shinken_core {
    launch_sanatize_fix fix_remove_shinken_core
    # Remove it in cfg file as well
    lib/remove-cfg-element.py /etc/shinken/packs/shinken-core/contacts.cfg -k contact_name -v shinken-core
}
# Will clean deprecated check after poller update see SEF-1053 / SEF-1143
function fix_02_05_00_fix_remove_deprecated_check {
    launch_sanatize_fix fix_remove_deprecated_check
}

function fix_02_05_00_remove_undefined_aix_templates {
    launch_sanatize_fix fix_remove_undefined_aix_templates
}

function fix_02_05_00_force_custom_data_uppercase {
    launch_sanatize_fix fix_custom_data_uppercase
}

function fix_02_05_00_host_name_inheritance {
    launch_sanatize_fix fix_host_name_inheritance
}

function fix_02_05_00_fix_flapping_thresholds {
    launch_sanatize_fix fix_flapping_thresholds
}

function fix_02_04_fix_business_impact {
    launch_sanatize_fix fix_business_impact
}

function fix_02_05_00_fix_bp_rule_args {
    launch_sanatize_fix fix_bp_rule_args
}

function fix_02_05_00_fix_uuid_in_sla_info {
    launch_sanatize_fix fix_uuid_in_sla_info
}

function fix_02_05_00_remove_hostgroups_in_working_area {
    launch_sanatize_fix remove_hostgroups_in_working_area
}

function fix_02_06_00_migrate_links_name_into_id {
    launch_sanatize_fix migrate_links_name_into_id
}

function fix_02_06_00_encrypt_service_overrides_protected_fields {
    launch_sanatize_fix fix_service_overrides_encryption
}

function fix_02_06_00_enable_synchronizer_database_module {
    launch_sanatize_fix fix_enable_synchronizer_database_module
}

function fix_02_06_00_make_index_for_sla_module {
    launch_sanatize_fix make_index_for_sla_module
}

function fix_02_06_00_fix_safety_replacement_character_in_synchronizer_properties {
    launch_sanatize_fix fix_safety_replacement_character_in_synchronizer_properties
}


# Set if missing the default_element_properties properties (and custom ones) in the arbiter and synchronizer daemon
# for installation prior to 2.04
function fix_02_04_assume_default_element_properties_present_in_configuration {
    for ii in `echo /etc/shinken/shinken.cfg   /etc/shinken/synchronizer.cfg`; do
        grep 'default_element_properties' $ii > /dev/null
        if [ $? == 1 ]; then
            printf "\ncfg_dir=_default/default_element_properties\ncfg_dir=/etc/shinken-user/configuration/default_element_properties\n\n" >> $ii
        fi
    done
}



# Set if missing the shinken.cfg overload parameter file
# for installation prior to 2.04
function fix_02_04_assume_shinken_cfg_overload {
    grep 'arbiter_cfg_overload' /etc/shinken/shinken.cfg > /dev/null
    new_path="/etc/shinken-user/configuration/daemons/arbiters/arbiter_cfg_overload.cfg"
    if [ $? == 1 ]; then
        # NOTE: set at the end, it's important
        printf "\ncfg_file=${new_path}\n\n" >> /etc/shinken/shinken.cfg
    else
        # the path is in the cfg
        # get the file path
        path=$(grep 'arbiter_cfg_overload' /etc/shinken/shinken.cfg | cut -d= -f2)
        # if the file does not exist
        if [ ! -f "$path" ]; then
            sed -i -e "s/${path////\\/}/${new_path////\\/}/" /etc/shinken/shinken.cfg
        fi
    fi

    # and the global data one
    grep 'global-data' /etc/shinken/shinken.cfg > /dev/null
    if [ $? == 1 ]; then
        # NOTE: set at the end, it's important
        printf "\ncfg_dir=/etc/shinken-user/configuration/global-data\n\n" >> /etc/shinken/shinken.cfg
    fi


}


# Set if missing the synchronizer.cfg overload parameter file
# for installation prior to 2.04
function fix_02_04_assume_synchronizer_cfg_overload {
    new_path="/etc/shinken-user/configuration/daemons/synchronizers/synchronizer_cfg_overload.cfg"
    grep 'synchronizer_cfg_overload' /etc/shinken/synchronizer.cfg > /dev/null
    if [ $? == 1 ]; then
        # NOTE: set at the end, it's important
        printf "\ncfg_file=${new_path}\n\n" >> /etc/shinken/synchronizer.cfg
    else
        # the path is in the cfg
        # get the file path
        path=$(grep 'synchronizer_cfg_overload' /etc/shinken/synchronizer.cfg | cut -d= -f2)
        # if the file does not exist
        if [ ! -f "$path" ]; then
            sed -i -e "s/${path////\\/}/${new_path////\\/}/" /etc/shinken/synchronizer.cfg
        fi
    fi

    # and the global data one
    grep 'global-data' /etc/shinken/synchronizer.cfg > /dev/null
    if [ $? == 1 ]; then
        # NOTE: set at the end, it's important
        printf "\ncfg_dir=/etc/shinken-user/configuration/global-data\n\n" >> /etc/shinken/synchronizer.cfg
    fi
}



# add default acl for share in UI visualisation SEF-1813
function fix_02_04_add_acl_share_ui_visualisation {
    grep 'acl_share_private' /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg > /dev/null
    if [ $? == 1 ]; then
        # NOTE: set at the end, it's important
        printf "\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "# This properties is used to control the user access of the share in visualisation interface.\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "# Possible values are all / none / read, create, organize, modify, delete\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#[DEFAULT:contact] acl_share_private = all\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#[DEFAULT:contact] acl_share_group = all\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#[DEFAULT:contact] acl_share_everybody = all\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#==================================================================================\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
    fi
}

function fix_02_04_acl_try_check_contact {
    grep 'acl_try_check_on_poller' /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg > /dev/null
    if [ $? == 1 ]; then
        # NOTE: set at the end, it's important
        printf "\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#==================================================================================\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "# This property is used to authorize the user to try check on production platform.\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "# Possible values are [0 : disable] / [1 : enable]\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#[DEFAULT:contact] acl_try_check_on_poller = 1\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "# This property is used to authorize the user to try check on configuration platform.\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "# Possible values are [0 : disable] / [1 : enable]\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#[DEFAULT:contact] acl_try_check_on_synchronizer = 1\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#==================================================================================\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
    fi
}

function fix_02_04_acl_in_tab_history {
    grep 'acl_in_tab_history' /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg > /dev/null
    if [ $? == 1 ]; then
        # NOTE: set at the end, it's important
        printf "\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#==================================================================================\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "# This property is used to set what to display in history tab in visualization interface. \n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "# Possible values are history_sla / history / sla\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "[DEFAULT:contact] acl_in_tab_history = history\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "[DEFAULT:contact] acl_show_sla_range = 0\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#==================================================================================\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
    fi
}

function fix_02_08_add_default_sla_threshold {
    grep 'sla_warning_threshold' /etc/shinken-user/configuration/default_element_properties/default_host_properties.cfg > /dev/null
    if [ $? == 1 ]; then
        # NOTE: set at the end, it's important
        printf "#==================================================================================\n" >> /etc/shinken-user/configuration/default_element_properties/default_host_properties.cfg
        printf "\n" >> /etc/shinken-user/configuration/default_element_properties/default_contact_properties.cfg
        printf "#======== default sla threshold =========\n" >> /etc/shinken-user/configuration/default_element_properties/default_host_properties.cfg
        printf "#[DEFAULT:host] sla_warning_threshold  = 99.0\n" >> /etc/shinken-user/configuration/default_element_properties/default_host_properties.cfg
        printf "#[DEFAULT:host] sla_critical_threshold = 97.0\n" >> /etc/shinken-user/configuration/default_element_properties/default_host_properties.cfg
        printf "#==================================================================================\n" >> /etc/shinken-user/configuration/default_element_properties/default_host_properties.cfg
    fi
}

# In 2.04 we clean the old configuration file (shinken.cfg and local-import.cfg) from void/useless files
# clean:
#cfg_file=commands.cfg
#cfg_file=timeperiods.cfg
#cfg_file=escalations.cfg
#cfg_file=dependencies.cfg
#cfg_file=servicegroups.cfg
#cfg_file=hostgroups.cfg
#cfg_file=contacts.cfg
#cfg_file=templates.cfg
#cfg_file=contactgroups.cfg
#cfg_file=time_templates.cfg
# and the cfg_dir=services in the local-import.cfg
function fix_02_04_clean_shinken_cfg_from_useless_files {
    for ii in `echo commands  timeperiods  escalations  dependencies  servicegroups  hostgroups  contacts  templates  contactgroups time_templates`; do
        # Remove from shinken.cfg if present
        sed "s/^cfg_file=$ii.cfg$//" /etc/shinken/shinken.cfg -i
        # also in the local-import.cfg as it can be list inside
        sed "s/^cfg_file=$ii.cfg$//" /etc/shinken/local-import.cfg -i
        # remove it from synchronizer.cfg as well
        sed "s/^cfg_file=$ii.cfg$//" /etc/shinken/synchronizer.cfg -i
        # and also the old file
        rm -f  "/etc/shinken/$ii.cfg"
    done
    # cfg_dir=services in the local-import.cfg
    sed "s/^cfg_dir=services$//" /etc/shinken/local-import.cfg -i
    rm -fr /etc/shinken/services
}


# Set if missing the synchronizer.cfg the lang=en
# for installation prior to 2.04
function fix_02_04_assume_synchronizer_cfg_lang {
    grep 'lang' /etc/shinken/synchronizer.cfg > /dev/null
    if [ $? == 1 ]; then
        # NOTE: set at the end, it's important
        printf "\nlang=en\n\n" >> /etc/shinken/synchronizer.cfg
    fi

}



# The default reactionner file should be fix with 2 things:
# * there were 3 n in reactionnner_tags
# * missing passive   0   parameter
function fix_02_03_03_u01_default_reactionner_file_arguments {
    cfg_file=/etc/shinken/reactionners/reactionner-master.cfg
    grep 'passive' $cfg_file > /dev/null
    # if cannot find it, add it
    if [ $? == 1 ]; then
          sed -i "s/}/    passive      0\n\n}/" $cfg_file
    fi

    # Also fix the triple nnn
    grep 'reactionnner_tags' $cfg_file > /dev/null
    # if can find it, fix
    if [ $? == 0 ]; then
          sed -i "s/reactionnner_tags/reactionner_tags/" $cfg_file
    fi

}


function fix_02_03_03_fix_empty_value_from_synchronizer_data {
    launch_sanatize_fix fix_empty_value_from_synchronizer_data
}

# in 2.04 we added a configuration options for the webui
function fix_02_04_add_overload_paths_to_webui {
    grep '\[OVERLOAD_FROM\]' /etc/shinken/modules/webui.cfg > /dev/null
    if [ $? == 1 ]; then
	    sed -i "s/}/    [OVERLOAD_FROM]            \/etc\/shinken\/_default\/daemons\/brokers\/modules\/webui.cfg\n    [OVERLOAD_FROM]            \/etc\/shinken-user\/configuration\/daemons\/brokers\/modules\/webui\/webui_cfg_overload.cfg\n}/" /etc/shinken/modules/webui.cfg
    fi
}


# Set if missing the synchronizer.cfg the bind_addr=127.0.0.1
# for installation prior to 2.04.00
function fix_02_04_assume_synchronizer_cfg_bind_addr {
    grep 'bind_addr' /etc/shinken/synchronizer.cfg > /dev/null
    if [ $? == 1 ]; then
        # NOTE: set at the end, it's important
        printf "\nbind_addr=0.0.0.0\n\n" >> /etc/shinken/synchronizer.cfg
    fi
}


# Fix the missing property human_timestamp before version 02_03_03_u02 (aka 2.04.01)
function fix_02_04_01_missing_human_timestamp_log {
    files="/etc/shinken/shinken.cfg    /etc/shinken/synchronizer.cfg"
    for ii in $files; do
       grep 'human_timestamp_log' $ii > /dev/null
       # if cannot find it, add it
       if [ $? == 1 ]; then
           printf "\n\nhuman_timestamp_log=1\n\n" >> $ii
       fi
    done
}

function fix_02_04_00_remove_metric_check_scheduler_poller {
    remove_metric "checks_available_cpu_time_used"
    remove_metric "notifications_available_cpu_time_used"
    remove_metric "available_cpu_time_used"
}

# SEF-2092 :  SLA calculation can be configured based on status : no_data and unknow can now be count as OK
function fix_02_04_00_migrate_sla_option {
    grep 'exclude_unknown' /etc/shinken/modules/sla.cfg > /dev/null
    if [ $? == 0 ]; then
        sed -i '/Unknown does not count for SLA/,/Unknown counts as DOWN/{/Unknown counts as DOWN/ s/.*/    \#  == Unknown periods ==\n    #    - include:  Only status is considered. [default]\n    #    - exclude:  Unknown are not counted from SLA considered period\n    #    - ok:       Unknown are considered as UP periods/; t; d}' /etc/shinken/modules/sla.cfg
        sed -i '/\"Missing data\" and \"Shinken inactive\" periods do not count for SLA/,/\"Missing data\" and \"Shinken inactive\" periods count as DOWN periods \[default\]/{/\"Missing data\" and \"Shinken inactive\" periods count as DOWN periods \[default\]/ s/.*/    \#  == No_data periods \(\"Missing data\" and \"Shinken inactive\"\) ==\n    #    - include:  Only status is considered. [default]\n    #    - exclude:  No_data are not counted from SLA considered period\n    #    - ok:       No_data are considered as UP periods/; t; d}' /etc/shinken/modules/sla.cfg
        sed -i 's/exclude_unknown\s*0/unknown_period        include/g' /etc/shinken/modules/sla.cfg
        sed -i 's/exclude_unknown\s*1/unknown_period        exclude/g' /etc/shinken/modules/sla.cfg
        sed -i 's/exclude_no_data\s*0/no_data_period        include/g' /etc/shinken/modules/sla.cfg
        sed -i 's/exclude_no_data\s*1/no_data_period        exclude/g' /etc/shinken/modules/sla.cfg
    fi
}


# Prior to the 2.04.00-RC204 there was no remote_user_case_sensitive parameter in the
# livestatus module definition. If not present, add it to default value (sensitive mode, to keep
# the same behavior as before)
function fix_02_04_00_missing_remote_user_case_sensitive_in_livestatus_cfg {
    # Check into the webui
    grep 'remote_user_case_sensitive' /etc/shinken/modules/livestatus.cfg > /dev/null
    if [ $? == 1 ]; then
        sed -i "s/}/    remote_user_case_sensitive      1\n\n}/" /etc/shinken/modules/livestatus.cfg
    fi
}

function display_event_container_info {
    printf "\n"
    printf "\n"
    print_red_header "IMPORTANT NOTICE : RELATED TO EVENT CONTAINER"

    printf "\033[33m\n"
    printf "Shinken Entreprise V02.08 introduces the Event container which enables you to track the events occuring in your monitored platform in real time.\n"
    printf "In order to use this feature you need to enable the modules in the Brokers and WebUI configuration files.\n"
    printf "Search for 'Activate the event manager' or 'Activer le Bac à événement' in Shinken documentation for more details.\n"
    printf "\033[0m\n"
}


function display_protected_fields_warning {
    printf "\n"
    printf "\n"
    print_red_header "IMPORTANT NOTICE : RELATED TO ENCRYPTION (PROTECTED FIELDS)"

    printf "\033[33m\n"

    if [ -n "$activate_database_encryption" ] && [ $activate_database_encryption -eq 1 ]; then
        printf "Your protected data will now be encrypted, using a secret key generated during\n"
        printf "installation with the key name you provided : \033[35m${encryption_key_name}\033[33m.\n\n\033[0m"
        printf " ─ You now need to make a backup of your key using the following command command :\n\n"
        printf "      \033[35mshinken-protected-fields-keyfile-export\033[0m\n\n"
        printf " ─ NOTE :\033[31m If you lose your key, you won't be able to restore a backup and you will have to contact your support.\033[0m\n\n"
    else
        printf "The possibility to encrypt configuration database fields is now available in Shinken Entreprise.\n"
        printf "\033[33mTo enable encryption :\033[31m\033[0m\n\n"
        printf " ─ Run the command \033[35mshinken-protected-fields-encryption-enable\033[0m\n"
        printf " ─ NOTE :\033[31m A encryption key will be used. You will be asked to back it up. If you lose it, you won't be able to restore a backup and you will have to contact your support.\033[0m\n\n"
    fi
    printf "\033[0m\n"

    printf "\033[33mTo disable encryption :\033[0m\n\n"
    printf " ─ Your secret key must still be present to uncrypt your data.\n"
    printf " ─ Run the command \033[35mshinken-protected-fields-encryption-disable\033[0m\n"

    if [ -z "$disable_user_input" ]; then
        printf "\033[31mHit ENTER to continue\033[0m\n"
        read
    fi
}

function ask_for_key {
    if [ ! -f "$KEYFILE" ] && [ $activate_database_encryption -eq 1 ]; then
        printf "\n\033[33mWe will generate a key to encrypt protected fields in the Synchronizer.\033\0m\n"

        while [ -z "$encryption_key_name" ]; do
            read -p "Enter your key name: " encryption_key_name
        done
        printf "\033[0m"
        shinken-protected-fields-keyfile-generate -n "$encryption_key_name"
    fi
}

function fix_02_05_00_add_crypted_passwords_install {
    source /var/lib/shinken/libexec/libs/shinken_protected_fields.sh
    source ../../../sources/shinken-patched/libexec/libs/shinken_protected_fields.sh 2>/dev/null # For Pycharm

    KEYFILE=$(spf_get_keyfile_name_from_synchronizer_cfg)
    if [ $activate_database_encryption -eq 1 ] ;then
        if [ -n "$encryption_key_name" ] ; then
            if [ ! -d $(dirname "$KEYFILE") ]; then
                mkdir -p $(dirname "$KEYFILE")
            fi
            if [ ! -f "$KEYFILE" ]; then
                shinken-protected-fields-keyfile-generate -n "$encryption_key_name"
            fi
        else
            ask_for_key
        fi
        sed -i -e 's/protect_fields__activate_encryption=0/protect_fields__activate_encryption=1/' /etc/shinken/synchronizer.cfg
        sed -i -e 's/protect_fields__activate_database_encryption=0/protect_fields__activate_database_encryption=1/' /etc/shinken/synchronizer.cfg
        sed -i -e "s/protect_fields__activate_interface_encryption=[01]/protect_fields__activate_interface_encryption=1/" /etc/shinken/synchronizer.cfg
    fi
}

function fix_02_05_00_add_crypted_passwords_update {
    source /var/lib/shinken/libexec/libs/shinken_protected_fields.sh
    source ../../../sources/shinken-patched/libexec/libs/shinken_protected_fields.sh 2>/dev/null # For Pycharm


    if ! grep -q 'protect_fields__encryption_keyfile' "$SYNCHRONIZER_CFG" ; then
        sed -i "/sync_history_lifespan=1440/a \
        \
#==================================================================================\n\
#=============== Protected fields security ====================\n\
                                                       # Encryption for protected fields\n\
protect_fields__activate_encryption=${activate_database_encryption}\n\
                                                       # File containing the encryption key\n\
protect_fields__encryption_keyfile=$KEYFILE\n\
                                                       # List of words contained in protected fields names\n\
                                                       # Default values : PASSWORD,PASSPHRASE,PASSE,DOMAINUSER,MSSQLUSER,MYSQLUSER,ORACLE_USER,SSH_USER,LOGIN\n\
protect_fields__substrings_matching_fields=PASSWORD,PASSPHRASE,PASSE,DOMAINUSER,MSSQLUSER,MYSQLUSER,ORACLE_USER,SSH_USER,LOGIN\n\
#==================================================================================\n\n\
        " "$SYNCHRONIZER_CFG"

    else
        sed -i -e "s/protect_fields__activate_encryption=0/protect_fields__activate_encryption=${activate_database_encryption}/" $SYNCHRONIZER_CFG
        sed -i -e "s/protect_fields__activate_database_encryption=[01]/protect_fields__activate_database_encryption=${activate_database_encryption}/" $SYNCHRONIZER_CFG
        sed -i -e "s/protect_fields__activate_interface_encryption=[01]/protect_fields__activate_interface_encryption=${activate_database_encryption}/" $SYNCHRONIZER_CFG
    fi

    # Make sure synchronizer.cfg matches the new parameters

    launch_sanatize_fix fix_update_protected_fields_parameters

    if [ $activate_database_encryption -eq 1 ]; then
        if [ ! -d $(dirname "$KEYFILE") ]; then
            mkdir -p $(dirname "$KEYFILE")
        fi
        if [ -f "$KEYFILE" ]; then
            printf "$(show_important_info "Encryption key already exists with name $(show_data "$(spf_get_key_name_from_key_file "$KEYFILE")")")\n\n"
            printf "If you want to overwrite it with a another key you need to migrate the database\n"
        elif [ -n "$encryption_key_name" ] ; then
            shinken-protected-fields-keyfile-generate -n "$encryption_key_name"
        else
            ask_for_key
        fi
    fi
}


# For update before 2.5.0, we need to add the listener_rest source
function fix_02_05_00_missing_listener_rest_in_synchronizer_master_cfg {
    # Check into the webui
    grep -vE '\s*#' /etc/shinken/synchronizers/synchronizer-master.cfg | grep -E '\s*sources' | grep listener-rest > /dev/null
    if [ $? == 1 ]; then
        sed -i '/\s*sources/s/$/,listener-rest/' /etc/shinken/synchronizers/synchronizer-master.cfg
    fi
}


# Set if missing the synchronizer.cfg entries for listeners & analyzers
function fix_02_05_00_missing_listeners_analyzers_in_synchronizer_cfg {
    grep 'listeners' /etc/shinken/synchronizer.cfg > /dev/null
    if [ $? == 1 ]; then
        printf "\ncfg_dir=listeners\n\n" >> /etc/shinken/synchronizer.cfg
    fi

    grep 'analyzers' /etc/shinken/synchronizer.cfg > /dev/null
    if [ $? == 1 ]; then
        printf "\ncfg_dir=analyzers\n\n" >> /etc/shinken/synchronizer.cfg
    fi

}

# Add nb_stored_days parameter to sla existing module.
function fix_02_05_00_add_nb_stored_days_parameter {
    grep 'nb_stored_days' /etc/shinken/modules/sla.cfg > /dev/null
    if [ $? == 1 ]; then
        sed -i "s/}/\n    # Duration in day to keep SLA info,\n    # Default value is -1. It mean SLA are keep forever, in this case to mongo database will grow endlessly.\n    # Minimal value is 7 day\n    #nb_stored_days   -1\n\n}\n/" /etc/shinken/modules/sla.cfg
    fi
}

function fix_02_05_00_move_overload_configuration_files {
    base_dir=/etc/shinken-user/configuration
    for daemon in synchronizer arbiter poller scheduler broker reactionner receiver ; do
        daemon_overload_dir=${base_dir}/daemons/${daemon}s
        overload_filename=${daemon}_cfg_overload.cfg
        if [ ! -d $daemon_overload_dir ]; then
            mkdir -p $daemon_overload_dir
            chown shinken:shinken $daemon_overload_dir
            src_config_file=${base_dir}/${overload_filename}
            dst_config_file=${daemon_overload_dir}/${overload_filename}
            if [ -f $src_config_file ]; then
                mv ${src_config_file} ${dst_config_file}
                printf "\033[31mThe file ${src_config_file} has been moved to ${dst_config_file}\033[0m\n"
            fi

            # The weird variable expansions in sed commands below are escaping slashes in the paths as detailed here :
            # http://www.gnu.org/software/bash/manual/html_node/Shell-Parameter-Expansion.html#Shell-Parameter-Expansion
            if [ $daemon == synchronizer ]; then
                sed -i -e "s/${src_config_file////\\/}/${dst_config_file////\\/}/" /etc/shinken/${daemon}*.cfg
            elif [ $daemon == arbiter ]; then
                sed -i -e "s/${src_config_file////\\/}/${dst_config_file////\\/}/" /etc/shinken/shinken.cfg
            else
                sed -i -e "s/${src_config_file////\\/}/${dst_config_file////\\/}/" /etc/shinken/daemons/${daemon}*.ini
            fi

        fi
    done
}


function fix_02_05_00_remove_unused_synchronizer_plugins {
   base_dir=/usr/lib/python2.${PYTHONMINOR}/site-packages/shinken/synchronizer/plugins
    for plugin in discovery newhosts packs ; do
        plugin_dir=${base_dir}/${plugin}
        if [ -d "$plugin_dir" ]; then
            rm -rf $plugin_dir
        fi
    done
}


function fix_02_05_00_add_cfg_dir_synchronizers_in_shinken_cfg {
    count=`grep -c "cfg_dir=synchronizers" /etc/shinken/shinken.cfg`
    if [ "$count" == "0" ]; then
        sed -i 's/cfg_dir=realms/cfg_dir=realms\ncfg_dir=synchronizers/' /etc/shinken/shinken.cfg
    fi
}


function fix_02_05_00_add_map_base_url_param_to_architecture_export_cfg {
    count=$(grep -c "map_base_url" /etc/shinken/modules/architecture-export.cfg)
    if [ "$count" == "0" ]; then
        sed -i 's/\(architecture_name.*\)/\1\n\n    # Base of URL used to display links in the Visualization UI \n    # Defaults to Arbiter URL if empty \n    # map_base_url  http:\/\/example.com\//' /etc/shinken/modules/architecture-export.cfg
    fi
}

function get_current_version {
    local current_version
    local need_version_from_context

    if which shinken-context-manage >/dev/null 2>/dev/null ; then
        current_version=$(shinken-context-manage --get-shinken-current-version  2>&1)
        # shellcheck disable=SC2001
        current_version=$(sed 's/^[[:alpha:]]\+//i' <<< "$current_version")  # Removing the leading letters before the version number
        if [[ "$current_version" == Usage* ]]; then
            need_version_from_context="true"
        fi
    else
        need_version_from_context="true"
    fi


    if [[ "$need_version_from_context" == "true" ]]; then
        # We are upgrading from an older version when --get-shinken-current-version was not available
        # IN that case, context.json is known to be a json file with the current version set as expected
        current_version=$(python -mjson.tool < /var/lib/shinken/context.json | grep current_version | sed 's/.*:\s*"\(.*\)".*/\1/')
    fi
    echo "$current_version"
}

function assert_current_version_not_too_old {
    current_version=$(get_current_version)
    current_major=${current_version%%.*}  # Removing all after the first dot
    current_minor=$(cut -d. -f2 <<< "${current_version}")

    # shellcheck disable=SC2001
    new_version=$(sed 's/^[[:alpha:]]\+//i' <<< "$VERSION")  # Removing the leading letters before the version number
    new_major=${new_version%%.*}
    new_minor=$(cut -d. -f2 <<< "$new_version")

    if [[ "$new_major" -eq "$current_major" ]] ; then
        if [ $(( 10#$current_minor + 1 )) -lt "$new_minor" ] ; then
	         doexit "You can only update from Shinken v$current_major.$((10#$new_minor - 1)).xx. Your current version is $current_version"
	      fi
	  else
         doexit "You can only update from Shinken v$current_major.$((10#$new_minor - 1)).xx. Your current version is $current_version"
    fi
}

function fix_02_06_00_create_default_synchronizer_backup_directory {
	DIR_PATH="/var/shinken-user/backup/synchronizer-module-database-backup"
	if [ ! -d $DIR_PATH ]; then
		display_synchronizer_backup_warning
        mkdir -p $DIR_PATH 2>/dev/null
        chown -R shinken:shinken $DIR_PATH 2>/dev/null
    fi
}

function fix_02_06_00_all_move_overload_files {
    launch_sanatize_fix fix_all_move_overload_files
}

function display_synchronizer_backup_warning {
    printf "\n"
    printf "\n"
    print_red_header "IMPORTANT NOTICE : ACTIVATING THE synchronizer-module-database-backup"

    printf "\n"
    printf "\033[31mWarning, backups will be made in the folder :\033[0m /var/shinken-user/backup/synchronizer-module-database-backup.\n"
    printf "\033[33mPlease check that shinken can access this folder and that it has about 2GB of free space\033[0m\n"
    printf "\n"

    if [ -z "$disable_user_input" ]; then
        printf "\033[31mHit ENTER to continue\033[0m\n"
        read
    fi
}

function fix_02_06_00_rename_shinken_protected_fields_properties_manage {
	BIN_PATH="/usr/sbin"
	if [ -e $BIN_PATH/shinken-protected-fields-properties-manage ]; then
		rm -f $BIN_PATH/shinken-protected-fields-properties-manage
	fi
}

function fix_02_07_00_add_news_elements_to_architecture_export {
        launch_sanatize_fix add_ssl_architecture_export
}

function fix_02_07_00_add_news_elements_to_listener_shinken {
    # do not apply if not synchronizer
    IS_SYNCHRONIZER=$(shinken-daemons-has synchronizer 2>/dev/null)
    # Only look at the mongodb is we are the synchronizer (backuping configuration)
    if [[ "$IS_SYNCHRONIZER" == "NO" ]];
    then
        return
    fi
    launch_sanatize_fix add_ssl_shinken_listener
}

function fix_02_07_00_normalize_se_uuids {
    # do not apply if not synchronizer
    IS_SYNCHRONIZER=$(shinken-daemons-has synchronizer 2>/dev/null)
    # Only look at the mongodb is we are the synchronizer (backuping configuration)
    if [[ "$IS_SYNCHRONIZER" == "NO" ]];
    then
        return
    fi
    launch_sanatize_fix normalize_se_uuids
}

function fix_02_07_00_update_notes_multi_url {
    # do not apply if not synchronizer
    IS_SYNCHRONIZER=$(shinken-daemons-has synchronizer 2>/dev/null)
    # Only look at the mongodb is we are the synchronizer (backuping configuration)
    if [[ "$IS_SYNCHRONIZER" == "NO" ]];
    then
        return
    fi
    launch_sanatize_fix update_notes_multi_url_syntax
}

function fix_02_07_00_add_rules_path_discovery {
    launch_sanatize_fix add_rules_path_discovery
}

function fix_02_07_00_add_nmap_mac_prefixes_path_discovery {
    launch_sanatize_fix add_nmap_mac_prefixes_path_discovery
}

function fix_02_07_03_remove_pidfile_comment_in_mongodb_config_files {
    launch_sanatize_fix remove_pidfile_comment_in_mongodb_config_files
}

function fix_02_07_04_add_internal_option_to_retention_mongo_cfg {
    launch_sanatize_fix add_internal_option_to_retention_mongo_cfg
}

# in 02.05 carbon cache was set as root & perf limited, so we remove theses limitations in carbon.conf
function fix_02_07_04_fix_carbon_cache_parameters {
    launch_sanatize_fix fix_carbon_cache_parameters
}

function fix_02_07_05_fix_presence_protection_on_items {
    launch_sanatize_fix remove_presence_protection
}

# We want to be sure that the samba packages are uptodate after the 02.07.06 update
function fix_02_07_06_fix_force_samba_update_for_centos7 {
    if [[ "X$LINUX_VERSION" == "XCENTOS-7" ]]; then
        RPM_DIR=dependencies/enterprise/packages/redhat-centos/7.X/centos
        rpm -Uvh $RPM_DIR/samba-*.rpm $RPM_DIR/libldb-*rpm $RPM_DIR/pytalloc-*.rpm $RPM_DIR/libsmbclient-*.rpm $RPM_DIR/libtalloc-*.rpm $RPM_DIR/libtevent-*.rpm $RPM_DIR/libwbclient-*.rpm $RPM_DIR/libtdb-*.rpm >/dev/null 2>/dev/null
    fi
}

function fix_02_08_00_add_sla_output_store_and_worker_options {
    launch_sanatize_fix add_sla_output_store_and_worker_options
}

function fix_02_08_01_add_map_realm_layout_to_architecture_export_module {
    launch_sanatize_fix add_map_realm_layout_to_architecture_export_module
}

function fix_02_08_01_add_retry_options_to_synchronizer_import_module {
    launch_sanatize_fix add_retry_options_to_synchronizer_import
}

function fix_02_08_01_clean_old_nagvis_shinken_tmp_files {
    launch_sanatize_fix clean_old_nagvis_shinken_tmp_files
}

function overwrite_shinken_packs {
    # At first launch we need to backup the folder in case of user already modified theses packs. Check also if folder /etc/shinken/packs/shinken exists :
    # => this function is also launched at install

    if [[ ! -e /var/lib/shinken/backup/shinken-packs ]] && [[ -e /etc/shinken/packs/shinken ]]; then
        mkdir -p /var/lib/shinken/backup/shinken-packs
        mv /etc/shinken/packs/shinken /var/lib/shinken/backup/shinken-packs
        mv /etc/shinken/packs/shinken-core /var/lib/shinken/backup/shinken-packs
    fi

    if [[ -e /etc/shinken/packs/shinken ]]; then
        rm -fr /etc/shinken/packs/shinken
    fi

    if [[ -e /etc/shinken/packs/shinken-core ]]; then
        rm -fr /etc/shinken/packs/shinken-core
    fi

    cp -rpf ./shinken-pack/* /etc/shinken/packs/
    chown -R shinken:shinken /etc/shinken/packs/
}