#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import re
from ..def_items import ITEM_TYPE, STOP_INHERITANCE_VALUES


# Tokens for all parsable rules.
# Essentially a data container, which associates a regexp match with a type, and can provides accessors to parsed values.
class BpRuleToken(object):
    def __init__(self, tok_type, match_object=None, start=None, end=None, val=None):
        self.type = tok_type
        if match_object:
            self.start = match_object.start()
            self.end = match_object.end()
            self.val = match_object.group(0).strip().replace('"', '')
    
    
    @staticmethod
    def sorted_insert(tok, token_list):
        i = 0
        for prev_tok in token_list:
            prev_start = prev_tok.start
            if prev_start < tok.start:
                i += 1
            else:
                break
        token_list.insert(i, tok)
    
    
    def get_node_flags(self):
        if self.type != 'node':
            raise SyntaxError("Internal BP rule evaluation error: get_node_flags for non-flags")
        if re.match(re.compile(r'^[gtlr]+:.+'), self.val):
            return self.val.split(':')[0]
        return ''
    
    
    def get_node_expr(self):
        if self.type != 'node':
            raise SyntaxError("Internal BP rule evaluation error: get_node_expr for non-node")
        if re.match(re.compile(r'^[gtlr]+:.+'), self.val):
            # Remove flags
            expr = ':'.join(self.val.split(':')[1:])
            if 'r' in self.get_node_flags() and expr[0] == expr[-1] == '/':
                # Remove regexp delimiters
                expr = expr[1:-1]
            return expr.strip()
        return self.val.strip()


class BpRuleParser():
    HOST_FLAGS = 'grt'
    # Not all of forbidden chars are used, because a non-escaped regex might use things like ? + *
    FORBIDDEN_CHARS_CLASS = '~!&|<>,()\[\]"~'
    STATE_RE = r'\s*(ok|warning|critical|unknown)\s*'
    TYPE_BY_FLAGS = {
        't'      : [ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERTPLS],
        'g'      : [ITEM_TYPE.HOSTGROUPS],
        'default': [ITEM_TYPE.HOSTS, ITEM_TYPE.CLUSTERS]
    }
    
    
    @staticmethod
    def compute_internal_bp_rule(bp_rule):
        bp_rule_tokenise = BpRuleParser._parse_bp_rule(bp_rule)
        _previous = ''
        _internal_bp_rule = []
        for token in bp_rule_tokenise:
            data = {'type': token.type}
            if token.type == 'node':
                data['flag'] = token.get_node_flags()
                data['content'] = token.get_node_expr()
                # Skip the regex and joker !
                if not BpRuleParser._is_regex_or_joker(data):
                    if _previous == 'comma':
                        data['item_type'] = 'services'
                    else:
                        data['item_type'] = BpRuleParser._get_type_by_flag(data['flag'])
                
                data['to_string'] = "%s:" % data['flag'] if data['flag'] else ""
                if 'r' in data['flag']:
                    data['to_string'] += "/%s/"
                elif data['content'] == "*" or data['content'] in STOP_INHERITANCE_VALUES:
                    data['to_string'] += '%s'
                else:
                    data['to_string'] += '"%s"'
            
            elif token.type in ['part_op', 'op', 'state_rule_of']:
                if _previous == 'state_rule' and token.type != 'state_rule_of':
                    data['to_string'] = "%s"
                else:
                    data['to_string'] = " %s "
                data['content'] = token.val
            elif token.type == 'xof':
                data['to_string'] = "%s "
                data['content'] = token.val
            elif token.type == "unknown_token":
                data['to_string'] = '%s'
                data['content'] = token.val
                data['has_error'] = True
                data['token'] = token
            else:
                data['to_string'] = '%s'
                data['content'] = token.val
            
            _previous = token.type
            _internal_bp_rule.append(data)
        
        return _internal_bp_rule
    
    
    @staticmethod
    def get_unparse_bp_rule(bp_rule_parsed):
        return ''.join([token['to_string'] % token['content'] for token in bp_rule_parsed])
    
    
    @staticmethod
    def _parse_bp_rule(pattern):
        # Order matters. First tokens take priority over lower ones in case a pattern matches several.
        tokens = [
            (re.compile(r'(default|-?\d+%%?%(states)s)\s*->\s*%(states)s' % {'states': BpRuleParser.STATE_RE}, re.IGNORECASE), 'state_rule'),
            (re.compile(r'-?\d+%?(, *-?\d*%?, *-?\d*%?)? *of:'), 'xof'),
            (re.compile(r'of\s*:'), 'state_rule_of'),
            (re.compile(r'[%(flags)s]+:/.*?[^\\]/' % {'flags': BpRuleParser.HOST_FLAGS}), 'node'),
            (re.compile(r'([%(flags)s]+:\s*)?"[^"~]+"' % {'flags': BpRuleParser.HOST_FLAGS}), 'node'),
            (re.compile(r'(<or>|<and>|<and not>)'), 'part_op'),
            (re.compile(r'([%(flags)s]+:\s*)?[^\s%(forbidden)s][^%(forbidden)s]*' % {'flags': BpRuleParser.HOST_FLAGS, 'forbidden': BpRuleParser.FORBIDDEN_CHARS_CLASS}), 'node'),
            (re.compile(r','), 'comma'),
            (re.compile(r'!'), 'not'),
            (re.compile(r'[&|]'), 'op'),
            (re.compile(r'[()\[\]]'), 'sep'),
            # When we are done tokenizing, only spaces and filling chars must be left, else we need to raise an error
            (re.compile(r'[^~\s]+'), 'unknown_token')
        ]
        found = []
        for token_re, token_type in tokens:
            while True:
                match = re.search(token_re, pattern)
                if match:
                    bp_rule_token = BpRuleToken(token_type, match)
                    
                    BpRuleToken.sorted_insert(bp_rule_token, found)
                    # The '~' is a neutral character that can't match any pattern.
                    # The idea is to fill what we previously matched to avoid pattern overlap and keep track of token order.
                    pattern = pattern[:match.start()] + ('~' * (match.end() - match.start())) + pattern[match.end():]
                else:
                    break
        return found
    
    
    # Utils
    @staticmethod
    def _is_regex_or_joker(token):
        return 'r' in token['flag'] or token['content'] == "*"
    
    
    @staticmethod
    def _is_linkable_token(token):
        if not token['type'] == 'node':
            return False
        if BpRuleParser._is_regex_or_joker(token):
            return False
        if token['item_type'] == 'services':
            return False
        if token['item_type'] == 'unknown_token':
            return False
        return True
    
    
    @staticmethod
    def _get_type_by_flag(flag):
        return BpRuleParser.TYPE_BY_FLAGS[flag or 'default']
