#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

NAME_NOT_IN_NODE = 'NAME_NOT_IN_NODE'
NAME_NOT_IN_TEMPLATE = 'NAME_NOT_IN_TEMPLATE'


class OPERAND(object):
    TEMPLATE_OP = 'template'
    AND_OP = '&'
    OR_OP = '|'


def _is_complex(pattern):
    for m in '()+&|,!':
        if m in pattern:
            return True
    return False


def _make_node(operand=None, sons=None, not_value=False, content=None):
    return {
        'operand'  : operand,
        'sons'     : sons or [],
        'not_value': not_value,
        'content'  : content
    }


def do_match(node, templates):
    match = False
    r_sons = [do_match(s, templates) for s in node.get('sons', [])]
    
    if node['operand'] == OPERAND.TEMPLATE_OP:
        content_id = node['content'].get('_id', None)
        exists = node['content']['exists']
        for template in templates:
            if exists and content_id == template.get('_id', None) and template.get('enabled', True):
                match = True
            if not exists and template.get('can_match_by_name', False) and node['content'].get('name', NAME_NOT_IN_NODE) == template.get('name', NAME_NOT_IN_TEMPLATE):
                match = True
    elif node['operand'] == OPERAND.OR_OP:
        match = True in r_sons
    elif node['operand'] == OPERAND.AND_OP:
        match = False not in r_sons
    
    if node.get('not_value', False):
        match = not match
    return match


def _do_parse_complex_exp(pattern):
    errors = []
    node = _make_node()
    pattern = pattern.strip()
    if pattern.startswith('!'):
        node['not_value'] = True
        pattern = pattern[1:]
    
    complex_node = _is_complex(pattern)
    
    if not complex_node:
        # If it's a not value, tag the node and find the name without this ! operator
        node['operand'] = OPERAND.TEMPLATE_OP
        node['content'] = pattern
        return node, errors
    
    in_part = False
    tmp = ''
    stacked_par = 0
    start_of_expression = True
    for _char in pattern:
        if _char in ",|&+":
            if start_of_expression:
                errors.append("validator.servicehosttpls_complex_expression_two_successive_operators")
            start_of_expression = True
            if in_part:
                tmp += _char
            else:
                # Oh we got a real cut in an expression, if so, cut it
                tmp = tmp.strip()
                if _char in ",|":
                    node['operand'] = OPERAND.OR_OP
                else:
                    node['operand'] = OPERAND.AND_OP
                if tmp != '':
                    son_node, son_errors = _do_parse_complex_exp(tmp)
                    son_node['not_value'] = node['not_value']
                    node['not_value'] = False
                    errors.extend(son_errors)
                    node['sons'].append(son_node)
                tmp = ''
        elif _char == '(':
            start_of_expression = True
            stacked_par += 1
            in_part = True
            tmp = tmp.strip()
            # Maybe we just start a part, but we got some things in tmp that should not be good in fact !
            if stacked_par == 1 and tmp != '':
                errors.append("validator.servicehosttpls_complex_expression_missing_operator")
            
            # If we are already in a part, add this (
            # but not if it's the first one so
            if stacked_par > 1:
                tmp += _char
        elif _char == ')':
            stacked_par -= 1
            if stacked_par < 0 or start_of_expression:
                errors.append("validator.servicehosttpls_complex_expression_too_many_closing_parenthesis")
                tmp = ''
                continue
            start_of_expression = False
            
            if stacked_par == 0:
                # print "THIS is closing a sub compress expression", tmp
                tmp = tmp.strip()
                son_node, son_errors = _do_parse_complex_exp(tmp)
                errors.extend(son_errors)
                node['sons'].append(son_node)
                in_part = False
                # OK now clean the tmp so we start clean
                tmp = ''
            else:
                # ok here we are still in a huge par, we just close one sub one
                tmp += _char
        elif _char == '!':
            if start_of_expression:
                tmp += _char
            else:
                errors.append("validator.servicehosttpls_complex_expression_bang")
        else:
            # Maybe it's a classic character, if so, continue
            start_of_expression = False
            tmp += _char
    
    # Be sure to manage the trainling part when the line is done
    tmp = tmp.strip()
    if tmp != '':
        son_node, son_errors = _do_parse_complex_exp(tmp)
        errors.extend(son_errors)
        node['sons'].append(son_node)
    if stacked_par > 0:
        errors.append("validator.servicehosttpls_complex_expression_too_many_opening_parenthesis")
    
    if start_of_expression:
        errors.append("validator.servicehosttpls_complex_expression_ends_with_operator")
    
    if node['operand'] == None and len(node['sons']) == 1:
        node['operand'] = node['sons'][0]['operand']
        node['content'] = node['sons'][0]['content']
        node['sons'] = node['sons'][0]['sons']
    
    return node, errors


def parse_complex_exp(exp):
    node, errors = _do_parse_complex_exp(exp)
    if errors:
        errors_list = ",".join(errors)
        raise SyntaxError(errors_list)
    return node


def _return_parent_node(n, ns, l):
    if n['not_value']:
        return '!(%s)' % n['operand'].join(ns)
    else:
        return '(%s)' % n['operand'].join(ns) if l else n['operand'].join(ns)


def get_unparse_complex_exp(parsed_exp):
    if not parsed_exp:
        return ''
    
    
    def flatten(node, level):
        return '%s%s' % ('!' if node['not_value'] else '', node['content'])
    
    
    return visit_node(parsed_exp, flatten, _return_parent_node)


def visit_node(node, return_leaf_node, return_parent_node=lambda n, ns, l: '', level=0):
    if not node:
        return
    if node['operand'] == OPERAND.TEMPLATE_OP:
        return return_leaf_node(node, level)
    else:
        return return_parent_node(node, [visit_node(son_node, return_leaf_node, return_parent_node, level + 1) for son_node in node['sons']], level)
