#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2020
# This file is part of Shinken Enterprise, all rights reserved.
import re
from collections import namedtuple
from ...component.abstract_component import AbstractComponent
from ...dao.def_items import DEF_ITEMS, ITEM_TYPE, NOT_TO_LOOK

ILLEGAL_CHARS = re.compile("""[`~!$%^&*"|'<>?,()=/+]""")
ILLEGAL_CHARS_SERVICE = re.compile(r"""[`~!$%^&*"|'<>?,()=+]""")

PROPERTIES_NOT_IN_TYPE = {
    ITEM_TYPE.SERVICETPLS        : set(['hostgroup_name', 'host_name']),
    ITEM_TYPE.SERVICESCLUSTERS   : set(['hostgroup_name']),
    ITEM_TYPE.SERVICESCLUSTERTPLS: set(['hostgroup_name']),
    ITEM_TYPE.HOSTTPLS           : set(['address']),
    ITEM_TYPE.CLUSTERS           : set([
        'parents',
        'for_all_users',
        'max_check_attempts',
        'check_interval',
        'retry_interval',
        'active_checks_enabled',
        'passive_checks_enabled',
        'poller_tag',
        'process_perf_data',
        'check_freshness',
        'freshness_threshold',
        'service_excludes',
    ]),
    ITEM_TYPE.CLUSTERTPLS        : set([
        'parents',
        'for_all_users',
        'max_check_attempts',
        'check_interval',
        'retry_interval',
        'active_checks_enabled',
        'passive_checks_enabled',
        'poller_tag',
        'process_perf_data',
        'check_freshness',
        'freshness_threshold',
        'service_excludes',
    ]),
    ITEM_TYPE.CONTACTS           : set(['for_all_users']),
    ITEM_TYPE.CONTACTTPLS        : set(['password']),
}
KNOWN_PROPERTIES = {}

ValidationMessage = namedtuple('ValidationMessage', ['level', 'message', 'params'])


class LEVEL_MESSAGE(object):
    OK = 'ok'
    WARNING = 'warning'
    ERROR = 'error'


class RulesComponent(AbstractComponent):
    
    def __init__(self):
        global KNOWN_PROPERTIES
        for item_type in DEF_ITEMS:
            cls = DEF_ITEMS[item_type]['class']
            properties_not_in_type = PROPERTIES_NOT_IN_TYPE.get(item_type, set())
            KNOWN_PROPERTIES[item_type] = {
                'properties'               : set(getattr(cls, 'properties', {}).keys()) - properties_not_in_type,
                'passthrough'              : set(getattr(cls, 'passthrough', {}).keys()) - properties_not_in_type,
                'blacklist'                : set(getattr(cls, 'blacklist', {}).keys()) - properties_not_in_type,
                'not_inherited_passthrough': set(getattr(cls, 'not_inherited_passthrough', {}).keys()) - properties_not_in_type,
            }
            
            known_props = KNOWN_PROPERTIES[item_type]
            known_props['all'] = NOT_TO_LOOK | known_props['properties'] | known_props['passthrough'] | known_props['not_inherited_passthrough']
    
    
    def init(self):
        pass
    
    class IS_UNKNOWN_PROPERTY(object):
        EMPTY_PROPERTY = 'empty_property'
        STARTING_WITH_DOLLAR_PROPERTY = 'starting_with_dollar_property'
        DATA_MUST_BE_IN_UPPER_CASE = 'data_must_be_in_upper_case'
        PROHIBITED_PROPERTY = 'prohibited_property'
        UNKNOWN_PROPERTY = 'unknown_property'
    
    @staticmethod
    def is_unknown_property(name_property, item_type):
        # tell if the properties is incorrect , incorrect properties are :
        #  - custom data not in uppercase
        #  - properties starting with $
        #  - properties in blacklist
        #  - properties not known in item (see arbiter objects / KNOWN_PROPERTIES)
        
        if not name_property:
            return [ValidationMessage(LEVEL_MESSAGE.ERROR, RulesComponent.IS_UNKNOWN_PROPERTY.EMPTY_PROPERTY, (name_property,))]
        if name_property == '_id':
            return []
        
        if '[FORCE]' in name_property:
            name_property = name_property.replace('[FORCE]', '').strip()
        
        if name_property[0] == '$':
            return [ValidationMessage(LEVEL_MESSAGE.ERROR, RulesComponent.IS_UNKNOWN_PROPERTY.STARTING_WITH_DOLLAR_PROPERTY, (name_property,))]
        if name_property[0] == '_' and name_property != name_property.upper():
            return [ValidationMessage(LEVEL_MESSAGE.ERROR, RulesComponent.IS_UNKNOWN_PROPERTY.DATA_MUST_BE_IN_UPPER_CASE, (name_property,))]
        if name_property in KNOWN_PROPERTIES[item_type]['blacklist']:
            return [ValidationMessage(LEVEL_MESSAGE.ERROR, RulesComponent.IS_UNKNOWN_PROPERTY.PROHIBITED_PROPERTY, (name_property,))]
        elif (item_type != ITEM_TYPE.TIMEPERIODS) and (name_property not in KNOWN_PROPERTIES[item_type]['all']) and (not name_property.startswith("_")):
            return [ValidationMessage(LEVEL_MESSAGE.ERROR, RulesComponent.IS_UNKNOWN_PROPERTY.UNKNOWN_PROPERTY, (name_property,))]
        
        return []
    
    class IS_UNKNOWN_TYPE(object):
        EMPTY_TYPE = "empty_type"
        UNKNOWN_PROPERTY = "unknown_type"
    
    @staticmethod
    def is_unknown_type(item_type):
        if not item_type:
            return [ValidationMessage(LEVEL_MESSAGE.ERROR, RulesComponent.IS_UNKNOWN_TYPE.EMPTY_TYPE, (item_type,))]
        if item_type not in KNOWN_PROPERTIES:
            return [ValidationMessage(LEVEL_MESSAGE.ERROR, RulesComponent.IS_UNKNOWN_TYPE.UNKNOWN_PROPERTY, (item_type,))]
        return []
    
    class IS_VALID_NAME(object):
        MISSING_NAME = 'missing_name'
        RESERVED_NAME = 'reserved_name'
        INVALID_CHAR_IN_NAME = 'invalid_char_in_name'
    
    @staticmethod
    def is_valid_name(_name, item_type):
        # Refuse objects with no or invalid name
        if not _name:
            return [ValidationMessage(LEVEL_MESSAGE.ERROR, RulesComponent.IS_VALID_NAME.MISSING_NAME, (_name,))]
        
        _name = _name.strip()
        invalid_name = ('null', 'shinken-service', 'shinken-host', 'shinken-cluster')
        if _name in invalid_name:
            return [ValidationMessage(LEVEL_MESSAGE.ERROR, RulesComponent.IS_VALID_NAME.RESERVED_NAME, (_name, invalid_name))]
        
        illegal_chars = ILLEGAL_CHARS_SERVICE if item_type in ITEM_TYPE.ALL_SERVICES else ILLEGAL_CHARS
        pattern_illegal_chars = (ILLEGAL_CHARS_SERVICE.pattern if item_type in ITEM_TYPE.ALL_SERVICES else ILLEGAL_CHARS.pattern)[1:-1]
        protected_string = '$KEY$' if item_type in ITEM_TYPE.ALL_SERVICES else ''
        testing_name = _name.replace(protected_string, '', 1)
        
        if re.search(illegal_chars, testing_name):
            return [ValidationMessage(LEVEL_MESSAGE.ERROR, RulesComponent.IS_VALID_NAME.INVALID_CHAR_IN_NAME, (_name, pattern_illegal_chars))]
        return []
