#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.basesubprocess import EventHandler
from shinken.log import PartLogger
from shinken.misc.type_hint import Dict, Any, NoReturn, Type, Optional, List

from shinkensolutions.api.synchronizer import ITEM_TYPE, ItemType, SourceTranslatePart, RulesComponent
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.distributor.option_distributor import DefineBy
from shinkensolutions.api.synchronizer.source.file_loader import FileLoader
from shinkensolutions.api.synchronizer.source.item import ITEM_TYPE_TO_SOURCE_ITEM
from shinkensolutions.api.synchronizer.source.item.origin_item import OriginItem
from shinkensolutions.api.synchronizer.source.item.source_item import SourceItem
from shinkensolutions.api.synchronizer.source.mapping.origin_to_source.mapper_element import MappingElement
from shinkensolutions.api.synchronizer.source.origin_item_description import OriginItemDescription
from shinkensolutions.api.synchronizer.source.source_configuration_value import SERVICE_MODE
from shinkensolutions.api.synchronizer.source.source_exception import SourceException
from shinkensolutions.api.synchronizer.source.validation_state import ValidationState
from shinkensolutions.api.synchronizer.source.api_item_properties import ApiItemProperties

UNMAPPED_PROPERTIES = 'unmapped_properties'
UNMAPPED_PROPERTY = 'unmapped_property'
DEFAULT_MAPPER = 'default_mapper'
MapperName = str
PropName = str
Mapping = Dict[PropName, MappingElement]


class FILE_STATUS(object):
    OK = 'ok'
    ERROR = 'error'
    NOT_EXISTING = 'not_existing'


class ProxyDisableMapper(object):
    def map(self, origin_item_type, origin_item, origin_raw_data=None, mapper_name=None):
        # type: (ItemType, OriginItem, Any, Optional[List[MapperName]]) -> SourceItem
        raise SourceException('The mapper service is disable. You can activate it with SourceConfiguration.set_service_mapping_origin_to_source.')


class ReloadMapperEvent(EventHandler):
    
    def __init__(self, mapper_origin_to_source):
        # type: (MapperOriginToSource) -> None
        super(ReloadMapperEvent, self).__init__('reload-mapper')
        self.mapper_origin_to_source = mapper_origin_to_source
    
    
    def callback(self):
        self.mapper_origin_to_source.reload_mapper()


class AbstractMapperOriginToSource(object):
    
    def __init__(self, item_type, mapping=None):
        # type: (ItemType, Mapping) -> None
        self.item_type = item_type
        self.mapping = mapping
    
    
    def map(self, origin_item, raw_data):
        # type: (OriginItem, Any) -> SourceItem
        source_item = self._get_source_item_instance()
        for origin_prop_name, mapping_rule in self.mapping.iteritems():
            if mapping_rule.is_valid and mapping_rule.source_prop:
                origin_prop_name = origin_prop_name.replace('.', '__')
                origin_item_value = self.get_item_property(origin_item, origin_prop_name)
                if origin_item_value:
                    setattr(source_item, mapping_rule.source_prop, origin_item_value)
        
        source_item.raw_data = raw_data
        self._post_mapping(source_item)
        return source_item
    
    
    def get_item_property(self, item, item_property):
        raise NotImplementedError()
    
    
    def _get_source_item_instance(self):
        # type: () -> SourceItem
        raise NotImplementedError()
    
    
    def _pre_mapping(self, origin_item):
        # type: (OriginItem) -> NoReturn
        pass
    
    
    def _post_mapping(self, source_item):
        # type: (SourceItem) -> NoReturn
        pass


class GenericDictMapperOriginToSource(AbstractMapperOriginToSource):
    def _get_source_item_instance(self):
        return ITEM_TYPE_TO_SOURCE_ITEM[self.item_type]()
    
    
    def get_item_property(self, item, item_property):
        if '.' in item_property:
            param_to_link_attr_name_split = item_property.split('.', 1)
            item = item.get(param_to_link_attr_name_split[0], {})
            item_property = param_to_link_attr_name_split[1]
            if item:
                return self.get_item_property(item, item_property)
            else:
                return ''
        else:
            return item.get(item_property, '')


class GenericObjectMapperOriginToSource(AbstractMapperOriginToSource):
    def _get_source_item_instance(self):
        return ITEM_TYPE_TO_SOURCE_ITEM[self.item_type]()
    
    
    def get_item_property(self, item, item_property):
        if '.' in item_property:
            param_to_link_attr_name_split = item_property.split('.', 1)
            item = getattr(item, param_to_link_attr_name_split[0], {})
            item_property = param_to_link_attr_name_split[1]
            if item:
                return self.get_item_property(item, item_property)
            else:
                return ''
        else:
            return getattr(item, item_property, '')


class MapperOriginToSource(object):
    
    def __init__(self, logger, translator, rules_component, file_loader, service_mode, mapper_class, origin_item_description, api_item_properties, mappers_name=None):
        # type: (PartLogger, SourceTranslatePart, RulesComponent,FileLoader, str, Type[AbstractMapperOriginToSource], OriginItemDescription, ApiItemProperties, Optional[List[MapperName]]) -> None
        self._service_mode = service_mode
        self._mapping = {}  # type:Dict[MapperName, Dict[ItemType, Mapping]]
        self._mappers = {}  # type: Dict[MapperName, Dict[ItemType, AbstractMapperOriginToSource]]
        self.logger = logger
        self._rules_component = rules_component
        self._translator = translator
        self.service_mode = service_mode
        self.mappers_name = mappers_name or [DEFAULT_MAPPER]
        self.file_loader = file_loader
        self.mapper_class = mapper_class
        self.user_mapping_files = []  # type: List[Dict]
        self.validation_state = ValidationState(self.logger, self._translator)  # type: ValidationState
        self.origin_item_description = origin_item_description
        self.api_item_properties = api_item_properties
        self.load()
    
    
    def reload_mapper(self):
        self.origin_item_description.reload()
        self.api_item_properties.reload()
        self.load()
    
    
    def load(self):
        self.logger.info('reload mapper')
        new_mapping = {UNMAPPED_PROPERTIES: {}}  # type: Dict[MapperName, Dict[ItemType, Mapping]]
        validation_state = ValidationState(self.logger, self._translator)
        user_mapping_files = []
        
        if self.origin_item_description.validation_state.has_error():
            user_mapping_files.append({'path': self.origin_item_description.user_file.path, 'status': FILE_STATUS.ERROR})
        elif self.origin_item_description.user_file.exist():
            user_mapping_files.append({'path': self.origin_item_description.user_file.path, 'status': FILE_STATUS.OK})
        else:
            user_mapping_files.append({'path': self.origin_item_description.user_file.path, 'status': FILE_STATUS.NOT_EXISTING})
        validation_state.update_from_validation_state(self.origin_item_description.validation_state)
        
        if self.api_item_properties.validation_state.has_error():
            user_mapping_files.append({'path': self.api_item_properties.user_file.path, 'status': FILE_STATUS.ERROR})
        elif self.api_item_properties.user_file.exist():
            user_mapping_files.append({'path': self.api_item_properties.user_file.path, 'status': FILE_STATUS.OK})
        else:
            user_mapping_files.append({'path': self.api_item_properties.user_file.path, 'status': FILE_STATUS.NOT_EXISTING})
        validation_state.update_from_validation_state(self.api_item_properties.validation_state)
        
        
        
        for mapper_name in self.mappers_name:
            new_mapping_by_type = {}  # type:Dict[ItemType, Mapping]
            new_mapping[mapper_name] = new_mapping_by_type
            files = self.file_loader.get_named_mapping_rule(mapper_name)
            file_default_mapping_rule = files.default_file
            file_user_mapping_rule = files.user_file
            
            try:
                default_mapping_rules = file_default_mapping_rule.load()
            except Exception as e:
                validation_state.add_error(translate_key='error.fail_to_load_file', params=(file_default_mapping_rule.path, str(e)))
                self.validation_state = validation_state
                return
            if not isinstance(default_mapping_rules, dict):
                validation_state.add_error(translate_key='error.fail_to_load_file', params=(file_default_mapping_rule.path,))
                self.validation_state = validation_state
                return
            
            user_mapping_rules = {}
            if self.service_mode != SERVICE_MODE.NOT_OVERLOAD_BY_USER:
                if file_user_mapping_rule.exist():
                    status = FILE_STATUS.OK
                    try:
                        user_mapping_rules = file_user_mapping_rule.load()
                    except Exception as e:
                        validation_state.add_error(translate_key='error.fail_to_load_file', params=(file_user_mapping_rule.path, str(e)))
                        self.validation_state = validation_state
                        status = FILE_STATUS.ERROR
                    if not isinstance(user_mapping_rules, dict):
                        validation_state.add_error(translate_key='error.fail_to_load_file', params=(file_user_mapping_rule.path,))
                        self.validation_state = validation_state
                        status = FILE_STATUS.ERROR
                else:
                    status = FILE_STATUS.NOT_EXISTING
                    validation_state.add_warning(translate_key='warning.missing_user_file', params=(file_user_mapping_rule.path,))
                user_mapping_files.append({'path': file_user_mapping_rule.path, 'status': status})
            
            self._build_mapping(mapper_name, new_mapping_by_type, default_mapping_rules, user_mapping_rules)
        
        self._add_unmapped_properties(new_mapping)
        self._set_description(new_mapping)
        self._validate(new_mapping, validation_state)
        
        for mapper_name in self.mappers_name:
            if mapper_name not in self._mappers:
                self._mappers[mapper_name] = {}
            for item_type in ITEM_TYPE.ALL_TYPES:
                self._mappers[mapper_name][item_type] = self.mapper_class(item_type, new_mapping.get(mapper_name, {}).get(item_type, {}))
        
        self._mapping = new_mapping
        self.validation_state = validation_state
        self.user_mapping_files = user_mapping_files
        return
    
    
    def _add_unmapped_properties(self, new_mapping):
        unmapped_properties = new_mapping[UNMAPPED_PROPERTIES]
        for item_type, known_properties in self.api_item_properties.properties_allows.iteritems():
            for known_property in known_properties:
                mappings = self._search_mapping(new_mapping, item_type, known_property)
                if not mappings:
                    if item_type not in unmapped_properties:
                        unmapped_properties[item_type] = {}
                    default_mapper_by_type = unmapped_properties[item_type]
                    default_mapper_by_type[known_property] = MappingElement(item_type, UNMAPPED_PROPERTY, known_property, translator=self._translator, is_mapped=False, api_item_properties=self.api_item_properties)
    
    
    def _set_description(self, new_mapping):
        for item_type, item_description in self.origin_item_description.origin_item_properties_description.iteritems():
            for item_property, property_description in item_description.iteritems():
                mappings = self._search_mapping(new_mapping, item_type, item_property)
                if mappings:
                    for mapping in mappings:
                        mapping.description = property_description
    
    
    @staticmethod
    def _search_mapping(new_mapping, item_type, item_property):
        # type: (Dict[MapperName, Dict[ItemType, Mapping]], ItemType, PropName) -> Optional[List[MappingElement]]
        mappings = []
        for mapping_by_type in new_mapping.itervalues():
            mapping = mapping_by_type.get(item_type, {}).get(item_property, None)
            if mapping:
                mappings.append(mapping)
        return mappings if mappings else None
    
    
    @staticmethod
    def _validate(new_mapping, validation_state):
        found_invalid_key = False
        for mappings in new_mapping.itervalues():
            for mapping in mappings.itervalues():
                for m in mapping.itervalues():
                    if not m.is_valid:
                        validation_state.extra_counter_warning_number += 1
                        found_invalid_key = True
        if found_invalid_key:
            validation_state.add_warning(translate_key='warning.invalid_key_in_mapping', params=(validation_state.extra_counter_warning_number,))
    
    
    def _build_mapping(self, mapper_name, new_mapping, default_mapping_rules, user_mapping_rules):
        # type: (MapperName, Dict[ItemType, Mapping], Dict, Dict) -> NoReturn
        
        for item_type, mapping in default_mapping_rules.iteritems():
            by_type_mapping = new_mapping.get(item_type, None)
            if by_type_mapping is None:
                new_mapping[item_type] = by_type_mapping = {}
            for origin_prop, source_prop in mapping.iteritems():
                mapping = MappingElement(item_type, source_prop, origin_prop, is_set_by_user=DefineBy.BY_DEFAULT, mapper_name=mapper_name, translator=self._translator, api_item_properties=self.api_item_properties)
                by_type_mapping[origin_prop] = mapping
        
        for item_type, mapping in user_mapping_rules.iteritems():
            by_type_mapping = new_mapping.get(item_type, None)
            if by_type_mapping is None:
                new_mapping[item_type] = by_type_mapping = {}
            for origin_prop, source_prop in mapping.iteritems():
                mapping = MappingElement(item_type, source_prop, origin_prop, is_set_by_user=DefineBy.BY_USER, mapper_name=mapper_name, translator=self._translator, api_item_properties=self.api_item_properties)
                by_type_mapping[origin_prop] = mapping
    
    
    def have_multi_mapper(self):
        return len(self.mappers_name) != 1
    
    
    def get_mapping(self):
        # type: () -> Dict[MapperName, Dict[ItemType, Mapping]]
        return self._mapping
    
    
    def map(self, origin_item_type, origin_item, origin_raw_data=None, mapper_name=DEFAULT_MAPPER):
        # type: (ItemType, OriginItem, Any, MapperName) -> SourceItem
        if self.validation_state.has_error():
            raise SourceException(self._translator.translate('error.mapper_error', ','.join(self.validation_state.get_errors())))
        source_item_factory = self._mappers[mapper_name][origin_item_type]
        if not origin_raw_data:
            origin_raw_data = origin_item
        return source_item_factory.map(origin_item, origin_raw_data)
