#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import os
import time

from mongo_collection import MongoCollection
from mongo_retry import retry_on_auto_reconnect
from shinken.log import LoggerFactory, PART_INITIALISATION, PartLogger
from shinken.misc.type_hint import NoReturn
from shinkensolutions.lib_modules.configuration_reader import read_string_in_configuration, read_bool_in_configuration, read_int_in_configuration
from shinkensolutions.ssh_mongodb.sshtunnelmongomgr import mongo_by_ssh_mgr


class MongoClient(object):
    def __init__(self, conf, logger=None, default_database='shinken'):
        # type: (object, PartLogger, str) -> NoReturn
        self._database = None
        
        if logger:
            self.logger = logger
        else:
            self.logger = LoggerFactory.get_logger()
        
        self.logger_init = self.logger.get_sub_part(PART_INITIALISATION).get_sub_part('MONGO')
        self.logger = self.logger.get_sub_part('MONGO')
        
        # Mongodb connection part
        self._name_database = read_string_in_configuration(conf, 'database', default_database)
        self._uri = read_string_in_configuration(conf, 'uri', 'mongodb://localhost/?w=1&fsync=false')
        self._replica_set = read_string_in_configuration(conf, 'replica_set', '')
        self._use_ssh_tunnel = read_bool_in_configuration(conf, 'use_ssh_tunnel', False)
        self._ssh_keyfile = read_string_in_configuration(conf, 'ssh_keyfile', '~shinken/.ssh/id_rsa')
        self._ssh_user = read_string_in_configuration(conf, 'ssh_user', os.getenv('USER'))
        self._use_ssh_retry_failure = read_int_in_configuration(conf, 'use_ssh_retry_failure', 1)
        
        self._auto_reconnect_max_try = read_int_in_configuration(conf, 'auto_reconnect_max_try', 3)
        self._auto_reconnect_sleep_between_try = read_int_in_configuration(conf, 'auto_reconnect_sleep_between_try', 3)
        
        self.logger_init.info('Parameter load for database connection')
        self.logger_init.info('   - database ------------------------- :[%s]' % self._name_database)
        self.logger_init.info('   - uri ------------------------------ :[%s]' % self._uri)
        self.logger_init.info('   - replica_set ---------------------- :[%s]' % self._replica_set)
        self.logger_init.info('   - use_ssh_tunnel ------------------- :[%s]' % self._use_ssh_tunnel)
        self.logger_init.info('   - ssh_keyfile ---------------------- :[%s]' % self._ssh_keyfile)
        self.logger_init.info('   - ssh_user ------------------------- :[%s]' % self._ssh_user)
        self.logger_init.info('   - use_ssh_retry_failure ------------ :[%s]' % self._use_ssh_retry_failure)
        self.logger_init.info('   - auto_reconnect_max_try ----------- :[%s]' % self._auto_reconnect_max_try)
        self.logger_init.info('   - auto_reconnect_sleep_between_try - :[%s]' % self._auto_reconnect_sleep_between_try)
    
    
    # Get a connection
    # * requester: string that identify who ask for this connection, used for logging and SSH process display
    def init(self, requester='unknown'):
        time_start = time.time()
        self.logger_init.info('Try to open a Mongodb connection to %s:%s' % (self._uri, self._name_database))
        con_result = mongo_by_ssh_mgr.get_connection(
            self._uri,
            self.logger_init,
            fsync=False,
            replica_set=self._replica_set,
            use_ssh=self._use_ssh_tunnel,
            ssh_keyfile=self._ssh_keyfile,
            ssh_user=self._ssh_user,
            ssh_retry=self._use_ssh_retry_failure,
            requestor=requester,
        )
        mongo_client = con_result.get_connection()
        self._database = getattr(mongo_client, self._name_database)
        self.logger_init.info('Open mongo connection done in %s' % self.logger.format_chrono(time_start))
    
    
    def get_collection(self, collection_name):
        return MongoCollection(collection_name, self._database, self._auto_reconnect_max_try, self._auto_reconnect_sleep_between_try, logger=self.logger)
    
    
    @retry_on_auto_reconnect
    def list_name_collections(self):
        return list(self._database.collection_names())
    
    
    @retry_on_auto_reconnect
    def _unsafe_rename_collection(self, old_name, new_name):
        getattr(self._database, old_name).rename(new_name)
    
    
    @retry_on_auto_reconnect
    def drop_collection(self, collection_name):
        getattr(self._database, collection_name).drop()
    
    
    @retry_on_auto_reconnect
    def repair_database(self):
        self.logger.info('Start repair/defragging database this operation may be long')
        self._database.repairDatabase()
    
    
    @retry_on_auto_reconnect
    def get_size_of_database(self):
        col_stats = self._database.command('dbstats')
        total_size = col_stats['storageSize']
        return total_size
    
    
    def collection_exist(self, collection_name):
        return collection_name in self.list_name_collections()
    
    
    def rename_collection(self, old_name, new_name):
        if self.collection_exist(new_name):
            self.drop_collection(old_name)
        self._unsafe_rename_collection(old_name, new_name)
