#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import json
import random
from datetime import datetime

from shinkensolutions.lib_checks.common import Result, RaiseOnExitOptionParser, ParseOptionError, EXIT_STATUS, BREAK_LINE, ShinkenUtils, HTMLTag, HTMLList, HTMLTable, Utils

random.seed()

VERSION = '0.1'
DAEMON_TYPE = 'broker'

MAX_SIZE_REQUEST_INFO = 2500

TAG_FOR_STATE = {
    EXIT_STATUS.OK      : HTMLTag.OK,
    EXIT_STATUS.WARNING : HTMLTag.WARNING,
    EXIT_STATUS.CRITICAL: HTMLTag.CRITICAL
}

DATE_FORMAT = '%Y/%m/%d'

result = Result()

parser = RaiseOnExitOptionParser('%prog [options] [--help]', version='%prog ' + VERSION)
parser.add_option('-H', '--hostname', dest='hostname', help='The hostname of the shinken daemon')
parser.add_option('-p', '--port', dest='port', type='int', help='The port of the shinken daemon', default='7772')
parser.add_option('-t', '--timeout', dest='timeout', type='int', default=3, help='timeout to connect to the shinken daemon. Default : 3')
parser.add_option('-m', '--minutes', dest='minutes_of_stats', type='int', default=1, help='Number of minutes worth of stats displayed. Default : 1 minute')
parser.add_option('-w', '--webui', dest='webui', default='', help='Name of the webui to check. Default : the first one')
parser.add_option('--shinkenversion', dest='shinken_supervisor_version', default='', help='The shinken version number used to compare with the monitored shinken. Mandatory if in shinken mode.')


def _parse_args():
    opts = None
    try:
        opts, args = parser.parse_args()
        if args:
            parser.error('Does not accept arguments.')
        if not opts.hostname:
            parser.error('Missing parameter hostname (-H/--hostname)')
        if not opts.port:
            parser.error('Missing parameter port (-p/--port)')
    except ParseOptionError as e:
        if e.msg:
            result.hard_exit(EXIT_STATUS.CRITICAL, 'Fail to parse command argument: %s %s' % (BREAK_LINE, BREAK_LINE.join(e.msg.split('\n'))))
        exit(0)
    return opts


def check_perf(event_container_module_info, sec_of_stat):
    status = EXIT_STATUS.OK
    
    if not event_container_module_info:
        result.add_check(status=EXIT_STATUS.CRITICAL, output='Cannot get stats for module. Checks the broker logs.')
        return
    elif event_container_module_info.get('error', ''):
        result.add_check(status=EXIT_STATUS.CRITICAL, output=event_container_module_info['error'])
        return
    
    top_request = event_container_module_info.get('top_request', False)
    top_request_duration = event_container_module_info.get('top_request_duration', 0)
    str_top_request_duration = Utils.print_human_readable_period(top_request_duration)
    if top_request:
        str_top_request = []
        header = ['request info', 'at', 'time']
        for r in top_request:
            r_duration = r.get('duration', 0)
            r_at = r.get('time', 0)
            r_filters = str(r.get('filters', ''))
            r_endpoint = r.get('stat_endpoint', '')
            r_name = r.get('stat_name', 0)
            
            if len(r_filters) > MAX_SIZE_REQUEST_INFO:
                r_filters = '%s ...' % r_filters[:MAX_SIZE_REQUEST_INFO]
            if len(r_endpoint) > MAX_SIZE_REQUEST_INFO:
                r_endpoint = '%s ...' % r_endpoint[:MAX_SIZE_REQUEST_INFO]
            
            r_filters = r_filters.replace('|', '&#124;')
            
            request_info = '<div>%s</div><div class="skn-endp">%s</div><div class="skn-fltr">%s</div>' % (r_name, r_endpoint, str(r_filters))
            
            line = [
                request_info,
                Utils.print_time(r_at),
                Utils.print_human_readable_period(r_duration),
            ]
            str_top_request.append(line)
        
        class_uuid = HTMLTable.generate_class_uuid()
        col_width = 20
        col_request_info_width = 60
        extra_style = [
            '.%s .skn-ict tbody tr th, .%s tbody tr td{width: %s%%; max-width: %s%%;}' % (class_uuid, class_uuid, col_width, col_width),
            '.%s .skn-ict tbody tr th:first-child, .%s .skn-ict tbody tr td:first-child {width: %s%%; max-width: %s%%;}' % (class_uuid, class_uuid, col_request_info_width, col_request_info_width),
            '.skn-endp, .skn-fltr {font-style:italic;color:gray;margin-left:10px;font-size: 10px;}',
        ]
        top_request = HTMLTable.table(header, str_top_request, all_col_same_width=False, class_uuid=class_uuid, extra_style=extra_style)
    else:
        top_request = 'No request'
    
    have_stats = event_container_module_info.get('have_stats', False)
    str_stat_duration = '%s minute' % (int(sec_of_stat / 60))
    if have_stats:
        nb_request_to_event_page = event_container_module_info.get('nb_request_to_event_page', 0)
        time_for_request_to_event_page = event_container_module_info.get('time_for_request_to_event_page', 0)
        nb_event_read = event_container_module_info.get('nb_event_read', 0)
        
        str_nb_request_to_event_page = Utils.print_human_readable_number(nb_request_to_event_page)
        str_time_for_request_to_event_page = Utils.print_human_readable_period(time_for_request_to_event_page)
        str_average = Utils.print_human_readable_period(float(time_for_request_to_event_page) / float(nb_request_to_event_page))
        str_nb_event_read = Utils.print_human_readable_number(nb_event_read)
        
        str_perfs = 'In the last %s there are %s requests to the event manager which take %s for all to resolve (average:%s)' % (str_stat_duration, str_nb_request_to_event_page, str_time_for_request_to_event_page, str_average)
        
        stat_last_period = [
            'Request to event manager : %s resolved in %s' % (str_nb_request_to_event_page, str_time_for_request_to_event_page),
            'Number of event read : %s' % str_nb_event_read,
        ]
        
        result.add_perf_data('request_number_in_last_min', nb_request_to_event_page)
        result.add_perf_data('request_time_in_last_min', time_for_request_to_event_page)
        result.add_perf_data('event_read_in_last_min', nb_event_read)
    
    else:
        str_perfs = 'In the last %s there are no requests done to the event manager' % str_stat_duration
        stat_last_period = [
            'No requests done',
            'Number of event read : 0',
        ]
    
    oldest_event_read = event_container_module_info.get('oldest_event_read', 0)
    oldest_event_read_duration = event_container_module_info.get('oldest_event_read_duration', 0)
    oldest_event_in_db = event_container_module_info.get('oldest_event_in_db', 0)
    
    str_oldest_event_read_duration = Utils.print_human_readable_period(oldest_event_read_duration)
    
    str_oldest_event_in_db = 'No event in database'
    if oldest_event_in_db:
        oldest_event_in_db = datetime.fromtimestamp(oldest_event_in_db)
        str_oldest_event_in_db = 'Oldest event in database : %s' % oldest_event_in_db.strftime(DATE_FORMAT),
    
    if oldest_event_read:
        nb_day_keep = event_container_module_info.get('nb_day_keep', 30)
        oldest_event_read_date = datetime.fromtimestamp(oldest_event_read)
        str_oldest_event_read = oldest_event_read_date.strftime(DATE_FORMAT)
        days_from_now = (oldest_event_read_date - oldest_event_in_db).days
        str_oldest_event_read = 'Oldest event read in last %s : %s (%s days from now). For day %s keep in database' % (str_oldest_event_read_duration, str_oldest_event_read, days_from_now, nb_day_keep)
        
        result.add_perf_data('oldest_event_read_from_now', days_from_now)
    else:
        str_oldest_event_read = 'No event read in last %s' % str_oldest_event_read_duration
    
    long_output = [
        HTMLList.header_list('In the last %s' % str_stat_duration, stat_last_period),
        HTMLList.header_list('Events info', [
            str_oldest_event_in_db,
            str_oldest_event_read
        ]),
        'Top requests for last %s :' % str_top_request_duration,
        top_request
    ]
    
    result.add_check(status=status, output=str_perfs, long_output=''.join(long_output))


def check_modules(raw_stats, sec_of_stat, webui_name):
    webui_list = raw_stats.get('module_stats', {}).get('webui', {})
    
    if not webui_list:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - There is no WebUI module on the Broker daemon.' % TAG_FOR_STATE[EXIT_STATUS.CRITICAL])
        return
    
    if not webui_name:
        webui_name, webui = next(webui_list.iteritems(), ('', {}))
    else:
        webui = next((w_info for w_name, w_info in webui_list.iteritems() if w_name == webui_name), {})
    
    if not webui:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - The WebUI module named "%s" is set on the Broker daemon but is not running.' % (TAG_FOR_STATE[EXIT_STATUS.CRITICAL], webui_name))
        return
    
    webui_modules = webui.get('modules', {})
    if not webui_modules:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - The WebUI module does not have any submodules.' % TAG_FOR_STATE[EXIT_STATUS.CRITICAL])
        return
    
    event_container_module = next(webui_modules.get('event_container', {}).itervalues(), None)
    if not event_container_module:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - There is no Event Container submodule on the WebUI module.' % TAG_FOR_STATE[EXIT_STATUS.CRITICAL])
        return
    
    output = []
    check_perf(event_container_module['raw_stats'], sec_of_stat)
    
    if result.status == EXIT_STATUS.CRITICAL:
        output.insert(0, 'Module on webui : %s is in a critical state.' % webui_name)
    elif result.status == EXIT_STATUS.WARNING:
        output.insert(0, 'Module on webui : %s has some anomalies.' % webui_name)
    elif result.status == EXIT_STATUS.OK:
        output.insert(0, 'Module on webui : %s is working as intended.' % webui_name)
    result.add_check(status=result.status, output=''.join(output), title=True)


def main():
    opts = _parse_args()
    daemon_adr = opts.hostname
    daemon_port = opts.port
    timeout = opts.timeout
    minutes_of_stats = opts.minutes_of_stats
    shinken_supervisor_version = opts.shinken_supervisor_version
    webui = opts.webui
    
    sec_of_stat = (minutes_of_stats * 60)
    html, connection_time = ShinkenUtils.request_get_daemon(result, DAEMON_TYPE, '%s:%s' % (daemon_adr, daemon_port), '/get_raw_stats?param=%s' % sec_of_stat, timeout=timeout)
    raw_stats = json.loads(html)
    ShinkenUtils.minimal_check(result, raw_stats, DAEMON_TYPE, shinken_supervisor_version)
    
    check_modules(raw_stats, sec_of_stat, webui)
    result.exit()


if __name__ == '__main__':
    main()
