#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.


import optparse

import pymongo
import json
import os
import re


def query_map_host_id_name(mongodb_shinken):
    col_sla_info = getattr(mongodb_shinken, 'sla_info')
    all_infos = col_sla_info.find({})
    return dict([(i['_id'], u'==BY_NAME==%s==%s==' % (i['host_name'], i['service_description'])) for i in all_infos if i['_id'] != 'SLA_INFO'])


def query_map_host_name_id(mongodb_shinken):
    col_sla_info = getattr(mongodb_shinken, 'sla_info')
    all_infos = col_sla_info.find({})
    return dict([(u'==BY_NAME==%s==%s==' % (i['host_name'], i['service_description']), i['_id']) for i in all_infos if i['_id'] != 'SLA_INFO'])


def query_map_user_id_name(mongodb_shinken):
    col_user = getattr(mongodb_shinken, 'user')
    all_infos = col_user.find({})
    return dict([(i['_id'], i['userName']) for i in all_infos])


def query_map_user_name_id(mongodb_shinken):
    col_user = getattr(mongodb_shinken, 'user')
    all_infos = col_user.find({})
    return dict([(i['userName'], i['_id']) for i in all_infos])


def export_hive_with_name(export_hive_uuid, mongodb_shinken):
    export_file = 'hive_exported_%s.json' % export_hive_uuid
    print 'start exporting hive [%s] with name in file : [%s]' % (export_hive_uuid, export_file)
    
    full_export = {
        'hive'      : None,
        'dashboards': [],
        'shares'    : [],
    }
    col_hive = getattr(mongodb_shinken, 'hive')
    col_dashboard = getattr(mongodb_shinken, 'dashboard')
    col_share = getattr(mongodb_shinken, 'share')
    
    user_map_id_name = query_map_user_id_name(mongodb_shinken)
    
    print '\nexporting hive : '
    hive = col_hive.find_one({'_id': export_hive_uuid})
    if not hive:
        print '   - Fail to found hive [%s]' % export_hive_uuid
    hive['owner']['uuid'] = user_map_id_name.get(hive['owner']['uuid'], hive['owner']['uuid'])
    print '   - Found hive [%s] owned by [%s]' % (hive['name'], hive['owner']['uuid'])
    
    full_export['hive'] = hive
    if not hive:
        print 'error : hive with id [%s] not found.' % export_hive_uuid
        return 1
    
    msgs = []
    for tile in hive['tiles']:
        dashboard = col_dashboard.find_one({'_id': tile['screen']['uuid']})
        if not dashboard:
            msgs.append('   - Fail to found dashboard [%s]' % tile['screen']['uuid'])
            continue
        msgs.append('   - Found dashboard [%s]' % dashboard['name'])
        full_export['dashboards'].append(dashboard)
    
    print '\nexporting %s dashboards : ' % len(msgs)
    for msg in msgs:
        print msg
    
    print '\nexporting shares : '
    for share in col_share.find({'screen.uuid': export_hive_uuid, 'type': 'personal', 'state.removed': {'$ne': True}}):
        share['owner']['uuid'] = user_map_id_name.get(share['owner']['uuid'], share['owner']['uuid'])
        full_export['shares'].append(share)
        print '   - Found share owned by : [%s]' % share['owner']['uuid']
    
    host_map_id_name = query_map_host_id_name(mongodb_shinken)
    
    export_string_all = json.dumps(full_export)
    export_string_all = export_string_all.decode('utf8')
    
    for _id, _name in host_map_id_name.iteritems():
        if '-' in _id:
            export_string_all = export_string_all.replace(_id, _name)
    
    for _id, _name in host_map_id_name.iteritems():
        if '-' not in _id:
            export_string_all = export_string_all.replace(_id, _name)
    
    export_string_all = export_string_all.encode('utf8')
    with open(export_file, 'w') as f:
        f.write(export_string_all)
    
    print '\nwrite file [%s] done' % export_file


def import_hive_with_name(import_hive_file, mongodb_shinken):
    print 'start importing hive in file %s' % import_hive_file
    
    if not os.path.exists(import_hive_file):
        print 'File %s not found!'
        return 1
    
    with open(import_hive_file, 'r') as f:
        export_as_string = f.read()
    export_as_string = export_as_string.decode('utf8')
    
    print 'load file [%s] done' % import_hive_file
    
    map_name_id = query_map_host_name_id(mongodb_shinken)
    
    to_replace_names = re.findall('(==BY_NAME==[^=]*==[^=]*==)', export_as_string)
    print 'searching host-check id from names'
    for to_replace_name in to_replace_names:
        export_as_string = export_as_string.replace(to_replace_name, map_name_id.get(to_replace_name, to_replace_name))
    
    user_map_name_id = query_map_user_name_id(mongodb_shinken)
    export = json.loads(export_as_string)
    for share in export['shares']:
        share['owner']['uuid'] = user_map_name_id.get(share['owner']['uuid'], share['owner']['uuid'])
    
    owned_by = export['hive']['owner']['uuid']
    export['hive']['owner']['uuid'] = user_map_name_id.get(export['hive']['owner']['uuid'], export['hive']['owner']['uuid'])
    
    col_hive = getattr(mongodb_shinken, 'hive')
    col_dashboard = getattr(mongodb_shinken, 'dashboard')
    col_share = getattr(mongodb_shinken, 'share')
    
    print '\nsave hive'
    col_hive.save(export['hive'])
    print '   - Save hive [%s] owned by [%s]' % (export['hive']['name'], owned_by)
    
    print '\nsave %s dashboards' % len(export['dashboards'])
    for dashboard in export['dashboards']:
        col_dashboard.save(dashboard)
        print '   - Save dashboards [%s]' % dashboard['name']
    
    print '\nsave shares'
    for share in export['shares']:
        col_share.save(share)
        print '   - Save share owned by [%s]' % owned_by
    
    print '\nsave in base [%s] done' % import_hive_file
    print 'reload [%s] done' % import_hive_file


def main():
    help_text = u"""
This tool allows to copy a hive from one Shinken to another even if the elements inside this hive do not have the same id.
The --export parameter is the id of a hive. You can find this id in the hive url:
http://172.16.0.190:7767/static/ui/index.html#/hive/ -->>>>> e1bd246d-379b-3c4c4c-c367-ac3a8c17832f <<<<<<<<--

Prerequisite:
The SLA module must be active: We use the information from the SLA module to map host and check names to their id.
It is necessary that the elements (hosts/checks/users) exist with the same name in the Visualization UI where the hive is imported.
It is necessary that the user who owns the hive has already been connected once in the Visualization UI where the hive is imported.
Only hives in private favorites can be exported.

Export of a hive:
export_hive_with_name.py -e e1bd246d-379b-3c4c-c367-ac3a8c17832f

Importing a hive
export_hive_with_name.py -i hive_exported_e1bd246d-379b-3c4c4c-c367-ac3a8c17832f.json
    """
    
    parser = optparse.OptionParser(usage=help_text)
    parser.add_option('-u', '--url', dest='url', default='localhost', help='URL of the mongo db [default : localhost]')
    parser.add_option('-i', '--import', dest='import_file', help='import hive')
    parser.add_option('-e', '--export', dest='export_hive_uuid', help='export hive')
    
    opts, args = parser.parse_args()
    
    import_file = opts.import_file
    export_hive_uuid = opts.export_hive_uuid
    
    mongodb_con = pymongo.MongoClient('mongodb://%s' % opts.url, fsync=False)
    mongodb_shinken = getattr(mongodb_con, 'shinken')
    
    if export_hive_uuid:
        return export_hive_with_name(export_hive_uuid, mongodb_shinken)
    elif import_file:
        return import_hive_with_name(import_file, mongodb_shinken)
    else:
        print 'Please choose a mode : import or export.\n'
        parser.print_help()
        return 1


if __name__ == '__main__':
    ret = main()
    exit(ret)
