#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import time

from ec_common import NO_STATS, NO_DATA, get_sample
from ec_database_connection import ECDatabaseConnection, ECDatabaseError
from ec_event import Event
from shinken.load import TopList
from shinken.misc.type_hint import NoReturn, Any
from shinken.thread_helper import Thread
from shinkensolutions.date_helper import timestamp_from_datetime, get_datetime_from_utc_to_local_time_zone

SAMPLING_SPEED = 5  # in sec
SAMPLING_SIZE = 60  # nb entry / time  = SAMPLING_SIZE * SAMPLING_SPEED
TOP_DURATION = 60 * 60 * 24  # 24h in sec
KEEP_LAST_EVENT_READ = 60 * 60 * 24  # 24h in sec


class ECReaderData:
    def __init__(self, has_data=True):
        self.has_data = has_data
        self.database_status = True
        
        self.have_stats = False
        self.stat_duration = 0
        self.nb_request_to_event_page = 0
        self.time_for_request_to_event_page = 0
        self.nb_event_read = 0
        self.oldest_event_in_db = 0
        self.oldest_event_read = 0
        self.oldest_event_read_duration = 0
        self.nb_day_keep = 0
        
        self.top_request = []
        self.top_request_duration = 0
    
    
    def export(self):
        return self.__dict__


class ECReaderStats(Thread):
    def __init__(self, ec_module_webui, database_connection):
        # type: (Any, ECDatabaseConnection) -> NoReturn
        super(ECReaderStats, self).__init__(loop_speed=SAMPLING_SPEED, logger=ec_module_webui.logger)
        self.ec_module_webui = ec_module_webui
        self.database_connection = database_connection
        self.request_sampling = [(-1, -1, -1) for _ in xrange(SAMPLING_SIZE)]
        self.request_cumulative = (0, 0, 0)
        self.event_read_sampling = [(-1, -1) for _ in xrange(SAMPLING_SIZE)]
        self.event_read_cumulative = (0, 0)
        self.top_request = TopList(5, descending=True, get_value=lambda x: x['duration'], duration=TOP_DURATION, get_time_entry=lambda x: x['time'], max_buffer_size=50)
        self.oldest_event_read = TopList(5, descending=True, get_value=lambda x: x['event_since'], duration=KEEP_LAST_EVENT_READ, get_time_entry=lambda x: x['time'], max_buffer_size=50)
    
    
    def get_thread_name(self):
        return 'ec-reader-stats'
    
    
    def loop_turn(self):
        if self.request_cumulative[0] != 0:
            self.request_sampling.pop()
            self.request_sampling.insert(0, self.request_cumulative)
        
        if self.event_read_cumulative[0] != 0:
            self.event_read_sampling.pop()
            self.event_read_sampling.insert(0, self.event_read_cumulative)
    
    
    def get_raw_stats(self, seconds_of_stats):
        start_time = time.time()
        data = ECReaderData()
        
        _start, _end = get_sample(self.request_sampling, seconds_of_stats)
        if _start == NO_STATS or _start == NO_DATA:
            data.have_stats = False
        else:
            data.have_stats = True
            data.stat_duration = _start[0] - _end[0]
            data.nb_request_to_event_page = _start[1] - _end[1]
            data.time_for_request_to_event_page = _start[2] - _end[2]
        
        _start, _end = get_sample(self.event_read_sampling, seconds_of_stats)
        if not (_start == NO_STATS or _start == NO_DATA):
            data.nb_event_read = _start[1] - _end[1]
        
        try:
            data.nb_day_keep = self.database_connection.get_day_keep_data()
        except ECDatabaseError as e:
            return {'error' : str(e)}
        except Exception:
            return {}
            
        data.top_request = self.top_request.get_items()
        data.top_request_duration = TOP_DURATION
        
        oldest_event_read = self.oldest_event_read.get_items(1)
        if oldest_event_read:
            oldest_event_read = oldest_event_read[0]
            data.oldest_event_read = timestamp_from_datetime(oldest_event_read['event_since'])
        
        data.oldest_event_read_duration = KEEP_LAST_EVENT_READ
        oldest_event_in_db = self.database_connection.find_oldest_event_in_db()
        if oldest_event_in_db:
            oldest_event_in_db = get_datetime_from_utc_to_local_time_zone(oldest_event_in_db['event_since'])
            data.oldest_event_in_db = timestamp_from_datetime(oldest_event_in_db)
        
        self.logger.log_perf(start_time, 'stats', 'compute reading stats', min_time=0, warn_time=2)
        return data.export()
    
    
    def read_event_stat(self, event):
        # type: (Event) -> NoReturn
        time_time = time.time()
        self.event_read_cumulative = (time_time, self.event_read_cumulative[1] + 1)
        oldest_event_read_entry = {
            'event_since': event.event_since,
            'time'       : time_time,
        }
        self.oldest_event_read.add_item(oldest_event_read_entry)
    
    
    def add_request_stat(self, stat_name, stat_endpoint, filters, stat_time):
        time_time = time.time()
        self.request_cumulative = (time_time, self.request_cumulative[1] + 1, self.request_cumulative[2] + stat_time)
        top_entry = {
            'stat_name'    : stat_name,
            'stat_endpoint': stat_endpoint,
            'filters'      : filters,
            'duration'     : stat_time,
            'time'         : time_time,
        }
        self.top_request.add_item(top_entry)
