#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.objects.module import Module as ShinkenModuleDefinition
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration, read_string_in_configuration
from sla_abstract_component import AbstractComponent
from sla_common import LIST_STATUS, STATUS
from sla_component_manager import ComponentManager

RECOMPUTE_OLD_SLA = 1
DO_NOT_RECOMPUTE_OLD_SLA = 0

WARNING_COUNTS_AS_OK = 1
WARNING_COUNTS_AS_WARNING = 0

DOWNTIME_PERIOD_INCLUDE = 0
DOWNTIME_PERIOD_EXCLUDE = 1
DOWNTIME_PERIOD_OK = 2
DOWNTIME_PERIOD_CRITICAL = 3

UNKNOWN_PERIOD_INCLUDE = 0
UNKNOWN_PERIOD_EXCLUDE = 1
UNKNOWN_PERIOD_OK = 2

NO_DATA_PERIOD_INCLUDE = 0
NO_DATA_PERIOD_EXCLUDE = 1
NO_DATA_PERIOD_OK = 2

DOWNTIME_PERIOD_MAP = {
    'exclude' : DOWNTIME_PERIOD_EXCLUDE,
    'include' : DOWNTIME_PERIOD_INCLUDE,
    'ok'      : DOWNTIME_PERIOD_OK,
    'critical': DOWNTIME_PERIOD_CRITICAL,
}

UNKNOWN_PERIOD_MAP = {
    'exclude': UNKNOWN_PERIOD_EXCLUDE,
    'include': UNKNOWN_PERIOD_INCLUDE,
    'ok'     : UNKNOWN_PERIOD_OK,
}

NO_DATA_PERIOD_MAP = {
    'exclude': NO_DATA_PERIOD_EXCLUDE,
    'include': NO_DATA_PERIOD_INCLUDE,
    'ok'     : NO_DATA_PERIOD_OK,
}


class ComputePercentSla(AbstractComponent):
    
    def __init__(self, conf, component_manager):
        # type: (ShinkenModuleDefinition, ComponentManager) -> None
        super(ComputePercentSla, self).__init__(conf, component_manager)
        self.recompute_old_sla = read_int_in_configuration(conf, 'recompute_old_sla', DO_NOT_RECOMPUTE_OLD_SLA)
        self.warning_counts_as_ok = read_int_in_configuration(conf, 'warning_counts_as_ok', WARNING_COUNTS_AS_WARNING)
        self.downtime_period = DOWNTIME_PERIOD_MAP.get(read_string_in_configuration(conf, 'downtime_period', 'include'), DOWNTIME_PERIOD_INCLUDE)
        self.unknown_period = UNKNOWN_PERIOD_MAP.get(read_string_in_configuration(conf, 'unknown_period', 'include'), UNKNOWN_PERIOD_INCLUDE)
        self.no_data_period = NO_DATA_PERIOD_MAP.get(read_string_in_configuration(conf, 'no_data_period', 'include'), NO_DATA_PERIOD_INCLUDE)
    
    
    def init(self):
        pass
    
    
    def tick(self):
        pass
    
    
    @staticmethod
    def compute_state_sum(ranges, info):
        if not ranges or len(ranges) == 0:
            # self.logger.debug('Trying to archive void day')
            # then simulate a void day, with an unknown state, the whole day, so tag this entry that we will 'lie' into it
            info['missing'] = True
            info['history_inactive'] = 1
            info['history_total'] = 1
            return
        
        total_period = 0
        for _range in ranges:
            start = _range['start']
            end = _range['end']
            status = _range['rc']
            period = end - start
            total_period += period
            
            if status == STATUS.OK:
                prefix = 'ok'
            elif status == STATUS.WARN:
                prefix = 'warn'
            elif status == STATUS.CRIT:
                prefix = 'crit'
            elif status == STATUS.UNKNOWN:
                prefix = 'unknown'
            elif status == STATUS.MISSING_DATA:
                prefix = 'missing'
            else:
                prefix = 'inactive'
            
            info['history_%s' % prefix] = info.get('history_%s' % prefix, 0) + period
            if _range['dt']:
                info['history_dt_%s' % prefix] = info.get('history_dt_%s' % prefix, 0) + period
        info['history_total'] = total_period
    
    
    @staticmethod
    def _set_sla_value(archive_day, sla_name, sla_value):
        if sla_value:
            archive_day[sla_name] = sla_value
        elif sla_name in archive_day:
            del archive_day[sla_name]
    
    
    def compute_sla(self, archive_day):
        sla_format = '%s%s%s%s' % (self.warning_counts_as_ok, self.unknown_period, self.no_data_period, self.downtime_period)
        sla_ok = archive_day.get('history_ok', 0)
        # self.logger.debug('sla_ok : history_ok')
        if self.warning_counts_as_ok:
            sla_ok += archive_day.get('history_warn', 0)
            # self.logger.debug('sla_ok : history_warn')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_CRITICAL):
                sla_ok -= archive_day.get('history_dt_warn', 0)
                # self.logger.debug('sla_ok : - history_dt_warn')
        if self.downtime_period == DOWNTIME_PERIOD_OK:
            # We add downtime time but without the downtime OK part which is already in ok_sum
            sla_ok += archive_day.get('history_dt_crit', 0)
            # self.logger.debug('sla_ok : history_dt_crit')
            if self.warning_counts_as_ok == WARNING_COUNTS_AS_WARNING:
                sla_ok += archive_day.get('history_dt_warn', 0)
                # self.logger.debug('sla_ok : history_dt_warn')
            if not self.unknown_period == UNKNOWN_PERIOD_OK:
                sla_ok += archive_day.get('history_dt_unknown', 0)
                # self.logger.debug('sla_ok : history_dt_unknown')
            if not self.no_data_period == NO_DATA_PERIOD_OK:
                sla_ok += archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0)
                # self.logger.debug('sla_ok : history_dt_missing + history_dt_inactive')
        if self.downtime_period == DOWNTIME_PERIOD_CRITICAL:
            # We remove downtime OK part form ok_sum
            sla_ok -= archive_day.get('history_dt_ok', 0)
            # self.logger.debug('sla_ok : - history_dt_ok')
        if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
            # We remove downtime OK part form ok_sum
            sla_ok -= archive_day.get('history_dt_ok', 0)
            # self.logger.debug('sla_ok : - history_dt_ok')
        if self.unknown_period == UNKNOWN_PERIOD_OK:
            sla_ok += archive_day.get('history_unknown', 0)
            # self.logger.debug('sla_ok : history_unknown')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_CRITICAL):
                sla_ok -= archive_day.get('history_dt_unknown', 0)
                # self.logger.debug('sla_ok : - history_dt_unknown')
        if self.no_data_period == NO_DATA_PERIOD_OK:
            sla_ok += archive_day.get('history_missing', 0) + archive_day.get('history_inactive', 0)
            # self.logger.debug('sla_ok : history_missing + history_inactive')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_CRITICAL):
                sla_ok -= (archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0))
                # self.logger.debug('sla_ok : - history_dt_missing - history_dt_inactive')
        
        sla_crit = archive_day.get('history_crit', 0)
        # self.logger.debug('sla_crit : history_crit[%s]'%history_crit)
        if self.downtime_period == DOWNTIME_PERIOD_CRITICAL:
            sla_crit += \
                archive_day.get('history_dt_ok', 0) + \
                archive_day.get('history_dt_warn', 0) + \
                archive_day.get('history_dt_unknown', 0) + \
                archive_day.get('history_dt_missing', 0) + \
                archive_day.get('history_dt_inactive', 0)
        if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK):
            sla_crit -= archive_day.get('history_dt_crit', 0)
        
        sla_warn = 0
        if self.warning_counts_as_ok == WARNING_COUNTS_AS_WARNING:
            sla_warn = archive_day.get('history_warn', 0)
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
                sla_warn -= archive_day.get('history_dt_warn', 0)
        
        sla_unknown = 0
        if self.unknown_period == UNKNOWN_PERIOD_INCLUDE:
            sla_unknown = archive_day.get('history_unknown', 0)
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
                sla_unknown -= archive_day.get('history_dt_unknown', 0)
        elif self.unknown_period == UNKNOWN_PERIOD_EXCLUDE and self.downtime_period == DOWNTIME_PERIOD_INCLUDE:
            sla_unknown = archive_day.get('history_dt_unknown', 0)
        
        sla_missing = 0
        sla_inactive = 0
        if self.no_data_period == NO_DATA_PERIOD_INCLUDE:
            sla_missing = archive_day.get('history_missing', 0)
            sla_inactive = archive_day.get('history_inactive', 0)
            # self.logger.debug('sla_missing : history_missing')
            # self.logger.debug('sla_inactive : history_inactive')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
                sla_missing -= archive_day.get('history_dt_missing', 0)
                sla_inactive -= archive_day.get('history_dt_inactive', 0)
                # self.logger.debug('sla_missing : - history_dt_missing')a
                # self.logger.debug('sla_inactive : - history_dt_inactive')
        elif self.no_data_period == NO_DATA_PERIOD_EXCLUDE and self.downtime_period == DOWNTIME_PERIOD_INCLUDE:
            sla_missing = archive_day.get('history_dt_missing', 0)
            sla_inactive = archive_day.get('history_dt_inactive', 0)
            # self.logger.debug('sla_missing : history_dt_missing')
            # self.logger.debug('sla_inactive : history_dt_inactive')
        
        sla_total = archive_day.get('history_ok', 0) + archive_day.get('history_warn', 0) + archive_day.get('history_crit', 0)
        # self.logger.debug('sla_total : history_ok + history_warn + history_crit')
        if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
            sla_total -= archive_day.get('history_dt_ok', 0) + archive_day.get('history_dt_warn', 0) + archive_day.get('history_dt_crit', 0)
            # self.logger.debug('sla_total : - history_dt_ok - history_dt_warn - history_dt_crit')
        if self.unknown_period in (UNKNOWN_PERIOD_INCLUDE, UNKNOWN_PERIOD_OK):
            sla_total += archive_day.get('history_unknown', 0)
            # self.logger.debug('sla_total : + history_unknown')
            if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
                sla_total -= archive_day.get('history_dt_unknown', 0)
                # self.logger.debug('sla_total : - history_dt_unknown')
        elif self.downtime_period in (DOWNTIME_PERIOD_INCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
            sla_total += archive_day.get('history_dt_unknown', 0)
            # self.logger.debug('sla_total : + history_dt_unknown')
        if self.no_data_period in (NO_DATA_PERIOD_INCLUDE, NO_DATA_PERIOD_OK):
            sla_total += (archive_day.get('history_missing', 0) + archive_day.get('history_inactive', 0))
            # self.logger.debug('sla_total : history_missing + history_inactive')
            if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
                sla_total -= archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0)
                # self.logger.debug('sla_total : - history_dt_missing - history_dt_inactive')
        elif self.downtime_period in (DOWNTIME_PERIOD_INCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
            sla_total += archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0)
            # self.logger.debug('sla_total : + history_dt_missing + history_dt_inactive')
        # self.logger.debug('sla_ok:[%s] / sla_total:[%s] = [%d]' % (sla_ok, sla_total, sla_ok * 100 / (sla_total if sla_total else 1)))
        
        self._set_sla_value(archive_day, 'sla_ok', sla_ok)
        self._set_sla_value(archive_day, 'sla_crit', sla_crit)
        self._set_sla_value(archive_day, 'sla_warn', sla_warn)
        self._set_sla_value(archive_day, 'sla_unknown', sla_unknown)
        self._set_sla_value(archive_day, 'sla_missing', sla_missing)
        self._set_sla_value(archive_day, 'sla_inactive', sla_inactive)
        
        archive_day['sla_total'] = sla_total
        archive_day['sla_format'] = sla_format
    
    
    def must_update_archive(self, archive):
        sla_format = '%s%s%s%s' % (self.warning_counts_as_ok, self.unknown_period, self.no_data_period, self.downtime_period)
        return self.recompute_old_sla and archive['sla_format'] != sla_format
    
    
    def update_sla(self, archive_day):
        must_update_archive = self.must_update_archive(archive_day)
        
        if must_update_archive:
            self.compute_sla(archive_day)
        
        # for current day we don't have archive entry so we take sla info even if we don't recompute old sla
        if not self.recompute_old_sla and 'archive_format' in archive_day:
            for prefix in LIST_STATUS:
                _sum = archive_day.get('archive_%s' % prefix, 0)
                self._set_sla_value(archive_day, 'sla_%s' % prefix, _sum)
            
            archive_day['sla_total'] = archive_day['archive_total']
            archive_day['sla_format'] = archive_day['archive_format']
        
        return must_update_archive
