#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.log import logger
from shinken.misc.filter import is_authorized


class WidgetService(object):
    app = None
    
    
    def set_app(self, app_):
        self.app = app_
    
    
    @staticmethod
    def _list_element(item_list):
        return [e.get_name() for e in item_list]
    
    
    def _widget_status_basic_infos(self, element):
        _type = element.__class__.my_type
        is_host = (_type == 'host')
        is_service = (_type == 'service')
        is_cluster = False
        e = self.app.helper.get_summary(element)
        
        notification_period = element.notification_period
        if notification_period:
            notification_period = notification_period.get_name()
        else:
            notification_period = 'always'
        
        d = {
            'status'                     : e['status'],
            'fullname'                   : element.get_full_name(),
            'display_name'               : element.display_name,
            'is_flapping'                : element.is_flapping,
            'class'                      : element.__class__.my_type,
            'current_notification_number': element.current_notification_number,
            'output'                     : element.output,
            'long_output'                : element.long_output,
            'context'                    : e['summary'],
            'got_business_rule'          : element.got_business_rule,
            'last_chk'                   : element.last_chk,
            'next_chk'                   : element.next_chk,
            'notes_url'                  : element.notes_url,
            'notes_multi_url'            : element.notes_multi_url,
            'contacts'                   : WidgetService._list_element(element.contacts),
            'contact_groups'             : element.contact_groups,
            'perf_data'                  : element.perf_data,
            'state_type'                 : element.state_type,
            'attempt'                    : element.attempt,
            'max_attempts'               : element.max_check_attempts,
            'check_type'                 : element.check_type,
            'last_change'                : {"status": element.last_state_id, "date": element.last_state_change},
            'flapping'                   : element.percent_state_change,
            'notification_period'        : notification_period,
            'notifications'              : element.notification_list,
            'notification_options'       : element.notification_options,
            'notifications_enabled'      : element.notifications_enabled,
            'is_unknown'                 : False,
            'unsight'                    : False
        }
        
        self.app.helper.get_ack_on_item(element, d)
        self.app.helper.get_downtime_on_item(element, d)
        if is_host:
            d['type'] = 'host'
            is_cluster = element.got_business_rule
            d['uuid'] = element.uuid
            d['host_name'] = element.host_name
            
            children_list = []
            
            for service in element.services:
                summary = self.app.helper.get_summary(service)
                
                children_list.append({
                    'uuid'                       : '%s-%s' % (element.uuid, service.uuid),
                    'status'                     : summary['status'],
                    'context'                    : summary['summary'],
                    'current_notification_number': service.current_notification_number,
                    'service_description'        : service.service_description,
                    'display_name'               : service.display_name,
                    'is_unknown'                 : False
                })
            
            d['children_list'] = sorted(children_list, key=lambda k: k['service_description'])
        
        if is_service:
            d['type'] = 'check'
            d['uuid'] = "%s-%s" % (element.host.uuid, element.uuid)
            d['uuid_parent'] = element.host.uuid
            d['host_name'] = element.host.host_name
            d['service_description'] = element.service_description
        if is_cluster:
            d['type'] = 'cluster'
        return d
    
    
    def api_get_elt_widget_basic(self, uuid, wanted_data=(), extend_checklist=False):
        self.app.response.content_type = 'application/json'
        user = self.app.get_user_auth()
        if not user:
            return self.app.abort(401, '[api_get_elt_widget_basic] user is not authenticated', True)
        
        # Look for an host or a service?
        elt = None
        _to_return = {'element': {}}
        if self.app.is_check_uuid(uuid):
            elt = self.app.datamgr.get_service_by_uuid(uuid)
            _is_host = False
        else:
            elt = self.app.datamgr.get_host_by_uuid(uuid)
            _is_host = True
        
        if elt is None:
            _to_return['element'] = {
                'uuid'      : uuid,
                'is_unknown': True
            }
            return _to_return
        
        if not is_authorized(elt, user, self.app):
            logger.info('[api_get_elt_widget_basic] User is not authorized to view this element')
            _to_return['element'] = self.app.helper.get_unauthorized_item(elt)
            if _is_host:
                _to_return['children_list'] = []
                for check in elt.services:
                    summary = self.app.helper.get_summary(check)
                    _check = {
                        'uuid'      : check.get_instance_uuid(),
                        'status'    : summary['status'],
                        'context'   : summary['summary'],
                        'is_unknown': False
                    }
                    _to_return['children_list'].append(_check)
            return _to_return
        
        infos = self._widget_status_basic_infos(elt)
        
        wanted_properties = [
            'service_description',
            'host_name',
            'current_notification_number',
            'status',
            'display_name',
            'context',
            'uuid',
            'type',
            'is_unknown',
            'unsight'
        ]
        if wanted_data:
            wanted_properties.extend(wanted_data)
        
        filtered_datas = {}
        for key in wanted_properties:
            if infos.get(key, None) is not None:
                filtered_datas[key] = infos[key]
        
        if not extend_checklist and infos['type'] != 'check':
            for check in infos['children_list']:
                del check['current_notification_number']
                del check['service_description']
                del check['display_name']
        
        _to_return['element'] = filtered_datas
        if infos['type'] != 'check':
            _to_return['children_list'] = infos['children_list']
        
        return _to_return
    
    
    def api_get_elt_widget_output(self, uuid):
        return self.api_get_elt_widget_basic(uuid, wanted_data=('long_output', 'output'))
    
    
    def api_get_elt_widget_checklist(self, uuid):
        return self.api_get_elt_widget_basic(uuid, extend_checklist=True)
    
    
    def api_get_elt_widget_full(self, uuid):
        return self.api_get_elt_widget_basic(uuid, wanted_data=('long_output', 'output'), extend_checklist=True)


widget_service = WidgetService()
