#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2016:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#    Martin Benjamin, b.martin@shinken-solutions.com
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import time
import hashlib
import shlex
import os
import subprocess
import signal

import psutil
from shinken.log import logger

if os.name == 'nt':
    import msvcrt
    import ctypes
    from ctypes import windll, byref, wintypes, GetLastError, WinError
    from ctypes.wintypes import HANDLE, DWORD, POINTER, BOOL
    
    TerminateProcess = ctypes.windll.kernel32.TerminateProcess
    
    # Use for  setting a socket to not blocking
    LPDWORD = POINTER(DWORD)
    PIPE_NOWAIT = wintypes.DWORD(0x00000001)
    ERROR_NO_DATA = 232
    # Function used to set pipe to not blocking mode
    SetNamedPipeHandleState = windll.kernel32.SetNamedPipeHandleState
    SetNamedPipeHandleState.argtypes = [HANDLE, LPDWORD, LPDWORD, LPDWORD]
    SetNamedPipeHandleState.restype = BOOL

# Try to read in non-blocking mode, from now this only from now on  Unix systems
try:
    import fcntl
except ImportError:
    fcntl = None

# The minimal execution time for very small probe that system cannot compute correctly
MINIMAL_EXECUTION_TIME = 0.002
id_action = 0

SHELL_PROPERTIES = ('id', 'status', 'command', 't_to_go', 'timeout', 'env', 'module_type', 'execution_time', 'u_time', 's_time', 'check_interval', 'command_name')

# TODO ADD list status
EXIT_STATUS_OK = 0
EXIT_STATUS_RETRY = 2
EXIT_STATUS_FAIL = 3

STATUS_TEXT_0_SCHEDULED = "scheduled"
STATUS_TEXT_1_SCHEDULED = "inpoller"
STATUS_TEXT_2_SCHEDULED = "queue"
STATUS_TEXT_3_SCHEDULED = "launched"
STATUS_TEXT_4_SCHEDULED = "timeout"
STATUS_TEXT_5_SCHEDULED = "done"
STATUS_TEXT_6_SCHEDULED = "waitdep"
STATUS_TEXT_7_SCHEDULED = "havetoresolvedep"
STATUS_TEXT_8_SCHEDULED = "waitconsume"
STATUS_TEXT_9_SCHEDULED = "zombie"


class ACTION_TYPES:
    EVENTHANDLER = 'eventhandler'
    NOTIFICATION = 'notification'
    CHECK = 'check'


class TooManyOpenFiles(Exception):
    def __init__(self, message):
        self.message = message


# When starting a sub process on unix we must:
# * se this process as sid, so when killed, will kill all sub process
def _at_unix_sub_process_start():
    os.setsid()
    # PROTECT AGAINST FILE DESCRIPTOR 0
    try:
        os.close(0)
    except OSError:  # was not open
        pass


class Action(object):
    CHECK_FINISHED_PERIOD_INITIAL = 0.0001
    
    @staticmethod
    def get_new_id():
        global id_action
        id_action += 1
        return id_action
    
    
    def __init__(self, _id, command, command_name, timeout, module_type, timeout_from='global'):
        if _id is None:
            self.id = self.get_new_id()
        else:
            self.id = _id
        self.env = []
        self.module_type = module_type
        self.shell_execution = False
        
        self.command = command
        self.command_name = command_name
        
        self.check_interval = 0
        self.status = ''
        self.check_time = -1
        self.average_cpu_time = 0
        
        self.output = ''
        self.perf_data = ''
        self.long_output = ''
        
        self.exit_status = -1
        self.execution_time = -1
        self.u_time = 0.0
        self.s_time = 0.0
        self.timeout = timeout
        self.timeout_from = timeout_from
        
        self._last_check_finished = -1  # last_poll -> _last_check_finished
        self.check_finished_period = Action.CHECK_FINISHED_PERIOD_INITIAL  # wait_time -> check_finished_period
        
        self.worker_id = -1  # worker id is choose at launch
        self.executor_id = 'none'  # executor id is choose at launch
        
        self._local_env = None
        self._stdoutdata = ''
        self._stderrdata = ''
        self._process = None
        
        # logger.debug('[action][%s] New action [%s]' % (self.id, self.command_name))
    
    
    # Ok when we load a previous created element, we should
    # not start at 0 for new object, so we must raise the Action.id
    # if need
    @staticmethod
    def assume_at_least_id(_id):
        global id_action
        id_action = max(id_action, _id)
    
    
    @staticmethod
    def __windows_set_pipe_no_wait(pipefd):
        h = msvcrt.get_osfhandle(pipefd)
        
        res = windll.kernel32.SetNamedPipeHandleState(h, byref(PIPE_NOWAIT), None, None)
        if res == 0:
            logger.error('Cannot set the command execution to not blocking. Please join the support. %s' % WinError())
    
    
    @staticmethod
    def __unix_set_pipe_no_wait(pipefd):
        fl = fcntl.fcntl(pipefd, fcntl.F_GETFL)
        fcntl.fcntl(pipefd, fcntl.F_SETFL, fl | os.O_NONBLOCK)
    
    
    # Try to read a fd in a non blocking mode
    @staticmethod
    def no_block_read(output):
        fd = output.fileno()
        if os.name == 'nt':
            # Important on windows: cannot block on the output.read() so
            # we will have to work wot hthe file descriptor directly
            # and loop until we void the full buffer
            Action.__windows_set_pipe_no_wait(fd)
            
            all_buffers = []
            while True:  # will be break when the buffer will be empty so we don't block the command execution
                try:
                    buf = os.read(fd, 1024)
                    all_buffers.append(buf)
                    if not buf:  # there is no more in the buffer too
                        return ''.join(all_buffers)
                except Exception as exp:
                    last_error = GetLastError()
                    if last_error == ERROR_NO_DATA:  # ok we did void it
                        return ''.join(all_buffers)
                    logger.error('Cannot read the command execution result: %s / %s. Please contact the support' % (last_error, exp))
                    return ''
        else:  # All unix, use fcntl for non blocking read
            Action.__unix_set_pipe_no_wait(fd)
            try:
                return output.read()  # will void the whole buffer
            except:  # no more in the buffer
                return ''
    
    
    def get_hash(self):
        # overload in notification and eventhandler
        if self.command:
            cmd = self.command
            if not isinstance(cmd, str):
                cmd = cmd.encode('utf8', 'ignore')
            return hashlib.md5(cmd).hexdigest()
        else:
            return None
    
    
    # Give the period of the action stats time to live.
    # NOTE: in minutes
    def get_saving_period(self):
        _saving_period = self.check_interval * 15 if self.check_interval != 0 else 60
        return _saving_period
    
    
    def get_local_environnement(self):
        """
        Mix the env and the environment variables into a new local
        env dict.

        Note: We cannot just update the global os.environ because this
        would effect all other checks.
        """
        # Do not use copy.copy() here, as the resulting copy still
        # changes the real environment (it is still a os._Environment
        # instance).
        local_env = os.environ.copy()
        for p in self.env:
            local_env[p] = self.env[p].encode('utf8')
            
        # We do not want that libs used for daemons be used for plugins
        if 'LD_PRELOAD' in local_env:
            del local_env['LD_PRELOAD']
            
        return local_env
    
    
    def execute(self):
        self.status = 'launched'
        self.check_time = time.time()
        self._last_check_finished = self.check_time
        
        self._local_env = self.get_local_environnement()
        
        # logger.debug('[action][%s] Action execute %s' % (self.id, self.status))
        
        # Initialize stdout and stderr.
        # we will read them in small parts if the fcntl is available
        
        ## OS specific part
        if os.name == 'nt':
            self._execute_nt()
        else:
            self._execute_unix()
    
    
    # Give a copy of self with only attributes listed in 'SHELL_PROPERTIES'
    def minimal_copy_for_exec(self, new_i):
        for prop in SHELL_PROPERTIES:
            setattr(new_i, prop, getattr(self, prop, None))
        return new_i
    
    
    def check_finished(self, max_plugins_output_length):
        # logger.debug('[action][%s] check_finished %s %s %s' % (self.id, self.status, (time.time() - self.check_time), self.timeout))
        
        now = time.time()
        if now - self._last_check_finished < self.check_finished_period:
            # We don't want make self._process.poll() each 0.1s if the _process take over time.
            # So if the _process don't finish in check_finished_period we will check in check_finished_period * 2.
            return
        
        self._last_check_finished = now
        _, _, child_utime, child_stime, _ = os.times()
        
        if self._process.poll() is None:
            self.check_finished_period = min(self.check_finished_period * 2, 0.1)
            # logger.debug("[action][%s] commande wasn't launch" % self.id)
            
            self._stdoutdata += Action.no_block_read(self._process.stdout)
            self._stderrdata += Action.no_block_read(self._process.stderr)
            
            if (now - self.check_time) > self.timeout:
                ## OS specific part
                if os.name == 'nt':
                    self._kill_nt()
                else:
                    self._kill_unix()
                
                # logger.debug("[action][%s] The action [%s] wasn't execute in %s second so we kill it." % (self.id, self.command_name, self.timeout))
                self.set_exit_status(EXIT_STATUS_FAIL, 'timeout', now)
                
                # Get the user and system time
                _, _, n_child_utime, n_child_stime, _ = os.times()
                self.u_time = n_child_utime - child_utime
                self.s_time = n_child_stime - child_stime
                self.u_time = max(self.u_time, MINIMAL_EXECUTION_TIME)
        else:
            # The command was to quick and finished even before we can polled it first.
            # So finish the read.
            self._stdoutdata += Action.no_block_read(self._process.stdout)
            self._stderrdata += Action.no_block_read(self._process.stderr)
            
            exit_status = self._process.returncode
            
            # if the exit status is abnormal, we add stderr to the output
            self._stdoutdata = self._stdoutdata + self._stderrdata
            if ('sh: -c: line 0: unexpected EOF while looking for matching' in self._stderrdata
                    or ('sh: -c:' in self._stderrdata and ': Syntax' in self._stderrdata)
                    or 'sh: Syntax error: Unterminated quoted string' in self._stderrdata):
                self._stdoutdata = self._stdoutdata + self._stderrdata
                exit_status = EXIT_STATUS_FAIL
            # Now grep what we want in the output
            self.parse_stdout_and_stderr(self._stdoutdata, max_plugins_output_length)
            
            self.set_exit_status(exit_status, 'done', time.time())
            
            # Also get the system and user times
            _, _, n_child_utime, n_child_stime, _ = os.times()
            self.u_time = n_child_utime - child_utime
            self.s_time = n_child_stime - child_stime
            self.u_time = max(self.u_time, MINIMAL_EXECUTION_TIME)
            
            # logger.debug("[action][%s] Commande done on [%.3f]s (u_t:[%.3f],s_t:[%.3f]) output [%s] long_output [%s]" % (self.id, self.execution_time, self.u_time, self.s_time, self.output, self.long_output))
    
    
    def set_exit_status(self, status_code, status_text, curr_time):
        self.exit_status = status_code
        self.status = status_text
        self.execution_time = curr_time - self.check_time
        
        # We can clean the useless properties now
        del self._stdoutdata
        del self._stderrdata
        del self._process
        del self._local_env
    
    
    def is_finish(self):
        # logger.debug('[action][%s] is_finish %s %s %s' % (self.id, self.status, (time.time() - self.check_time), self.timeout))
        return self.status in ('done', 'timeout')
    
    
    def get_cpu_time(self):
        if self.execution_time == -1:
            raise Exception("Action[%s] didn't run. You can't ask get_cpu_time now." % self.id)
        
        if os.name == 'nt':
            # TODO to get the u_time on windows we need to get the PID of the launch commande
            return 0
            # return self.execution_time
        else:
            return self.u_time
    
    
    def parse_stdout_and_stderr(self, out, max_plugins_output_length):
        # logger.debug("[action][%s] parse_stdout to get %s chars" % (self.id, max_plugins_output_length))
        # Squeeze all output after max_plugins_output_length
        out = out[:max_plugins_output_length]
        # manage escaped pipes
        out = out.replace('\|', '___PROTECT_PIPE___')
        # Then cuts by lines
        elts = out.split('\n')
        # For perf data
        elts_line1 = elts[0].split('|')
        # First line before | is output, and strip it
        self.output = elts_line1[0].strip().replace('___PROTECT_PIPE___', '|')
        # Init perfdata as void
        self.perf_data = ''
        # After | is perfdata, and strip it
        if len(elts_line1) > 1:
            self.perf_data = elts_line1[1].strip().replace('___PROTECT_PIPE___', '|')
        # Now manage others lines. Before the | it's long_output
        # And after it's all perf_data, \n join
        long_output = []
        in_perfdata = False
        for line in elts[1:]:
            # if already in perfdata, direct append
            if in_perfdata:
                self.perf_data += ' ' + line.strip().replace('___PROTECT_PIPE___', '|')
            else:  # not already in? search for the | part :)
                elts = line.split('|', 1)
                # The first part will always be long_output
                long_output.append(elts[0].strip().replace('___PROTECT_PIPE___', '|'))
                if len(elts) > 1:
                    in_perfdata = True
                    self.perf_data += ' ' + elts[1].strip().replace('___PROTECT_PIPE___', '|')
        # long_output is all non output and perfline, join with \n
        self.long_output = '\n'.join(long_output).strip()
        
        # Force strings to be unicode from here
        _encoding = 'utf8'
        # windows encoding is special, it's not the windows-1252 or latin-1 that is used in a shell execution, but cp850
        # came from: sys.stdout.encoding run under windows
        if os.name == 'nt':
            _encoding = 'cp850'
        if isinstance(self.output, str):
            self.output = self.output.decode(_encoding, 'ignore')
        if isinstance(self.long_output, str):
            self.long_output = self.long_output.decode(_encoding, 'ignore')
        if isinstance(self.perf_data, str):
            self.perf_data = self.perf_data.decode(_encoding, 'ignore')
    
    
    def _get_cmd(self):
        # 2.7 and higher Python version need a list of args for cmd and if not force shell
        # (if, it's useless, even dangerous)
        # 2.4->2.6 accept just the string command
        
        # logger.info("[action][%s] run in shell [%s] command [%s]" % (self.id, self.shell_execution, self.command_name))
        
        if self.shell_execution:
            cmd = self.command.encode('utf8', 'ignore')
        else:
            try:
                cmd = shlex.split(self.command.encode('utf8', 'ignore'))
            except Exception, exp:
                self.output = 'Not a valid shell command: ' + exp.__str__()
                
                # logger.debug("[action][%s] Not a valid shell command: %s" % (self.id, exp.__str__()))
                raise exp
        return cmd
    
    
    def get_return_from(self, a, lang='en', run_from='poller'):
        # force a valid lang value
        if lang not in ['fr', 'en']:
            lang = 'en'
        if a.status == 'timeout':
            if lang == 'fr':
                cfg_from = {'global': 'la configuration globale', 'host': u'la configuration de l\'hôte', 'check': 'la configuration du check', 'command': 'la configuration de la commande'}
                self.output = u"%s [ %s ] :<br>Le check a été interrompu parce qu'il a dépassé le temps d'exécution maximum. (limite: %i secondes définie par %s)" % (run_from.upper(), self.executor_id, self.timeout, cfg_from[self.timeout_from])
            else:
                cfg_from = {'global': 'the global configuration', 'host': 'the host configuration', 'check': 'the check configuration', 'command': 'the command configuration'}
                self.output = "%s [ %s ] :<br>The check has been killed because it's running time was bigger than the maximum timeout (limit: %i seconds defined by %s)" % (run_from.upper(), self.executor_id, self.timeout, cfg_from[self.timeout_from])
            self.long_output = ''
        else:
            self.output = a.output
            self.long_output = a.long_output
        
        # If we have a missing plugin, translate the error message
        if '__MISSING_PLUGIN__' in self.output:
            plugin = self.output.replace('__MISSING_PLUGIN__', '')
            txt = {'fr': r'Nous ne trouvons pas la sonde [%s] sur le %s', 'en': r'We cannot find the plugin [%s] on the %s'}
            self.output = txt.get(lang, '(Unknown lang parameter)We cannot find the plugin [%s] on the %s') % (plugin, run_from)
        elif '__BAD_QUOTATION__' == self.output:
            txt = {'fr': u"Erreur de syntaxe: votre commande et/ou ses paramètres ont un souci dans la fermeture des arguments pour l'éxécution sur le système. (un caractère \" est en trop ou bien non fermé). Le système refuse de l'éxécuter.",
                   'en': u"Syntaxe error: your command and/or argument definitions have an error in the quotation close (a string \" is not closed). The system refuse to execute it."}
            self.output = txt.get(lang, '(Unknown lang parameter)%s' % txt.get('en'))
        self.exit_status = a.exit_status
        self.check_time = a.check_time
        self.execution_time = a.execution_time
        self.perf_data = a.perf_data
        self.u_time = a.u_time
        self.s_time = a.s_time
    
    
    # -------- Windows OS
    def _execute_nt(self):
        # logger.debug("[action][%s] Execute commande %s force_shell %s" % (self.id, self.command_name, force_shell))
        
        self.shell_execution = True
        
        try:
            cmd = self._get_cmd()
            
            self._process = subprocess.Popen(
                cmd,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                env=self._local_env,
                shell=True)
        except WindowsError, exp:
            self.set_exit_status(EXIT_STATUS_FAIL, 'timeout', time.time())
            logger.info("[action][%s] We kill the process: %s %s" % (self.id, exp, self.command_name))
        except Exception as e:
            self.set_exit_status(EXIT_STATUS_FAIL, 'timeout', time.time())
            logger.info("[action][%s] We kill the process: [%s] [%s]" % (self.id, self.command_name, e))
    
    
    def _kill_nt(self):
        TerminateProcess(int(self._process._handle), -1)
    
    
    # -------- Unix OS
    def _execute_unix(self):
        
        # The preexec_fn set to give sons a same process group.
        # It is also used to set the proctitle to hide passwords
        # See http://www.doughellmann.com/PyMOTW/subprocess/ for detail about this.
        try:
            cmd = self._get_cmd()
            # cmd = '/tmp/random.sh' # <-- pour test
            # The lib obfuscate_cmdline.c is used to obfuscate the command line for the checks, in order to hide
            # protected fields from commands like "top" or "ps"
            # It works by replacing the argv[] array by a blank zone of memory, after having copied it elsewhere
            # to be able to call main() with the real arguments.
            # COMMAND_NAME is an environment variable which will be used by the lib to replace argv[0] with
            # This is a C program to be compiled using the following command ; the poller expects
            # to find it in /var/lib/shinken/libexec/obfuscate_cmdline.so
            
            # gcc -Wall -O2 -fpic -shared -ldl -o obfuscate_cmdline.so private_libs/shinkensolutions/obfuscate_cmdline/obfsucate_cmdline.c
            # LD_PRELOAD=/.../obfuscate_cmdline.so theprogram theargs...
            
            # Original source code taken from: https://unix.stackexchange.com/a/403918/119298
            
            # JEAN: 9 juillet 2018: je n'arrive pas a fixer le soucis de obfuscate_cmdline quand il ets utilise par perl+optargs
            # donc pour l'instant on desactive cette modification.
            # self._local_env["LD_PRELOAD"] = "/var/lib/shinken/libexec/obfuscate_cmdline.so"
            # self._local_env['COMMAND_NAME'] = self.command_name.encode("utf8", "ignore")
            
            # NOTE: the close_fds means that the sub process won't have any of our file descriptors, like socket and logs
            #       that's a good idea for security, but it have a cost: 3.5% of more CPU usage (both nice and sys)
            
            self._process = subprocess.Popen(
                cmd,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                close_fds=True,
                shell=self.shell_execution,
                env=self._local_env,
                preexec_fn=_at_unix_sub_process_start
            )
        except OSError as exp:
            self.set_exit_status(EXIT_STATUS_FAIL, 'done', time.time())
            logger.error("[action][%s] Fail launching command: %s %s %s" % (self.id, self.command_name, exp, self.shell_execution))
            # Maybe it's just a shell we try to exec. But not allowed without shell_execution
            if exp.errno == 2:
                self.output = '__MISSING_PLUGIN__%s' % (self.command.split()[0])
            else:
                self.output = exp.__str__()
            
            # Maybe we run out of file descriptor. It's not good at all!
            if exp.errno == 24 and exp.strerror == 'Too many open files':
                raise TooManyOpenFiles('toomanyopenfiles')
                
                # logger.debug("[action][%s] Execute commande [%s] force_shell [%s] pid [%s]" % (self.id, self.command_name, force_shell, self._process.pid))
        except ValueError as e:
            self.set_exit_status(EXIT_STATUS_FAIL, 'done', time.time())
            if str(e) == 'No closing quotation':
                self.output = '__BAD_QUOTATION__'
            else:
                self.output = 'ERROR: The command execution returns an error: %s.' % str(e)
        except Exception as e:
            self.set_exit_status(EXIT_STATUS_FAIL, 'timeout', time.time())
            logger.info("[action][%s] We kill the process: [%s] [%s][%s]" % (self.id, self.command_name, e, type(e)))
    
    
    def _kill_unix(self):
        pid = self._process.pid
        # We kill a process group because we launched them with preexec_fn=os.setsid
        # so we can launch a whole kill tree instead of just the first one
        os.killpg(pid, signal.SIGKILL)
        # Try to force close the descriptors, because python seems to have problems with them
        for fd in [self._process.stdout, self._process.stderr]:
            try:
                fd.close()
            except:
                pass
        # Wait so we do not have a zombie process
        try:
            os.waitpid(pid, 0)
        except:
            pass
