#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


from .log import logger
from .withworkersandinventorymodule import WithWorkersAndInventoryModule

HOST_SERVICE_BROKS = ('initial_host_status', 'update_host_status', 'host_check_result', 'host_next_schedule',
                      'initial_service_status', 'update_service_status', 'service_check_result', 'service_next_schedule')


class WorkerBasedBrokerModule(WithWorkersAndInventoryModule):
    
    def __init__(self, mod_conf):
        super(WorkerBasedBrokerModule, self).__init__(mod_conf)
    
    
    # Get raw stats value, give last minute results
    def get_raw_stats(self, param=''):
        data = super(WorkerBasedBrokerModule, self).get_raw_stats(param=param)
        for worker_id, worker in self._workers.iteritems():
            alive = worker['process'].is_alive()
            qsize = worker['instance']._get_main_process_to_worker_broks_queue_size()
            inventories = worker['inventories']
            nb_items = 0
            nb_hosts = 0
            nb_clusters = 0
            nb_checks = 0
            for inventory in inventories.values():
                nb_items += (inventory.get_len() if inventory else 0)
                hosts = inventory.get_all_hosts()
                for host, host_summary in hosts.iteritems():
                    nb_checks += len(host_summary.get_checks())
                    if host_summary.is_cluster():
                        nb_clusters += 1
                    else:
                        nb_hosts += 1
            
            worker_info = {
                'alive'                            : alive,
                'main_process_to_worker_queue_size': qsize,
                'number_of_managed_items'          : nb_items,
                'number_of_managed_hosts'          : nb_hosts,
                'number_of_managed_clusters'       : nb_clusters,
                'number_of_managed_checks'         : nb_checks
            }
            if data['workers'][worker_id]:
                data['workers'][worker_id].update(worker_info)
            else:
                data['workers'][worker_id] = worker_info
        return data
    
    
    def _send_brok_to_worker(self, worker_id, brok):
        worker = self._workers[worker_id]['instance']
        worker._send_brok_to_worker(brok)
    
    
    def _send_brok_to_all_workers(self, brok):
        for worker_id, worker in self._workers.iteritems():
            self._send_brok_to_worker(worker_id, brok)
    
    
    # For broks we have want:
    # * All hosts & checks broks => only in the worker that manage the element
    # * others broks: every one
    def manage_brok(self, brok):
        brok_type = brok.type
        
        if not self.want_brok(brok):
            return
        
        super(WorkerBasedBrokerModule, self).manage_brok(brok)
        
        # if it's not a host/service brok, send to everyone
        if brok_type not in HOST_SERVICE_BROKS:
            self._send_brok_to_all_workers(brok)
            return
        
        # Ok it's a host/service brok, so only one worker should be selected
        host_uuid = brok.data['host_uuid']
        if not host_uuid:  # no host can means a host brok, so directly look at instance_uuid
            host_uuid = brok.data['instance_uuid']
        
        # v02.07.04 (jean) = protection to detect the root cause of the malformed host_uuid here
        # jean: I suppose it's because of bad retention=True on uuids parameters, but protect against a (future) regression
        if '-' in host_uuid:
            logger.warning('[%s] [INVENTORY] There are still a malformed host_uuid : %s, we will fix it but please report to your support.' % (self.get_name(), host_uuid))
            host_uuid = host_uuid.split('-', 1)[0]
        
        worker_id = self._host_uuids_to_worker_cache.get(host_uuid, None)
        if worker_id is not None:
            self._send_brok_to_worker(worker_id, brok)
            return
        
        realm = brok.data['realm']
        
        if not host_uuid:
            logger.error('[%s] [INVENTORY] Cannot find valid uuid for %s : %s' % (self.get_name(), brok.type, brok.data))
            return
        
        for (worker_id, worker_entry) in self._workers.iteritems():
            inventory = worker_entry['inventories'].get(realm, None)  # maybe the realm is not know currently
            if inventory is None:
                logger.error('[%s] [INVENTORY] CANNOT FIND INVENTORY FOR %s' % (self.get_name(), brok_type))
                continue
            
            if inventory.have_host_uuid(host_uuid):
                self._send_brok_to_worker(worker_id, brok)
                # set the cache
                self._host_uuids_to_worker_cache[host_uuid] = worker_id
                return
        
        logger.error('[%s] [INVENTORY] We cannot find a valid worker that manage the host %s' % (self.get_name(), host_uuid))
    
    
    # Every second we propose to the worker to have a tick call if they want to (like send in queue broks)
    def hook_tick(self, daemon):
        for worker_entry in self._workers.values():
            worker = worker_entry['instance']
            worker.in_main_process_tick()
