#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import os
import time

from shinken.action import Action
from shinken.property import BoolProp, IntegerProp, FloatProp, StringProp
from shinken.log import logger

NO_END_VALIDITY = -1
MONITORING_CHECK_CONSUME_DEBUG_FLAG = os.environ.get('MONITORING_CHECK_CONSUME_DEBUG_FLAG', '0') == '1'


class CHECK_CAUSE:
    SCHEDULE = 'schedule'
    FORCE = 'force'
    RETRY = 'retry'
    DEPENDENCY = 'dependency'


# IMPORTANT : if you change the values, SANATIZE the retention data!
# IMPORTANT2: if you are adding a new state, FILL THE RETENTION DATA COMMENT about
#        what you are doing with this state. If you don't know, don't commit!
class CHECK_STATUS:
    SCHEDULED = 'scheduled'
    INPOLLER = 'inpoller'
    TIMEOUT = 'timeout'
    WAITCONSUME = 'waitconsume'
    WAITDEP = 'waitdep'
    HAVETORESOLVEDEP = 'havetoresolvedep'
    ALL_DEP_ARE_FINISH = 'all_dep_are_finish'
    MUST_RECREATE_DEP = 'must_recreate_dep'
    ZOMBIE = 'zombie'


class Check(Action):
    # AutoSlots create the __slots__ with properties and
    # running_properties names
    
    ### DO NOT ENABLE AUTOSLOTS, we need the __dict__
    # __metaclass__ = AutoSlots
    
    my_type = 'check'
    
    ############ WARNING: DO NOT ADD a property without look at retention part!
    # if you don't know what it means, ask Jean and don't commit.
    properties = {
        'is_a'                                 : StringProp(default='check'),
        'type'                                 : StringProp(default=''),
        '_in_timeout'                          : BoolProp(default=False),
        'status'                               : StringProp(default=''),
        'exit_status'                          : IntegerProp(default=3),
        'state'                                : IntegerProp(default=0),
        'output'                               : StringProp(default=''),
        'long_output'                          : StringProp(default=''),
        'ref'                                  : IntegerProp(default=-1),
        't_to_go'                              : IntegerProp(default=0),
        'original_t_to_go'                     : IntegerProp(default=0),
        'depend_on'                            : StringProp(default=[]),
        'dep_check'                            : StringProp(default=[]),
        'check_time'                           : IntegerProp(default=0),
        'execution_time'                       : FloatProp(default=0.0),
        'u_time'                               : FloatProp(default=0.0),
        's_time'                               : FloatProp(default=0.0),
        'perf_data'                            : StringProp(default=''),
        'check_type'                           : IntegerProp(default=0),
        'poller_tag'                           : StringProp(default='None'),
        'reactionner_tag'                      : StringProp(default='None'),
        'env'                                  : StringProp(default={}),
        'internal'                             : BoolProp(default=False),
        'module_type'                          : StringProp(default='fork'),
        'executor_id'                          : StringProp(default='none'),
        'from_trigger'                         : BoolProp(default=False),
        'check_interval'                       : IntegerProp(default=0),
        'shell_execution'                      : BoolProp(default=0),
        'cause'                                : StringProp(default=CHECK_CAUSE.SCHEDULE),
        'state_validity_period'                : IntegerProp(default=None),
        'state_validity_period_when_go_in_soft': IntegerProp(default=None),
        'warning_threshold_cpu_usage': IntegerProp(default=-1),
        'timeout'                    : IntegerProp(default=10),
        'is_dummy_check'             : BoolProp(default=False),
    }
    
    # theses fields have links to other objects, so we CANNOT serialize/retention them
    # and they must be recreated
    properties_to_remove_for_retention = ('ref', 'depend_on', 'depend_on_me')
    
    
    def __init__(self, status, command, ref, t_to_go,
                 depend_on_me=None,
                 id=None,
                 timeout=10,
                 timeout_from=None,
                 poller_tag='None',
                 reactionner_tag='None',
                 env={},
                 module_type='fork',
                 from_trigger=False,
                 dependency_check=False,
                 shell_execution=False,
                 command_name='MISSING_NAME',
                 cause=CHECK_CAUSE.SCHEDULE,
                 warning_threshold_cpu_usage=-1,
                 is_dummy_check=False,
                 ):
        # print "IN CHECK id", id
        super(Check, self).__init__(id, command, command_name, timeout, module_type, timeout_from=timeout_from)
        self.is_a = 'check'
        self.type = ''
        self.cause = cause
        
        self.warning_threshold_cpu_usage = warning_threshold_cpu_usage
        self.check_interval = getattr(ref, 'check_interval', 0)
        self._in_timeout = False
        self.timeout = timeout
        self.status = status
        self.exit_status = 3
        self.command = command
        self.output = ''
        self.long_output = ''
        self.ref = ref
        # self.ref_type = ref_type
        self.t_to_go = t_to_go
        # remember the first t_to_go, as with orphan management we can change t_to_go in the future
        self.original_t_to_go = self.t_to_go
        self.depend_on = []  # Father check
        if depend_on_me is None:
            self.depend_on_me = []
        else:
            self.depend_on_me = [depend_on_me]
        self.check_time = 0
        self.execution_time = 0
        self.u_time = 0  # user execution time
        self.s_time = 0  # system execution time
        self.perf_data = ''
        self.check_type = 0  # which kind of check result? 0=active 1=passive
        self.poller_tag = poller_tag
        self.reactionner_tag = reactionner_tag
        self.module_type = module_type
        self.env = env
        
        # If it's a business rule, manage it as a special check
        if ref and ref.got_business_rule or command.startswith('_internal'):
            self.internal = True
        else:
            self.internal = False
        self.from_trigger = from_trigger
        self.dependency_check = dependency_check
        self.shell_execution = shell_execution
        self.state_validity_period = None
        self.state_validity_period_when_go_in_soft = None
        
        self._result_modulations_are_applied = False  # do not apply result modulation twice
        self.is_dummy_check = is_dummy_check
        
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            if ref:
                logger.info('[monitoring] Creating a check for %s (check id=%s)' % (self.ref.get_full_name(), self.id))
            else:
                logger.info('[monitoring] Creating a dummy check from check id=%s' % (self.id))
        
        ############ WARNING: DO NOT ADD a property without look at retention part!
        # if you don't know what it means, ask Jean and don't commit.
    
    
    # Call by pickle when load retention
    def __setstate__(self, state):
        for prop_name, value in state.iteritems():
            if prop_name == 'dep_check':
                prop_name = 'depend_on_me'
            setattr(self, prop_name, value)
        
        if not hasattr(self, 'state_validity_period'):
            self.state_validity_period = 5 * 60
        if not hasattr(self, 'state_validity_period_when_go_in_soft'):
            self.state_validity_period_when_go_in_soft = 5 * 60
    
    
    def set_cause(self, cause):
        self.cause = cause
    
    
    def copy_shell(self):
        """return a copy of the check but just what is important for execution
        So we remove the ref and all
        """
        # We create a dummy check with nothing in it, just defaults values
        dummy_check = Check('', '', None, '', id=self.id, shell_execution=self.shell_execution, command_name=self.command_name, cause=self.cause, is_dummy_check=True)
        return self.minimal_copy_for_exec(dummy_check)
    
    
    def is_launchable(self, t):
        return t > self.t_to_go
    
    
    # late means: more than 10s without being get by a poller
    def is_late(self):
        return time.time() > self.original_t_to_go + 10
    
    
    def __str__(self):
        return "Check %d status:%s command:%s exit_status:%s t_to_go:%s ref:%s" % \
               (self.id, self.status, self.command.strip(), self.exit_status, self.t_to_go, self.ref)
    
    
    def get_id(self):
        return self.id
    
    
    def set_type_passive(self):
        self.check_type = 1
    
    
    def is_dependent(self):
        return self.dependency_check
    
    
    # Look if all output* strings are in unicode-utf8
    def assert_unicode_strings(self):
        # if str, go in unicode
        properties = ('output', 'long_output', 'perf_data')
        for prop in properties:
            value = getattr(self, prop)
            if isinstance(value, str):
                new_value = value.decode('utf8', 'ignore')
                setattr(self, prop, new_value)
    
    
    def apply_result_modulations(self, result_modulations):
        if self._result_modulations_are_applied:
            return
        
        self._result_modulations_are_applied = True  # do not apply result twice
        # Before setting state, modulate them
        for rm in result_modulations:
            if rm is not None:
                module_return = rm.module_return(self.exit_status, self.output, self.long_output)
                if module_return != self.exit_status:
                    self.exit_status = module_return
                    break
    
    
    # A check can be consumed only if:
    # * it's a waitconsume (obvious)
    # * it's a waitdep that all dependant checks are done (so depend_on is now void)
    def can_be_consume(self):
        return self.status == CHECK_STATUS.WAITCONSUME or self.status == CHECK_STATUS.ALL_DEP_ARE_FINISH
    
    
    def launch_consume(self):
        item = self.ref
        item.consume_result(self)
    
    
    # We need to launch dependency checks if:
    # * we are in a bad state
    # * we are not already in a waiting dependency phase
    def need_to_launch_dependency_checks(self):
        return self.exit_status != 0 and self.status == CHECK_STATUS.WAITCONSUME
    
    
    # if another check is asking for a dependency, maybe we are a valid check that can be executed
    # SCHEDULED => OK, change the time to go now
    # INPOLLER => already exiting, just register the check as a son
    # TIMEOUT/WAITCONSUME => just in time!  just register the check as a son
    # WAITDEP => no problem, will be consume when all dep wil be done, so can register as a son, don't have to modify time
    # HAVETORESOLVEDEP => transitient state, cannot be here
    # ALL_DEP_ARE_FINISH => just in time! just register the check as a son
    # MUST_RECREATE_DEP => possible, just register the check, don't have to modify time
    # ZOMBIE => NO! will die
    def can_be_hot_hook_for_dependency(self):
        return self.status in (CHECK_STATUS.SCHEDULED, CHECK_STATUS.INPOLLER, CHECK_STATUS.TIMEOUT,
                               CHECK_STATUS.WAITCONSUME, CHECK_STATUS.WAITDEP, CHECK_STATUS.HAVETORESOLVEDEP,
                               CHECK_STATUS.ALL_DEP_ARE_FINISH, CHECK_STATUS.MUST_RECREATE_DEP)
    
    
    # Someone is asking us to go ASAP, and as the scheduler have a
    # time indexed list of checks, we must give back our old time
    # so the scheduler will be able to find the old index
    def force_time_change(self, new_time):
        old_time = self.t_to_go
        self.t_to_go = new_time
        self.original_t_to_go = new_time
        self.set_cause(CHECK_CAUSE.FORCE)
        
        return old_time
    
    
    def hot_hook_for_dependency(self, son_check):
        self.depend_on_me.append(son_check)
        new_time = None
        old_time = None
        if self.status == CHECK_STATUS.SCHEDULED:
            new_time = int(time.time())
            old_time = self.force_time_change(new_time)
        return old_time, new_time
    
    
    def register_dependency_checks(self, checks_ids):
        self.status = CHECK_STATUS.WAITDEP
        for check_id in checks_ids:
            self.depend_on.append(check_id)
            # Ok, no more need because checks are not take by host/service, and not returned
    
    
    # We did have a father check that just finish, remove it from our list
    def remove_dependency_check(self, dependency_check_id):
        if self.status != CHECK_STATUS.WAITDEP:
            logger.error('ERROR: trying to remove a dependency on a check %s that is not waiting for dependency (current state=%s).' % (dependency_check_id, self.status))
            return
        if dependency_check_id in self.depend_on:
            self.depend_on.remove(dependency_check_id)
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            logger.info("[monitoring] The check id=%s on %s is removing the father check %s and now still have %s father checks" % (self.id, self.ref.get_full_name(), dependency_check_id, len(self.depend_on)))
        # No more deps? congrats, we will be able to be consumed
        if len(self.depend_on) == 0:
            self.status = CHECK_STATUS.ALL_DEP_ARE_FINISH
            if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                logger.info("[monitoring] The check id=%s on %s do not have any more dependency. Going to is state ALL_DEP_ARE_FINISH." % (self.id, self.ref.get_full_name()))
    
    
    # Now our host/service did fully consume us, we need to :
    # * go zombie if no-one is waiting for us
    # * go havetoresolvedep is son checks are waiting for us (so they can be unlock)
    #   -> so we are removing ourselve from them, and we go zombie
    def set_fully_consumed(self):
        # we must warn our sons that we did finish
        if len(self.depend_on_me) != 0:
            my_id = self.id
            self.status = CHECK_STATUS.HAVETORESOLVEDEP  # yes it won't last, but it's better for reader to understand
            for dependent_checks in self.depend_on_me:
                # Ok, now dependent will no more wait c
                dependent_checks.remove_dependency_check(my_id)
        # In all case we did finish this check, go zombie
        # REMOVE OLD DEP CHECK -> zombie
        self.status = CHECK_STATUS.ZOMBIE
    
    
    def _get_core_object_for_retention(self):
        new_object = self.copy_shell()  # already manage creator call
        # now copy interesting property, but not the one we know we cannot serialize
        for prop, value in self.__dict__.iteritems():
            if prop in self.properties_to_remove_for_retention:
                continue
            setattr(new_object, prop, value)
        
        # Be sure the problematic properties are removed
        for prop in self.properties_to_remove_for_retention:
            delattr(new_object, prop)
        return new_object
    
    
    ##### Retention part:
    # For each state, what we are doing for retention:
    # => in all cases, remove the 'ref', we are not able to serialize it
    #     SCHEDULED => drop, we are able to recreate it
    #     INPOLLER => drop, we will recreate it, poller will give us back, but it's not a problem, it will be droped
    #     TIMEOUT/WAITCONSUME => just returned, can be consumed but remove dependency if there are (son will recreate it anyway)
    #     WAITDEP  => we did consume it once, so cannot be lost. Must be reconsume but:
    #                  - do not take output/result from it (already done)
    #                  - recreate the depency (like we did for go in WAITDEP)
    #                  => so we are setting it to a new state "MUST_RECREATE_DEP"
    #     ALL_DEP_ARE_FINISH => => we did consume it once, so cannot be lost. Must be reconsume but:
    #                  - do not take output/result from it (already done)
    #     MUST_RECREATE_DEP => can be in retention if the daemon is stoped very just before the first turn, so nothing to do
    #     HAVETORESOLVEDEP => transitient case (just for workflow to be clear in a draft), cannot be access in retention
    #     ZOMBIE => just drop, we already did finish with it
    def get_object_for_retention(self):
        # First look at state, because lot of them just return nothing
        to_drop_status = (CHECK_STATUS.SCHEDULED, CHECK_STATUS.INPOLLER, CHECK_STATUS.ZOMBIE)
        to_set_in_recreate_status = (CHECK_STATUS.WAITDEP, CHECK_STATUS.MUST_RECREATE_DEP)
        impossible_status = (CHECK_STATUS.HAVETORESOLVEDEP,)
        just_drop_dependency = (CHECK_STATUS.WAITCONSUME, CHECK_STATUS.TIMEOUT, CHECK_STATUS.ALL_DEP_ARE_FINISH,)
        
        status = self.status
        if status in to_drop_status:
            return None
        elif status in to_set_in_recreate_status:
            retention_object = self._get_core_object_for_retention()
            # drop all about dependency
            retention_object.depend_on = []
            retention_object.depend_on_me = []
            retention_object.status = CHECK_STATUS.MUST_RECREATE_DEP
            return retention_object
        elif status in just_drop_dependency:
            retention_object = self._get_core_object_for_retention()
            # drop all about dependency
            retention_object.depend_on = []
            retention_object.depend_on_me = []
            # No need to set a new status
            return retention_object
        elif status in impossible_status:
            logger.error('The check state %s should not be present in retention. Please fill a bug with your log and retention data.' % status)
            return None
        else:
            logger.error('The check state %s is not managed. Please fill a bug with your log and retention data.' % status)
            return None
    
    
    # We are at the end of the retention load, we must check:
    # * WAITCONSUME/TIMEOUT/ALL_DEP_ARE_FINISH => nothing to do, we will just consume them
    # * MUST_RECREATE_DEP => ask our ref to recreate dependency checks, and go in WAITDEP status
    #                        but beware: if there are no more dependency, cannot stay in MUST_RECREATE_DEP and go ALL_DEP_ARE_FINISH
    def restore_from_retention(self):
        # Just returned checks are ready to consume, they are already clean from dep in the retention save part
        if self.status == CHECK_STATUS.WAITCONSUME or self.status == CHECK_STATUS.TIMEOUT or self.status == CHECK_STATUS.ALL_DEP_ARE_FINISH:
            if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                logger.info('[retention] [loading] [%s] the check was reload from the retention with state %s so its results can be load.' % (self.ref.get_full_name(), self.status))
            return
        elif self.status == CHECK_STATUS.MUST_RECREATE_DEP:
            are_dep_checks_launched = self.ref.raise_dependencies_check(self)
            if not are_dep_checks_launched:
                self.status = CHECK_STATUS.ALL_DEP_ARE_FINISH
            if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
                logger.info('[retention] [loading] [%s] the check was reload from the retention and ask for check father dependency in order to finish the check return analysis.' % self.ref.get_full_name())
            # protect against bad states here
            if self.status not in (CHECK_STATUS.WAITDEP, CHECK_STATUS.ALL_DEP_ARE_FINISH):
                logger.error('The check state %s was unable to be restored in a valid state. Please fill a bug with your log and retention.' % self.status)
                self.status = CHECK_STATUS.ALL_DEP_ARE_FINISH
            return
        else:
            logger.error('The check status %s is not managed for retention loading. Please fill a bug.' % self.status)
            self.status = CHECK_STATUS.ALL_DEP_ARE_FINISH
            return


# For the retention we are looking at the list of checks of a host/service, and return
# retention object that are ok with the retention status
def get_for_retention_callback(ref, checks):
    if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
        logger.info('[retention][%s] Asking retention for %s' % (ref.get_full_name(), [check.id for check in checks]))
    retention_objects = []
    for check in checks:
        if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
            logger.info('[retention][%s]  - check id=%s status=%s (is dummy=%s)' % (ref.get_full_name(), check.id, check.status, check.is_dummy_check))
        check_for_retention = check.get_object_for_retention()
        if check_for_retention:
            retention_objects.append(check_for_retention)
    if MONITORING_CHECK_CONSUME_DEBUG_FLAG:
        logger.info('[retention][%s] Giving back %s checks for retention' % (ref.get_full_name(), len(retention_objects)))
    return retention_objects
