#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

"""
 This is the class of the dispatcher. Its role is to dispatch
 configurations to other elements like schedulers, reactionner,
 pollers, receivers and brokers. It is responsible for high availability part. If an
 element dies and the element type has a spare, it sends the config of the
 dead one to the spare
"""

import time
import random
import threading
from numbers import Number

from .satellitelink import SatelliteLink
from .log import logger

# Always initialize random :)
random.seed()


# Dispatcher Class
class Dispatcher:
    # Load all elements, set them as not assigned
    # and add them to elements, so loop will be easier :)
    def __init__(self, conf, arbiter, arbiter_trace, configuration_incarnation):
        self.arbiter = arbiter
        self.arbiter_trace = arbiter_trace
        # Pointer to the whole conf
        self.conf = conf
        self.realms = conf.realms
        # Direct pointer to important elements for us
        self.configuration_incarnation = configuration_incarnation
        
        for sat_type in ('arbiters', 'schedulers', 'reactionners', 'brokers', 'receivers', 'pollers'):
            setattr(self, sat_type, getattr(self.conf, sat_type))
            
            # for each satellite, we look if current arbiter have a specific
            # satellitemap value setted for this satellite if so, we give this map to
            # the satellite (used to build satellite URI later)
            if arbiter is None:
                continue
            
            key = sat_type[:-1] + '_name'  # i.e: schedulers -> scheduler_name
            for satellite in getattr(self, sat_type):
                sat_name = getattr(satellite, key)
                satellite.set_arbiter_satellitemap(arbiter.satellitemap.get(sat_name, {}))
        
        # Some flag about dispatch need or not
        self.dispatch_ok = False
        self.first_dispatch_done = False
        
        # Compute for receivers & brokers the list of ALL realms they can be talk about so the receiver/broker
        # configuration will be able to have it
        for realm in self.realms:
            for daemon_type in ('receiver', 'broker'):
                realm_accessible_satellites = realm.get_potential_satellites_by_type(daemon_type)
                for satellite in realm_accessible_satellites:
                    satellite.set_accessible_realm(realm.get_name())
        
        # Prepare the satellites confs
        for satellite in self._get_satellites_but_not_the_schedulers():
            satellite.prepare_for_conf(self.configuration_incarnation)
        
        # Some properties must be given to satellites from global
        # configuration, like the max_plugins_output_length to pollers
        parameters = {'max_plugins_output_length': self.conf.max_plugins_output_length}
        for poller in self.pollers:
            poller.add_global_conf_parameters(parameters)
        # and for all types, give some parameters too
        for s in self._get_satellites_but_not_the_schedulers():
            parameters = {'human_timestamp_log': self.conf.human_timestamp_log}
            s.add_global_conf_parameters(parameters)
        
        # Reset need_conf for all schedulers.
        for s in self.schedulers:
            s.need_conf = True
        
        # We need to prepare the realm structure that keep trace of current satellites assocations (which
        # poller is linked to which shard_id)
        for realm in self.realms:
            realm.reset_all_satellites_associations()
    
    
    def _get_all_shards(self):
        shards = []
        for realm in self.realms:
            for shard in realm.shards.itervalues():
                shards.append(shard)
        return shards
    
    
    def _get_satellites_but_not_the_schedulers(self):
        r = []
        for _type in (self.pollers, self.reactionners, self.brokers, self.receivers):
            for daemon in _type:
                r.append(daemon)
        return r
    
    
    def _get_all_satellites(self):
        r = self._get_satellites_but_not_the_schedulers()
        r.extend(self.schedulers)
        return r
    
    
    def get_satellite_connections(self):
        sats_infos = []
        for sat in self._get_all_satellites():
            sat_info = {}
            sat_info['address'] = sat.address
            sat_info['port'] = sat.port
            sat_info['type'] = sat.my_type
            sat_info['name'] = getattr(sat, sat.my_type + '_name')
            sat_info['proto'] = 'https' if sat.use_ssl else 'http'
            sat_info['uri'] = '%s://%s:%i/' % (sat_info['proto'], sat_info['address'], sat_info['port'])
            sats_infos.append(sat_info)
        
        return sats_infos
    
    
    def _do_one_update_infos(self, elt, arbiter_trace):
        # print "Updating elements", elt.get_name(), elt.__dict__
        elt.update_infos(arbiter_trace)
        # Not alive needs new need_conf and spare too if they do not have already a conf
        # REF: doc/shinken-scheduler-lost.png (1)
        if not elt.alive or hasattr(elt, 'conf') and elt.conf is None:
            elt.need_conf = True
    
    
    # checks alive elements
    def check_alive(self):
        t0 = time.time()
        self.arbiter_trace['arbiter_time'] = time.time()
        _threads = []
        for elt in self._get_all_satellites():
            t = threading.Thread(None, target=self._do_one_update_infos, name='do_one_update_infos', args=(elt, self.arbiter_trace))
            t.daemon = True
            t.start()
            _threads.append(t)
        # Now wait for all to finish before going too far
        for t in _threads:
            t.join()
        
        for arb in self.arbiters:
            # If not me, but not the master too
            if arb != self.arbiter and arb.spare:
                arb.update_infos(self.arbiter_trace)
        logger.debug('[CHECK] time to check if all elements are alive: %.3fs' % (time.time() - t0))
    
    
    def assert_inventories_dispatch(self):
        t0 = time.time()
        
        _threads = []
        # First broker inventory as they can take lot of time
        for realm in self.realms:
            t = threading.Thread(None, target=realm.assert_brokers_inventories, name='assert_brokers_inventories')
            t.daemon = True
            t.start()
            _threads.append(t)
        # Now wait for all to finish before going too far
        for t in _threads:
            t.join()
        
        _threads = []
        # Then receiver inventory as they can take lot of time
        for realm in self.realms:
            t = threading.Thread(None, target=realm.assert_receivers_inventories, name='assert_receivers_inventories')
            t.daemon = True
            t.start()
            _threads.append(t)
        # Now wait for all to finish before going too far
        for t in _threads:
            t.join()
        logger.debug('[CHECK] Time to check (and send all inventories): %.3fs' % (time.time() - t0))
    
    
    def _check_other_arbiters_dispatch(self):
        logger.debug('[DISPATCH] Checking that others arbiters configuration are sync with us')
        # We are a spare, we should not talk to the master
        if self.arbiter.spare:
            logger.info('We are an arbiter spare so we do not force dispatch to the other arbiter.')
            return
        
        self.arbiter_trace['arbiter_time'] = time.time()
        # Check if the other arbiter has a conf, but only if I am a master
        for arb in self.arbiters:
            # If not me and I'm a master
            if arb.alive and arb.reachable and arb != self.arbiter:
                if not arb.have_conf(self.conf.magic_hash):
                    if not hasattr(self.conf, 'whole_conf_pack'):
                        logger.error('CRITICAL: the arbiter try to send a configuration but it is not a MASTER one?? Look at your configuration.')
                        continue
                    
                    conf_to_send = {'full_conf': self.conf.whole_conf_pack, 'arbiter_trace': self.arbiter_trace}
                    arb.put_conf(conf_to_send)
                
                #  I'm the master, just don't run now
                arb.do_not_run()
    
    
    def _check_schedulers_dispatch(self):
        logger.debug('[DISPATCH] Checking that all shards are managed by a scheduler, and by master schedulers before spare ones')
        # We check for confs to be dispatched on alive scheds. If not dispatched, need dispatch :)
        # and if dispatch on a failed node, remove the association, and need a new dispatch
        for realm in self.realms:
            self.dispatch_ok &= realm.check_schedulers_dispatch(self.first_dispatch_done, self.arbiter_trace, self.configuration_incarnation)
    
    
    def _check_satellites_dispatch(self):
        logger.debug('[DISPATCH] Checking that satellites daemons (poller, reactionners, brokers, ...) are linked to active schedulers')
        for realm in self.realms:
            self.dispatch_ok &= realm.check_satellites_dispatch(self.first_dispatch_done)
    
    
    # Check if all active items are still alive & the result goes into self.dispatch_ok
    def check_dispatch(self):
        # Check that the other arbiters
        self._check_other_arbiters_dispatch()
        
        self._check_schedulers_dispatch()
        
        self._check_satellites_dispatch()
    
    
    # Imagine a world where... oh no, wait...
    # Imagine a master got the conf and the network is down a spare takes it (good :) ). Like the Empire, the master
    # strikes back! It was still alive! (like Elvis). It still got conf and is running! not good!
    # Bad dispatch: a link that has a conf but I do not allow this so I ask it to wait a new conf and stop kidding.
    def check_bad_dispatch(self):
        logger.debug('[DISPATCH] Checking daemons that manage a shard/configuration but should not have (like rogue daemons)')
        for realm in self.realms:
            realm.check_bad_dispatch(self.realms, self.first_dispatch_done, self.arbiter_trace)
    
    
    # Manage the dispatch
    # REF: doc/shinken-conf-dispatching.png (3)
    def dispatch(self):
        logger.debug('[DISPATCH] Start to dispatch shards & satellites configurations')
        # Ok, we pass at least one time in dispatch, so now errors are True errors
        self.first_dispatch_done = True
        
        # If no needed to dispatch, do not dispatch :)
        if self.dispatch_ok:
            return
        
        t0 = time.time()
        threads = []
        for realm in self.realms:
            thread = threading.Thread(None, target=realm.dispatch_schedulers, name='dispatch_schedulers_%s' % realm.get_name(), args=(self.arbiter_trace, self.configuration_incarnation))
            thread.daemon = True
            thread.start()
            threads.append(thread)
        for thread in threads:
            thread.join()
        logger.debug('[DISPATCH] All realms schedulers dispatched in %.2fs' % (time.time() - t0))
        
        # We pop conf to dispatch, so it must be no more conf...
        shards_to_dispatch = [shard for shard in self._get_all_shards() if not shard.is_assigned]
        nb_missed = len(shards_to_dispatch)
        if nb_missed > 0:
            logger.error("[DISPATCH] MISSING SHARDS  All schedulers shards are not dispatched: %d are missing" % (nb_missed))
        else:
            logger.info("[DISPATCH] SHARD ALL SENT   All %d schedulers shards are dispatched." % (len(self._get_all_shards())))
            self.dispatch_ok = True
        
        # Sched without conf in a dispatch ok are set to no need_conf so they do not raise dispatch where no use
        if self.dispatch_ok:
            for scheduler in self.schedulers.items.values():
                if scheduler.conf is None:
                    scheduler.need_conf = False
        
        arbiters_cfg = {}
        for arb in self.arbiters:
            arbiters_cfg[arb.id] = arb.give_satellite_cfg()
        
        # We put the satellites conf with the "new" way so they see only what we want
        t0 = time.time()
        threads = []
        for realm in self.realms:
            thread = threading.Thread(None, target=realm.dispatch_satellites, name='dispatch_satellites_%s' % realm.get_name(), args=(arbiters_cfg, self.arbiter_trace,))
            thread.daemon = True
            thread.start()
            threads.append(thread)
        for thread in threads:
            thread.join()
        logger.debug('[DISPATCH] All realms satellites dispatched in %.2fs' % (time.time() - t0))
    
    
    # interface to run the do_disable_previous_run_daemons into a thread
    def disable_previous_run_daemons(self, last_run_alive_daemons):
        t = threading.Thread(None, target=self._do_disable_previous_run_daemons, name='do_disable_previous_run_daemons', args=(last_run_alive_daemons,))
        t.daemon = True
        t.start()
    
    
    # Find the new disabled daemons and ask them to stop the work
    # They are now disabled, we don't need their work anymore
    def _do_disable_previous_run_daemons(self, last_run_alive_daemons):
        # convert the current daemon dot dict and str representation like stored in the retention
        current_daemons = {}
        for _type, daemons in self.conf.get_all_daemons().iteritems():
            current_daemons[_type] = [d.give_satellite_cfg() for d in daemons]
            for current_daemon in current_daemons[_type]:
                # generate a pseudo unique key depending on the daemon
                unique_key = u"%s:%s" % (current_daemon['address'], current_daemon['port'])
                current_daemon['unique_key'] = unique_key
        
        # we need to keep only the satellites that are not anymore in conf
        all_daemons_to_stop = []
        for _type, last_run_daemons in last_run_alive_daemons.iteritems():
            for last_run_daemon in last_run_daemons:
                # generate a pseudo unique key depending on the daemon
                unique_key = u"%s:%s" % (last_run_daemon['address'], last_run_daemon['port'])
                daemon_presently_running = any(unique_key == d['unique_key'] for d in current_daemons[_type])
                # the daemon was running previously but not anymore running
                if not daemon_presently_running:
                    all_daemons_to_stop.append(last_run_daemon)
        
        # We now need to convert the config (dict representation) into SatelliteLink and ask them to stop
        for daemon_to_stop in all_daemons_to_stop:
            # we need to have each key as list to instantiate a SatelliteLink
            for k, v in daemon_to_stop.iteritems():
                if v is True:
                    daemon_to_stop[k] = ['1', ]
                elif v is False:
                    daemon_to_stop[k] = ['0', ]
                elif isinstance(v, Number):
                    daemon_to_stop[k] = [unicode(v), ]
                if not isinstance(daemon_to_stop[k], list):
                    daemon_to_stop[k] = [v, ]
            sat_link = SatelliteLink(daemon_to_stop)
            sat_link.fill_default()
            sat_link.pythonize()
            sat_link.wait_new_conf()
            logger.info("[DISPATCH] DISABLED DAEMON  The element [%s] at [%s] was managed previously but have been disabled. We are asking it to go in sleep mode." % (sat_link.name, sat_link.uri))
