#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


from .satellitelink import SatelliteLinks
from .property import BoolProp, IntegerProp, StringProp, ListProp
from .http_client import HTTPExceptions
from .withinventorysatellitelink import WithInventorySatelliteLink


class ReceiverFilter(object):
    def __init__(self, filter_template, data_templates):
        self._filter_template = filter_template
        self._data_templates = set(data_templates)
    
    
    def template_match_data_filter(self, template_name):
        return template_name in self._data_templates
    
    
    def template_match_template_filter(self, template_name):
        return template_name == self._filter_template


class ReceiverLink(WithInventorySatelliteLink):
    """Please Add a Docstring to describe the class here"""
    
    id = 0
    my_type = 'receiver'
    properties = WithInventorySatelliteLink.properties.copy()
    properties.update({
        'receiver_name'                          : StringProp(fill_brok=['full_status'], to_send=True),
        'port'                                   : IntegerProp(default='7772', fill_brok=['full_status']),
        'manage_sub_realms'                      : BoolProp(default='1', fill_brok=['full_status']),
        'direct_routing'                         : BoolProp(default='0', fill_brok=['full_status'], to_send=True),
        
        'elements_sharding_enabled'              : BoolProp(default='0', to_send=True),
        'elements_sharding_filter_by_template'   : StringProp(default='', to_send=True),
        'elements_sharding_add_data_of_templates': ListProp(default='', to_send=True),
        'elements_sharding_weight'               : IntegerProp(default='1', to_send=True),
        
    })
    
    
    def __init__(self, *args, **kwargs):
        super(ReceiverLink, self).__init__(*args, **kwargs)
    
    
    def get_name(self):
        return self.receiver_name
    
    
    def register_to_my_realm(self):
        self.realm.receivers.append(self)
    
    
    def push_host_names(self, sched_id, hnames):
        try:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                self.add_failed_check_attempt()
                return
            
            self.con.get('ping')
            self.con.post('push_host_names', {'sched_id': sched_id, 'hnames': hnames}, wait='long')
        except HTTPExceptions, exp:
            self.add_failed_check_attempt(reason=str(exp))
    
    
    # We give a filter object with our data but only if we are a manage sub element receiver
    def get_inventory_templates_filter(self):
        if not self.elements_sharding_enabled:
            return None
        return ReceiverFilter(self.elements_sharding_filter_by_template, self.elements_sharding_add_data_of_templates)


class ReceiverLinks(SatelliteLinks):
    """Please Add a Docstring to describe the class here"""
    
    name_property = "receiver_name"
    inner_class = ReceiverLink
    
    
    # Receivers can have properties about interesting host templates
    # - elements_sharding_enabled = 1          => should give/have templates
    # - elements_sharding_add_data_of_templates            => will have only the host with this template
    # - elements_sharding_weight => hosts will have data about theses templates only
    def get_inventory_filters_by_receivers(self):
        res = []
        for receiver in self:
            receiver_filter = receiver.get_inventory_templates_filter()
            if receiver_filter:
                res.append(receiver_filter)
        return res
