#!/usr/bin/env python
# -*- coding: utf-8 -*-

#
# Copyright (C) 2009-2016:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#     Martin Benjamin, b.martin@shinken-solutions.com
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import base64
import cPickle
import threading
import zlib
import time
import copy
import traceback

from .daemon import Daemon, Interface
from .log import logger, get_chapter_string
from .http_client import HTTPClient, HTTPExceptions

# Try to see if we are in an android device or not
is_android = True
try:
    import android
except ImportError:
    is_android = False

CHAPTER_CONNECTION = get_chapter_string('CONNECTION')


# Interface for Arbiter, our big MASTER It gives us our conf
class IForArbiter(Interface):
    doc = 'Remove a scheduler connexion (internal)'
    
    
    # Arbiter ask us to do not manage a scheduler_id anymore
    # I do it and don't ask why
    def remove_from_conf(self, sched_id):
        try:
            sched_id = int(sched_id)
        except Exception:
            logger.error('Arbiter did ask to remove a shard %s but the format is invalid' % sched_id)
            return
        logger.info('Arbiter is asking me to remove a shard: %s' % (sched_id))
        try:
            del self.app.schedulers[sched_id]
        except KeyError:
            logger.info('Arbiter asked me to remove such a shard, but I do not have it %s in my keys: %s' % (sched_id, self.app.schedulers.keys()))
    
    
    remove_from_conf.doc = doc
    
    doc = 'Return the managed configuration ids (internal)'
    
    
    # Arbiter ask me which sched_id I manage, If it is not ok with it
    # It will ask me to remove one or more sched_id
    def what_i_managed(self):
        what_i_manage = self.app.what_i_managed()
        logger.debug("The arbiter asked me what I manage. It's %s" % what_i_manage)
        return what_i_manage
    
    
    what_i_managed.need_lock = False
    what_i_managed.doc = doc
    
    doc = 'Ask the daemon to drop its configuration and wait for a new one'
    
    
    # Call by arbiter if it thinks we are running but we must do not (like
    # if I was a spare that take a conf but the master returns, I must die
    # and wait a new conf)
    # Us: No please...
    # Arbiter: I don't care, hasta la vista baby!
    # Us: ... <- Nothing! We are dead! you don't get it or what??
    # Reading code is not a job for eyes only...
    def wait_new_conf(self):
        logger.info("Arbiter wants me to wait for a new configuration")
        super(IForArbiter, self).wait_new_conf()
        self.app.schedulers.clear()
    
    
    wait_new_conf.doc = doc
    
    doc = 'Push broks objects to the daemon (internal)'
    
    
    # NB: following methods are only used by broker
    # Used by the Arbiter to push broks to broker
    def push_broks(self, broks):
        with self.app.arbiter_broks_lock:
            self.app.arbiter_broks.extend(broks.values())
    
    
    push_broks.method = 'post'
    # We are using a Lock just for NOT lock this call from the arbiter :)
    push_broks.need_lock = False
    push_broks.doc = doc
    
    doc = 'Get the external commands from the daemon (internal)'
    
    
    # The arbiter ask us our external commands in queue
    # Same than push_broks, we will not using Global lock here,
    # and only lock for external_commands
    def get_external_commands(self):
        with self.app.external_commands_lock:
            cmds = self.app.get_external_commands()
            raw = cPickle.dumps(cmds)
            raw = base64.b64encode(zlib.compress(raw))
        return raw
    
    
    get_external_commands.need_lock = False
    get_external_commands.doc = doc
    
    doc = 'Does the daemon got configuration (receiver)'
    
    
    ### NB: only useful for receiver
    def got_conf(self):
        return self.app.cur_conf is not None
    
    
    got_conf.need_lock = False
    got_conf.doc = doc
    
    doc = 'Push hostname/scheduler links (receiver in direct routing)'
    
    
    # Use by the receivers to got the host names managed by the schedulers
    def push_host_names(self, sched_id, hnames):
        self.app.push_host_names(sched_id, hnames)
    
    
    push_host_names.method = 'post'
    push_host_names.doc = doc


class IStats(Interface):
    """
    Interface for various stats about executor activity
    """
    
    doc = 'Get raw stats from the daemon'
    
    
    def get_raw_stats(self, param=''):
        res = super(IStats, self).get_raw_stats(param)
        return res
    
    
    get_raw_stats.doc = doc
    get_raw_stats.need_lock = False


class BaseSatellite(Daemon):
    """Super class for Broker, scheduler, reactioner, poller"""
    
    # Should we look at passive property for external connection
    is_using_passive_connection_information = False
    
    
    def __init__(self, name, config_file, is_daemon, do_replace, debug, debug_file, daemon_id=0):
        super(BaseSatellite, self).__init__(name, config_file, is_daemon, do_replace, debug, debug_file, daemon_id)
        # Ours schedulers
        self.schedulers = {}
        # Our arbiters
        self.arbiters = {}
        
        # Our pollers and reactionners, maybe void for some daemons
        self.pollers = {}
        self.reactionners = {}
        self.receivers = {}
        # Now we create the interfaces
        self._add_http_interface(IForArbiter(self))
        
        # Can have a queue of external_commands given by modules will be taken by arbiter to process
        self.external_commands = []
        self.external_commands_lock = threading.RLock()
        
        self.activated = True
        self.spare = False
        # We will have a tread by distant satellites, so we must protect our access
        self.satellite_lock = threading.RLock()
        
        # Keep broks so they can be eaten by a broker
        self.broks = {}
    
    
    # Someone ask us our broks. We send them, and clean the queue
    def get_broks(self):
        res = copy.copy(self.broks)
        self.broks.clear()
        return res
    
    
    # The arbiter can resent us new conf in the pyro_daemon port.
    # We do not want to loose time about it, so it's not a blocking  wait, timeout = 0s
    # If it send us a new conf, we reinit the connections of all schedulers
    def watch_for_new_conf(self, timeout):
        self.sleep(timeout)
    
    
    # Give the arbiter the data about what I manage as shards/schedulers to know if we should
    # be updated or not. We will return the shards ids and the push_flavor or them
    # Note: maybe the arbiter did just send us a new_conf, but we did not consume it, so
    # if we have one, we should look in it instead of the running conf that will be changed at
    # the end of our turn
    def what_i_managed(self):
        r = {}
        
        # Maybe we did receive a new configuration but did not consume it
        with self.satellite_lock:
            if self.new_conf:  # got a new conf, use it instead of the running one
                for (sched_id, scheduler) in self.new_conf['schedulers'].iteritems():
                    r[sched_id] = scheduler['push_flavor']
                return r
        
        # no new conf, and no conf at all? we are just void
        if not self.already_have_conf:
            return None
        
        # ok look in the running configuration
        for (k, v) in self.schedulers.iteritems():
            # Beware of the synchronizers! it's a fake entry and we should not take it
            if not v.get('unmanaged_by_arbiter', False):
                r[k] = v['push_flavor']
        
        return r
    
    
    # Call by arbiter to get our external commands
    def get_external_commands(self):
        res = self.external_commands
        self.external_commands = []
        return res
    
    
    def get_satellite_connections(self):
        res = []
        # logger.debug('FUCK %s %s %s %s' % (self.arbiters, self.pollers, self.reactionners, self.schedulers))
        for (t, ds) in [('receiver', getattr(self, 'receivers', None)), ('poller', getattr(self, 'pollers', None)), ('reactionner', getattr(self, 'reactionners', None)), ('scheduler', getattr(self, 'schedulers', None))]:
            if not ds:
                continue
            
            for e in ds.values():
                # scehduler can have directly scehduler object here, skip it
                if not isinstance(e, dict):
                    continue
                # if synchronizer dummy entry, skip it
                if e.get('unmanaged_by_arbiter', False):
                    continue
                # skip passive pollers on schedulers
                if self.is_using_passive_connection_information and 'passive' in e and not e['passive']:
                    continue
                proto = 'https' if e['use_ssl'] else 'http'
                d = {'name': e['name'], 'type': t, 'address': e['address'], 'proto': proto, 'uri': e['uri'], 'port': e['port']}
                if 'passive' in e:
                    d['passive'] = e['passive']
                res.append(d)
        return res
        # Get the good tabs for links by the kind. If unknown, return None
    
    
    def get_link_from_type(self, daemon_type, daemon_id):
        t = {
            'scheduler': self.schedulers,
            'arbiter'  : self.arbiters,
        }
        with self.satellite_lock:
            return t.get(daemon_type, {}).get(daemon_id, None)
    
    
    def get_any_link_from_type(self, daemon_type):
        t = {
            'scheduler': self.schedulers,
            'arbiter'  : self.arbiters,
        }
        with self.satellite_lock:
            return len(t.get(daemon_type, {})) != 0
    
    
    # Check if we do not connect to often to this
    def is_connection_try_too_close(self, elt):
        now = time.time()
        last_connection = elt['last_connection']
        if now - last_connection < 5:
            return True
        return False
    
    
    # initialize or re-initialize connection with scheduler
    def pynag_con_init(self, sat_entry):
        daemon_type = sat_entry['type']
        
        if daemon_type == 'scheduler':
            # If sched is not active, I do not try to init
            # it is just useless
            is_active = sat_entry['active']
            if not is_active:
                return
        
        # If we try to connect too much, we slow down our tests
        if self.is_connection_try_too_close(sat_entry):
            return
        
        # Ok, we can now update it
        sat_entry['last_connection'] = time.time()
        
        # DBG: print "Init connection with", links[id]['uri']
        
        # DBG: print "Running id before connection", daemon_incarnation
        uri = sat_entry['uri']
        timeout = sat_entry.get('timeout', 3)
        data_timeout = sat_entry.get('data_timeout', 120)
        name = sat_entry['name']
        try:
            sat_entry['con'] = HTTPClient(uri=uri, strong_ssl=sat_entry['hard_ssl_name_check'], timeout=timeout, data_timeout=data_timeout)
        except HTTPExceptions as exp:
            # But the multiprocessing module is not compatible with it!
            # so we must disable it immediately after
            logger.info('%s Connection problem to the %s %s (uri="%s"): %s' % (CHAPTER_CONNECTION, daemon_type, name, uri, str(exp)))
            sat_entry['con'] = None
            return
        
        before_ = time.time()
        did_connect = self.ping_and_check_distant_daemon(sat_entry)
        elapsed = time.time() - before_
        
        if did_connect:
            logger.info('%s Connection OK to the %s %s in %.3fs (uri="%s", ping_timeout=%ss, transfert_timeout=%ss)' % (CHAPTER_CONNECTION, daemon_type, name, elapsed, uri, timeout, data_timeout))
    
    
    # For a new distant daemon, if it is a scheduler, ask for a new full broks generation
    def _manage_new_distant_daemon_incarnation(self, entry, old_incar, new_incar):
        raise NotImplemented('_manage_new_distant_daemon_incarnation')
    
    
    def ping_and_check_distant_daemon(self, sat_entry):
        con = sat_entry['con']
        daemon_type = sat_entry['type']
        if con is None:
            self.pynag_con_init(sat_entry)
            con = sat_entry['con']
            if con is None:
                return False
        try:
            # initial ping must be quick
            con.get('ping')
            new_incar = con.get('get_daemon_incarnation')
            new_incar = float(new_incar)
            new_incarnation = False
            # protect daemon_incarnation from modification
            with self.satellite_lock:
                # data transfer can be longer
                daemon_incarnation = sat_entry['daemon_incarnation']
                # logger.debug("type[%s] daemon_incarnation old[%s]/new[%s]" % (daemon_type, daemon_incarnation, new_incar))
                
                # The schedulers have been restarted: it has a new run_id.
                # So we clear all verifs, they are obsolete now.
                if new_incar != daemon_incarnation:
                    logger.debug("[%s] New daemon incarnation for the %s %s: %s (was %s)" % (self.name, daemon_type, sat_entry['name'], new_incar, daemon_incarnation))
                    new_incarnation = True
                # Ok all is done, we can save this new incarnation
                sat_entry['daemon_incarnation'] = new_incar
            if new_incarnation:
                self._manage_new_distant_daemon_incarnation(sat_entry, daemon_incarnation, new_incar)
            return True
        except HTTPExceptions as exp:
            logger.error("Connection problem to the %s %s: %s" % (daemon_type, sat_entry['name'], str(exp)))
            sat_entry['con'] = None
            return False
        except KeyError as exp:
            logger.error("the %s '%s' is not initialized: %s" % (daemon_type, sat_entry['name'], str(exp)))
            sat_entry['con'] = None
            return False
    
    
    # Daemon get something from distant, should be implemented!
    def get_jobs_from_distant(self, e):
        raise NotImplementedError('get_jobs_from_distant')
    
    
    # By default we are connecting to everyone
    # only the scheduler will skip some pollers/reactionners because it need to
    # connect only to passive one
    def should_connect_to_distant_satellite(self, satellite_type, distant_link):
        return True
    
    
    def _do_satellite_thread(self, s_type, s_id, t_name):
        try:
            self.do_satellite_thread(s_type, s_id, t_name)
        except:
            err = traceback.format_exc()
            logger.warning('The thread for the %s %s is in error (%s): but it will restart. Your monitoring is still working. Please fill a bug with your log.' % (s_type, t_name, err.replace('\n', '---')))
    
    
    def do_satellite_thread(self, s_type, s_id, t_name):
        logger.debug("SATELLITE THREAD: Starting thread to exchange with %s [%s]" % (s_type, t_name))
        
        with self.satellite_lock:
            distant_link = self.get_link_from_type(s_type, s_id)
            if distant_link is None:  # already down?
                return
        
        uri = distant_link['uri']
        while True:
            with self.satellite_lock:
                # first look if we are still need or not
                distant_link = self.get_link_from_type(s_type, s_id)
            
            if distant_link is None or (uri != distant_link['uri']):  # no more present, or uri did change? EXIT!!!
                logger.info('SATELLITE THREAD: The connection thread to the %s with the id %s (%s) is not longer need as this daemon is no more that same as before. Restarting it.' % (s_type, s_id, t_name))
                return
            # For some distant satellite, we have a thread but currently the distant daemon is not
            # interesting (not passive, etc). we need to have a thread so it's already running, and ready as soon as
            # the configuration will change
            if self.should_connect_to_distant_satellite(s_type, distant_link):
                # we have an entry, so we can ping it
                if self.ping_and_check_distant_daemon(distant_link):
                    # so now if it change, we can know that we must drop this thread
                    # logger.debug("[Broks] SATELLITE THREAD: get broks from daemon:[%s] id:[%s] push_flavor:[%s] name:[%s]" % (s_type, s_id, e['daemon_incarnation'], t_name))
                    self.get_jobs_from_distant(distant_link)
                
            time .sleep(1)
    
    
    # We will look for satellites, and if we don't have a thread or a dead one, start a new one
    def assert_valid_satellite_threads(self):
        with self.satellite_lock:
            types = {'scheduler': self.schedulers, 'poller': self.pollers, 'reactionner': self.reactionners, 'receiver': self.receivers}
            for (satellite_type, satellite_definition) in types.iteritems():
                for (satellite_id, satellite_entry) in satellite_definition.iteritems():
                    self._assert_one_satellite_thread(satellite_type, satellite_id, satellite_entry)
    
    
    def _assert_one_satellite_thread(self, satellite_type, satellite_id, satellite_entry):
        thread = satellite_entry['thread']
        restart = False
        
        if thread is None:
            restart = True
        elif not thread.is_alive():
            thread.join(1)
            restart = True
        
        if restart:
            thread_name = "satellite-thread:%s:%s" % (satellite_type, satellite_entry['name'])
            thread = threading.Thread(None, target=self._do_satellite_thread, name=thread_name, args=(satellite_type, satellite_id, thread_name))
            thread.daemon = True
            thread.start()
            satellite_entry['thread'] = thread
