#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

import os
import shinkensolutions.shinkenjson as json
import shinkensolutions.api as api_synchronizer
from shinken.misc.type_hint import NoReturn, Dict, Any
from .configuration_component import ConfigurationComponent
from .logger_component import LoggerComponent
from .abstract_component import AbstractComponent
from shinken.log import PartLogger

Path = str
Lang = str
PartName = str
TranslateKey = str


class TranslatePart(object):
    def __init__(self, logger, current_language):
        # type: (PartLogger, Lang) -> None
        
        self.logger = logger
        self._current_language = current_language
        self._contents = {}
        self._contents_getter = None
        self._ = self.translate
        self._load()
    
    
    def translate(self, translate_key, *args):
        # type: (TranslateKey, Any) -> str
        translation = self._contents_getter(translate_key, None)
        if translation is None:
            self.logger.error('Translate: cannot find %s in the lang %s' % (translate_key, self._current_language))
            return 'TO_TRAD(%s)' % translate_key
        return translation % args if args else translation
    
    
    def is_translate_existing(self, translate_key):
        # type: (TranslateKey) -> bool
        return translate_key in self._contents
    
    
    def _load(self):
        raise NotImplementedError()
    
    
    def _read_dico_file(self, dico_file_path):
        with open(dico_file_path, 'r') as f:
            buf = f.read()
        lines = buf.splitlines()
        new_lines = []
        for line in lines:
            line = line.strip()
            if line.startswith('//'):
                continue
            if line.startswith('var lang'):
                line = '{'
            if line.startswith('};'):
                line = '}'
            new_lines.append(line)
        buf = '\n'.join(new_lines)
        raw_trad = json.loads(buf)
        flat_trad = {}
        self._flat_dict(raw_trad, flat_trad)
        return flat_trad
    
    
    def _flat_dict(self, node, flat_trad_dico, key_name=''):
        _key_name = ''
        for k, v in node.iteritems():
            if key_name:
                _key_name = '.'.join([key_name, k])
            else:
                _key_name = k
            
            if isinstance(v, dict):
                self._flat_dict(v, flat_trad_dico, _key_name)
            else:
                flat_trad_dico[_key_name] = v


class DefaultTranslatePart(TranslatePart):
    def _get_dico_file_path(self):
        # type: () -> Path
        _lang_path = os.path.join(os.path.abspath(os.path.dirname(__file__)), os.path.pardir, 'htdocs', 'js', 'traductions')
        _path_dico_file = os.path.join(_lang_path, self._current_language + '.js')
        if not os.path.exists(_path_dico_file):
            self.logger.error('Cannot load the lang file %s: no such file' % _path_dico_file)
            return ''
        
        return _path_dico_file
    
    
    def _load(self):
        dico_file_path = self._get_dico_file_path()
        flat_trad = self._read_dico_file(dico_file_path)
        
        self._contents = flat_trad
        self._contents_getter = flat_trad.get


class SourceTranslatePart(TranslatePart):
    def __init__(self, logger, current_language, source_type):
        self.source_type = source_type
        super(SourceTranslatePart, self).__init__(logger, current_language)
    
    
    def _get_dico_file_path(self):
        # type: () -> Path
        
        _lang_path = os.path.join(os.path.abspath(os.path.dirname(__file__)), os.path.pardir, 'htdocs', 'js', 'traductions')
        _path_dico_file = os.path.join(_lang_path, self._current_language + '.js')
        if not os.path.exists(_path_dico_file):
            self.logger.error('Cannot load the lang file %s: no such file' % _path_dico_file)
            return ''
        
        return _path_dico_file
    
    
    def _load(self):
        source_dico_folder = '/var/lib/shinken/modules/%s/translation/' % self.source_type
        default_source_dico_folder = os.path.join(os.path.abspath(os.path.dirname(api_synchronizer.__file__)), 'synchronizer', 'source', 'translation')
        
        default_source_dico_file_path = os.path.join(default_source_dico_folder, self._current_language + '.js')
        source_dico_file_path = os.path.join(source_dico_folder, self._current_language + '.js')
        
        flat_trad = self._read_dico_file(default_source_dico_file_path)
        if os.path.exists(source_dico_file_path):
            source_flat_trad = self._read_dico_file(source_dico_file_path)
            for k, v in source_flat_trad.iteritems():
                flat_trad[k] = v
        
        self._contents = flat_trad
        self._contents_getter = flat_trad.get


class TranslateComponent(AbstractComponent):
    
    def __init__(self, configuration_component, logger_component):
        # type: (ConfigurationComponent, LoggerComponent) -> NoReturn
        self.logger = logger_component.get_logger()
        self._current_language = configuration_component.lang
        self.parts = {}  # type: Dict[PartName, TranslatePart]
    
    
    def init(self):
        self.parts['__default__'] = DefaultTranslatePart(self.logger, self._current_language)
    
    
    def source_translator(self, source_type):
        # type:  (PartName) -> TranslatePart
        part_name = 'source_translator_part_%s' % source_type
        if part_name not in self.parts:
            self.parts[part_name] = SourceTranslatePart(self.logger, self._current_language, source_type)
        return self.parts[part_name]
    
    
    def translator(self, part='__default__'):
        # type:  (PartName) -> TranslatePart
        return self.parts[part]
