#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import sys
import time
import datetime
import copy
import traceback
from pymongo.errors import BulkWriteError, InvalidOperation
from shinken.log import logger
from pprint import pformat
from hashlib import sha256
from .def_items import ITEM_STATE, ITEM_TYPE
from .dataprovider.dataprovider_mongo import DataProviderMongo
from .datamanagerV2 import DataManagerV2, get_type_item_from_class
from .crypto import DatabaseCipher
from .helpers import get_default_value, add_unique_value_and_handle_plus_and_null

app = None


def migrate_hosts_contacts_properties(app, hosts):
    # Host and host templates must be changed from the contacts into:
    # * view_contacts/groups
    # * notification_contacts/groups
    # * edition_contacts/groups
    # This only applies if the host or host template does not have any defined view/notification/edition_contact/groups
    properties = ['contacts', 'contact_groups']
    prefixs = ['view', 'notification', 'edition']
    must_view_prefixs = ['notification', 'edition']
    for item in hosts:
        for property in properties:
            for prefix in prefixs:
                new_property = '%s_%s' % (prefix, property)
                if property in item and new_property in item:
                    # New keys win over old one, del it from object and skip migration
                    del item[property]
            
            if property not in item:
                continue
            
            value = item[property]
            del item[property]
            
            for prefix in prefixs:
                new_property = '%s_%s' % (prefix, property)
                item[new_property] = value
        # Now, duplicate edition/notification contacts in view contacts if not already allowed
        _default_view_contacts = get_default_value(ITEM_TYPE.HOSTS, 'view_contacts')[0] or 'nobody'
        
        for property in properties:
            view_key = 'view_%s' % property
            for prefix in must_view_prefixs:
                new_property = '%s_%s' % (prefix, property)
                if item.get(new_property, '') and (_default_view_contacts == 'nobody' or item.get(view_key, '')):
                    item[view_key] = add_unique_value_and_handle_plus_and_null(item.get(view_key, ''), item[new_property], handle_plus=True)


def migrate_contact_contactgroups(contacts):
    # Contacts and contact templates must be changed from the contacts into:
    # * contact_groups/contactgroups
    # This only applies if the contact or contact template does not have any defined contactgroups
    for contact in contacts:
        if contact.get('contact_groups', None):
            contact['contactgroups'] = contact.get('contact_groups')
            del contact['contact_groups']


def migrate_database(_app):
    global app, migration_datamanager
    app = _app
    # We use our datamanager with no callback
    migration_datamanager = DataManagerV2(DataProviderMongo(app.mongodb_db, app.database_cipher), synchronizer=app, use_default_callbacks=False)
    
    _remove_old_tmp_collection()
    _migrate_hosts_contacts_properties_in_db()
    _migrate_contacts_contactgroups_properties_in_db()
    _migrate_database_part_protected_properties()
    _check_old_host_to_cluster_boolean()
    _check_clusters_to_cluster_boolean()
    _add_source_name_in_discovery_confs()
    _migrate_nb_elements_in_last_synchronization()
    _remove_basic_element_from_database()


def _migrate_nb_elements_in_last_synchronization():
    for last_sync_col_name in ("last_synchronization", "last_synchronizations"):
        must_exec_bulk = False
        last_synchronization_col = getattr(app.mongodb_db, last_sync_col_name)
        bulk_ops = last_synchronization_col.initialize_unordered_bulk_op()
        _last_syncs = list(last_synchronization_col.find())
        for last_synchronization in _last_syncs:
            saved_values = last_synchronization.get('saved', {})
            if saved_values is None:
                continue
            for item_class, value in saved_values.items():
                # if value is an int, we need to set this as dict with an nb_elements keys
                if isinstance(value, int):
                    last_synchronization['saved'][item_class] = {
                        'warning'    : 0,
                        'error'      : 0,
                        'nb_elements': value,
                    }
                    bulk_ops.find({'_id': last_synchronization['_id']}).replace_one(last_synchronization)
                    must_exec_bulk = True
        
        if must_exec_bulk:
            try:
                bulk_ops.execute()
            except (TypeError, ValueError) as e:
                logger.info("[migrate_nb_elements_in_last_synchronization] %s" % traceback.format_exc())


def _add_source_name_in_discovery_confs():
    discovery_confs_col = getattr(app.mongodb_db, "discovery_confs")
    migration_needed_confs = discovery_confs_col.find({'source_name': None})
    for disco_conf in migration_needed_confs:
        disco_conf['source_name'] = 'discovery'
        discovery_confs_col.save(disco_conf)


def _migrate_database_part_protected_properties():
    conf_protected_fields_keyfile = app.conf.protect_fields__encryption_keyfile
    col = getattr(app.mongodb_db, "synchronizer-info")
    current_protected_fields = col.find_one({'_id': 'protected_fields_info'})
    
    if not current_protected_fields:
        current_protected_fields = {}
    
    if current_protected_fields.get('protect_fields__activate_database_encryption', False) == False and not app.conf.protect_fields__activate_encryption:
        return
    
    if current_protected_fields.get('protect_fields__activate_database_encryption', '0') == '1':
        if not conf_protected_fields_keyfile:
            logger.critical("[migrate_protected_fields]  Please set the parameter 'protect_fields__encryption_keyfile' to point to the right key file in /etc/shinken/synchronizer.cfg")
            logger.critical("[migrate_protected_fields]  It might be a typing mistake, or the parameter could be overloaded in another configuration file")
            logger.critical("[migrate_protected_fields]  The synchronizer will not start because it will not be able to process encrypted data.")
            logger.critical("[migrate_protected_fields]  Protection is activated but but the parameter 'protect_fields__encryption_keyfile' is not defined")
            sys.exit(1)
    
    try:
        complete_key = open(app.conf.protect_fields__encryption_keyfile).read().strip()
        key_value = complete_key[complete_key.index("|") + 1:]
        key_name = complete_key[:complete_key.index("|")].strip().decode('utf8', 'ignore')
    except (OSError, IOError) as e:
        logger.critical("[migrate_protected_fields]  Make sure the file exists and that the user 'shinken' is allowed to read it")
        logger.critical("[migrate_protected_fields]  The synchronizer will not start because it will not be able to process encrypted data.")
        logger.critical("[migrate_protected_fields]  Cannot read the protected fields secret file '%s' : %s " % (app.conf.protect_fields__encryption_keyfile, str(e)))
        sys.exit(1)
    except ValueError:
        logger.critical("[migrate protected_fields]  The keyfile seems to be corrupted. If you have an export available, you can use it to restore the correct key")
        logger.critical("[migrate protected_fields]  using the command shinken-protected-fields-keyfile-restore and restart the synchronizer.")
        logger.critical("[migrate_protected_fields]  The synchronizer will not start in order not to corrupt data.")
        logger.critical("[migrate protected_fields]  The key contained in the keyfile does not have the right structure.")
        sys.exit(1)
    
    configuration_protected_fields = {
        '_id'                                         : 'protected_fields_info',
        'protect_fields__substrings_matching_fields'  : app.conf.protect_fields__substrings_matching_fields.split(","),
        'protect_fields__activate_database_encryption': app.conf.protect_fields__activate_encryption,
        'protect_fields__encryption_keyfile_hash'     : sha256(key_value).hexdigest(),
        'protect_fields__encryption_key_name'         : key_name
    }
    
    if configuration_protected_fields.get('protect_fields__encryption_keyfile_hash') != current_protected_fields.get('protect_fields__encryption_keyfile_hash') or \
            configuration_protected_fields.get('protect_fields__encryption_key_name') != current_protected_fields.get('protect_fields__encryption_key_name'):
        still_extracted = False
    else:
        still_extracted = current_protected_fields.get('extracted_key', False)
    
    if _ask_must_do_protected_properties_migration(current_protected_fields, configuration_protected_fields):
        _migrate_protected_properties(current_protected_fields, configuration_protected_fields)
        
        if current_protected_fields.get('protect_fields__activate_database_encryption', False) != configuration_protected_fields['protect_fields__activate_database_encryption']:
            _migrate_protected_properties_sources_config(app.conf.protect_fields__activate_encryption)
        
        from_data = {
            'protect_fields__substrings_matching_fields'  : current_protected_fields.get('protect_fields__substrings_matching_fields', []),
            'protect_fields__activate_database_encryption': current_protected_fields.get('protect_fields__activate_database_encryption', ""),
            'protect_fields__encryption_keyfile_hash'     : current_protected_fields.get('protect_fields__encryption_keyfile_hash', ""),
            'protect_fields__encryption_key_name'         : current_protected_fields.get('protect_fields__encryption_key_name', ""),
            'extracted_key'                               : current_protected_fields.get('extracted_key', False)
        }
        
        to_data = {
            'protect_fields__substrings_matching_fields'  : configuration_protected_fields.get('protect_fields__substrings_matching_fields', []),
            'protect_fields__activate_database_encryption': configuration_protected_fields.get('protect_fields__activate_database_encryption', ""),
            'protect_fields__encryption_keyfile_hash'     : configuration_protected_fields.get('protect_fields__encryption_keyfile_hash', ""),
            'protect_fields__encryption_key_name'         : configuration_protected_fields.get('protect_fields__encryption_key_name', ""),
            'extracted_key'                               : still_extracted
        }
        this_migration = {'date': str(datetime.datetime.now()), 'from': from_data, 'to': to_data}
        last_migrations = current_protected_fields.get('last_migrations', [])
        last_migrations.insert(0, this_migration)
        configuration_protected_fields['last_migrations'] = last_migrations[:5]
        configuration_protected_fields['extracted_key'] = still_extracted
        col.save(configuration_protected_fields)


def _ask_must_do_protected_properties_migration(src_config, dst_config):
    src_fields = set(src_config.get('protect_fields__substrings_matching_fields', []))
    dst_fields = set(dst_config.get('protect_fields__substrings_matching_fields', []))
    
    src_activated = src_config.get('protect_fields__activate_database_encryption', False)
    dst_activated = dst_config.get('protect_fields__activate_database_encryption', False)
    
    if not (src_activated or dst_activated):
        logger.info("[migrate_protected_fields] Protected fields encryption in database disabled => nothing to do")
        return False
    
    if src_config.get('protect_fields__encryption_key_name') != dst_config.get('protect_fields__encryption_key_name') and src_activated:
        logger.critical("[migrate_protected_fields]  The Protected Field key file was modified.")
        logger.critical("[migrate_protected_fields]  The synchronizer will not start in order not to corrupt data.")
        logger.critical("[migrate_protected_fields]  The name of the key in the configuration file is : %s", dst_config.get('protect_fields__encryption_key_name'))
        logger.critical("[migrate_protected_fields]  The name of the key in the database is           : %s", src_config.get('protect_fields__encryption_key_name'))
        logger.critical("[migrate_protected_fields]  Please restore the correct key using the command 'shinken-protected-fields-keyfile-restore'")
        logger.critical("[migrate_protected_fields]  The Protected Field key file was modified.")
        sys.exit(1)
    
    if src_config.get('protect_fields__encryption_keyfile_hash', '') != dst_config.get('protect_fields__encryption_keyfile_hash', '') and src_activated:
        logger.critical("[migrate_protected_fields]  The Protected Field key file was modified")
        logger.critical("[migrate_protected_fields]  The synchronizer will not start in order not to corrupt data.")
        logger.critical("[migrate_protected_fields]  Please restore the correct key named [%s] using the command 'shinken-protected-fields-keyfile-restore'",
                        src_config.get('protect_fields__encryption_key_name', 'UNKNOWN NAME'))
        sys.exit(1)
    
    if (src_activated == dst_activated) and (src_fields == dst_fields):  # Nothing to do
        return False
    
    return True


def _migrate_protected_properties_sources_config(db_cipher_activated):
    db_cipher = DatabaseCipher(True, [], app.conf.protect_fields__encryption_keyfile)
    
    collection = getattr(app.mongodb_db, 'sources-configuration')
    for source in app.sources:
        name = source.get_name()
        source_conf = collection.find_one({'_id': name})
        if source_conf:
            for os_type, fields in source.get_configuration_fields().iteritems():
                for field_name, properties in fields.iteritems():
                    value = source_conf[os_type][field_name]
                    if value and properties['protected']:
                        if db_cipher_activated:
                            value = db_cipher._cipher_value(value)
                        else:
                            value = db_cipher._uncipher_value(value)
                    source_conf[os_type][field_name] = value
            collection.save(source_conf)


def _migrate_protected_properties(src_config, dst_config):
    """ If one config is an empty dict, we suppose no encryption occurs"""
    
    src_fields = set(src_config.get('protect_fields__substrings_matching_fields', []))
    dst_fields = set(dst_config.get('protect_fields__substrings_matching_fields', []))
    
    src_activated = src_config.get('protect_fields__activate_database_encryption', False)
    dst_activated = dst_config.get('protect_fields__activate_database_encryption', False)
    
    if src_activated and not dst_activated:
        # Decrypt all
        dst_fields = set()
    elif not src_activated and dst_activated:
        # Crypt all
        src_fields = set()
    
    fields_to_uncipher = src_fields
    fields_to_cipher = dst_fields
    
    fields_to_uncipher_list = ",".join(list(fields_to_uncipher))
    fields_to_cipher_list = ",".join(list(fields_to_cipher))
    
    logger.debug("[migrate_protected_fields] Fields to uncrypt:[%s]" % fields_to_uncipher)
    logger.debug("[migrate_protected_fields] Fields to encrypt:[%s]" % fields_to_cipher)
    
    db_uncipher = DatabaseCipher(True, fields_to_uncipher_list, app.conf.protect_fields__encryption_keyfile)
    db_cipher = DatabaseCipher(True, fields_to_cipher_list, app.conf.protect_fields__encryption_keyfile)
    
    # Browse all relevant collection
    filter_col_name = set(('configuration', 'data', 'merge_from_sources', 'newelements', 'changeelements'))
    all_collection_names = app.mongodb_db.collection_names(include_system_collections=False)
    collections_to_process = [{'collection_name': c, 'class_name': "%ss" % c.split('-')[-1]} for c in all_collection_names if c.split('-')[0] in filter_col_name]
    collections_to_process.extend([{'collection_name': "%s_confs" % source.source_name, 'class_name': ITEM_TYPE.REMOTE_SYNCHRONIZER} for source in app.sources if source.module_type == 'synchronizer-collector-linker'])
    for collection_to_process in collections_to_process:
        collection_name = collection_to_process['collection_name']
        logger.debug("[migrate_protected_fields] Processing collection:[%s]" % collection_name)
        collection = getattr(app.mongodb_db, collection_name)
        bulk_ops = collection.initialize_unordered_bulk_op()
        class_name = collection_to_process['class_name']
        items = collection.find()
        for item in items:
            item_type = get_type_item_from_class(class_name, item)
            working_item = copy.deepcopy(item)
            item_state = None
            if collection_name.startswith('changeelements'):
                item_state = ITEM_STATE.CHANGES
            try:
                if fields_to_uncipher:
                    working_item = db_uncipher.uncipher(working_item, item_type=item_type, item_state=item_state)
                if fields_to_cipher:
                    working_item = db_cipher.cipher(working_item, item_type=item_type, item_state=item_state)
                if cmp(item, working_item):
                    # item_to_save = copy.deepcopy(working_item)
                    bulk_ops.find({'_id': item['_id']}).replace_one(working_item)
            except (TypeError, ValueError) as e:
                logger.error("[migrate_protected_fields] error in before or after save, item skip")
                logger.print_stack()
        
        try:
            # pymongo splits the operation into multiple bulk executes() according to this link :
            # http://api.mongodb.com/python/2.7rc1/examples/bulk.html#bulk-insert
            bulk_ops.execute()
            logger.debug("[migrate_protected_fields] Bulk Done")
        except BulkWriteError as bulk_exception:
            logger.error("[migrate_protected_fields] BulkWriteError in collection [%s]:[%s]" % (collection.name, pformat(bulk_exception.details)))
        except InvalidOperation as mongo_exception:
            # do not log error when mongo tel us that there is nothing to do
            if not "No operations to execute" in mongo_exception.message:
                logger.error("[migrate_protected_fields] InvalidOperation in collection [%s]:[%s]" % (collection.name, pformat(mongo_exception.message)))


# migration for data pre V02.04.00
# Look for all hosts/hosttemplates/cluster/service/servicetemplate and change the "contacts"
# property into a view/notify/edit lists
def _migrate_hosts_contacts_properties_in_db():
    t0 = time.time()
    for item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS):
        for item_state in (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION):
            to_migrate_items = migration_datamanager.find_items(item_type, item_state, where={'$or': [{'contacts': {'$exists': True}}, {'contact_groups': {'$exists': True}}]})
            if len(to_migrate_items) == 0:
                continue
            logger.info('[migrate_hosts_contacts_properties] DATA MIGRATION   we are migrating old contacts to the view/notification/edit format. item_type:[%s] item_state:[%s] number of objects:[%s]' % (
                item_type, item_state, len(to_migrate_items)))
            migrate_hosts_contacts_properties(app, to_migrate_items)
            for item in to_migrate_items:
                migration_datamanager.save_item(item, item_type=item_type, item_state=item_state)
    logger.info('[migrate_hosts_contacts_properties] DATA MIGRATION   was done in %.2fs' % (time.time() - t0))


def _migrate_contacts_contactgroups_properties_in_db():
    for item_type in (ITEM_TYPE.CONTACTS, ITEM_TYPE.CONTACTTPLS):
        for item_state in (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION):
            _change_properties_name(item_type, item_state, 'contact_groups', 'contactgroups')
        
        for source_name in (source.get_name() for source in app.sources):
            _change_properties_name(item_type, ITEM_STATE.RAW_SOURCES, 'contact_groups', 'contactgroups', item_source=source_name)


def _change_properties_name(item_type, item_state, old_property, new_property, item_source=None):
    if item_source:
        to_migrate_items = migration_datamanager.find_items(item_type, item_state, item_source=item_source, where={old_property: {'$exists': True}})
    else:
        to_migrate_items = migration_datamanager.find_items(item_type, item_state, where={old_property: {'$exists': True}})
    if len(to_migrate_items) == 0:
        return
    t0 = time.time()
    
    for item in to_migrate_items:
        value = item.get(old_property)
        del item[old_property]
        item[new_property] = value
        if item_source:
            migration_datamanager.save_item(item, item_type=item_type, item_state=item_state, item_source=item_source)
        else:
            migration_datamanager.save_item(item, item_type=item_type, item_state=item_state)
    
    logger.info('[Synchronizer] DATA MIGRATION   was done in %.2fs' % (time.time() - t0))


# Migrate all host/cluster in V02.04.00 in V02.05.00 version
def _check_old_host_to_cluster_boolean():
    for item_state in (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION):
        for item_type in (ITEM_TYPE.CLUSTERS, ITEM_TYPE.HOSTS):
            clusters = migration_datamanager.find_items(item_type, item_state, where={'check_command': {'$regex': '^bp_rule'}})
            for cluster in clusters:
                logger.info('[Synchronizer] DATA MIGRATION   we are migrating old cluster element %s with a is_cluster boolean in the area %s' % (cluster.get('host_name', 'UNKNOWN'), item_state))
                migrate_to_clusters(cluster)
                migration_datamanager.save_item(cluster, item_type=ITEM_TYPE.CLUSTERS, item_state=item_state)


# Migrate objects from 2.5 / early 2.6 lacking "is_cluster" property
def _check_clusters_to_cluster_boolean():
    for item_state in (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION):
        for item_type in (ITEM_TYPE.ALL_HOST_CLASS):
            clusters = migration_datamanager.find_items(item_type, item_state, where={'bp_rule': {'$exists': True}, "is_cluster": {'$exists': False}})
            for cluster in clusters:
                logger.info('[Synchronizer] DATA MIGRATION   we are migrating cluster element %s with a is_cluster boolean in the area %s' % (cluster.get('host_name', 'UNKNOWN'), item_state))
                migrate_to_clusters(cluster)
                migration_datamanager.save_item(cluster, item_type=ITEM_TYPE.CLUSTERS, item_state=item_state)


def migrate_to_clusters(item):
    cmd = item.get('check_command')
    if cmd and cmd.startswith('bp_rule'):
        if '!' in cmd:
            args = '!'.join(cmd.split('!')[1:])
            if args:
                item['bp_rule'] = args
        del item['check_command']
    if 'bp_rule' in item:
        item['is_cluster'] = '1'


def _remove_basic_element_from_database():
    # We remove the shinken-host and generic-service now we add it in synchronier-import module so UI configuration didn't see it.
    col = getattr(app.mongodb_db, 'configuration-production-host')
    col.remove({'_id': 'f7127b3a5ae011e58d51080027f08538'})
    col.remove({'name': 'shinken-host'})
    col = getattr(app.mongodb_db, 'configuration-stagging-host')
    col.remove({'_id': 'f7127b3a5ae011e58d51080027f08538'})
    col.remove({'name': 'shinken-host'})
    col = getattr(app.mongodb_db, 'configuration-production-service')
    col.remove({'_id': '57ecf196c6a311e5958f08002709eeca'})
    col = getattr(app.mongodb_db, 'configuration-stagging-service')
    col.remove({'_id': '57ecf196c6a311e5958f08002709eeca'})


def _remove_old_tmp_collection():
    all_collection_names = app.mongodb_db.collection_names(include_system_collections=False)
    for col_name in all_collection_names:
        if col_name.startswith('tmp-'):
            app.mongodb_db.drop_collection(col_name)
