#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.objects.service import PREFIX_LINK_UUID, PREFIX_LINK_DFE_KEY
from ...dao.helpers import split_and_strip_list
from ...dao.def_items import SERVICE_EXCLUDES_BY_ID, ITEM_TYPE, ITEM_STATE
from validator import Validator


class MergeValidator(Validator):
    
    def __init__(self, app, datamanagerV2_for_existing_items, datamanagerV2_for_merged_items):
        super(MergeValidator, self).__init__(app, datamanagerV2_for_existing_items)
        self.datamanagerV2_for_merged_items = datamanagerV2_for_merged_items
        
    def _validate_contacts(self, item, item_type):
        validation = []
        validation.extend(self._rule_ACL_visualisation_UI(item))
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_dollar_in_data(item))
        return validation
    
    
    def _validate_contacttpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_ACL_visualisation_UI(item))
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_dollar_in_data(item))
        return validation
    
    
    def _validate_hosts(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_macros_in_check_command(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._rule_valide_service_excludes_by_id_checks_exists(item, item_type))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        return validation
    
    
    def _validate_hosttpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_macros_in_check_command(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._rule_valide_service_excludes_by_id_checks_exists(item, item_type))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        return validation
    
    
    def _validate_clusters(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._remove_check_command(item))
        validation.extend(self._rule_valide_service_excludes_by_id_checks_exists(item, item_type))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        return validation
    
    
    def _validate_clustertpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._remove_check_command(item))
        validation.extend(self._rule_valide_service_excludes_by_id_checks_exists(item, item_type))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        return validation
    
    
    def _validate_services(self, item, item_type):
        on_check_override = False
        if item_type.startswith('override_'):
            on_check_override = True
            item_type = item_type.replace('override_', '')
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_dollar_in_data(item))
        if not on_check_override:
            validation.extend(self._rule_valide_duplicate_name_for_service(item, item_type))
        validation.extend(self._rule_valide_host_name(item, item_type))
        validation.extend(self._rule_valide_hostgroup_names(item, item_type))
        validation.extend(self._rule_valide_data_duplicate_foreach(item))
        validation.extend(self._rule_valide_macros_in_check_command(item))
        validation.extend(self._rule_valide_flap_threshold(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        if item_type in (ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESCLUSTERTPLS):
            validation.extend(self._rule_valide_complex_expression(item, item_type))
        return validation
    
    
    def _validate_servicetpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_duplicate_name_for_service(item, item_type))
        validation.extend(self._rule_valide_host_name(item, item_type))
        validation.extend(self._rule_valide_hostgroup_names(item, item_type))
        validation.extend(self._rule_valide_data_duplicate_foreach(item))
        validation.extend(self._rule_valide_macros_in_check_command(item))
        validation.extend(self._rule_valide_flap_threshold(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        return validation
    
    
    def _validate_macromodulations(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_data_duplicate_foreach(item))
        return validation
    
    
    def _validate_timeperiods(self, item, item_type):
        validation = []
        return validation
    
    
    def _validate_resultmodulations(self, item, item_type):
        validation = []
        return validation
    
    
    def _validate_businessimpactmodulations(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_business_impact(item, item_type))
        return validation
    
    
    def _validate_commands(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_macro_in_command_line(item))
        validation.extend(self._rule_valide_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        return validation
    
    
    def _validate_notificationways(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_business_impact(item, item_type))
        return validation
    
    
    def _validate_escalations(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_notification_interval_property(item))
        return validation
    
    
    def _validate_contactgroups(self, item, item_type):
        validation = []
        return validation
    
    
    def _validate_hostgroups(self, item, item_type):
        validation = []
        return validation
    
    
    def _remove_check_command(self, item):
        item.pop('check_command', None)
        return []


    def _rule_valide_service_excludes_by_id_checks_exists(self, item, item_type):
        validation = []
        value = item.get(SERVICE_EXCLUDES_BY_ID)
    
        if not value:
            return validation
    
        if item_type == ITEM_TYPE.HOSTS:
            ordered_list_of_types_to_check = (ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESHOSTTPLS)
        elif item_type == ITEM_TYPE.HOSTTPLS:
            ordered_list_of_types_to_check = (ITEM_TYPE.SERVICESHOSTTPLS,)
        elif item_type == ITEM_TYPE.CLUSTERS:
            ordered_list_of_types_to_check = (ITEM_TYPE.SERVICESCLUSTERS, ITEM_TYPE.SERVICESCLUSTERTPLS)
        elif item_type == ITEM_TYPE.CLUSTERTPLS:
            ordered_list_of_types_to_check = (ITEM_TYPE.SERVICESCLUSTERTPLS,)
        else:
            validation.append(self._create_message(self.app._('validator.service_excludes_unmanaged_type') % self.app._("type.%s" % item_type), level=Validator._LEVEL_VALIDATOR_ERROR))
            return validation
    
        formatted_excludes = []
        for exclude in split_and_strip_list(value):
            if PREFIX_LINK_UUID not in exclude:
                validation.append(self._create_message(self.app._('validator.service_excludes_only_by_id') % exclude, level=Validator._LEVEL_VALIDATOR_WARNING))
                return validation
    
            (check_name, check_id) = exclude.split(PREFIX_LINK_UUID, 1)
            check = None
            for check_type in ordered_list_of_types_to_check:
                check = self.datamanagerV2.find_item_by_id(check_id, item_state=ITEM_STATE.STAGGING, item_type=check_type)
                if check:
                    break
            if check:
                if check.get('duplicate_foreach'):
                    validation.append(self._create_message(self.app._('validator.service_excludes_dfe_check') % exclude, level=Validator._LEVEL_VALIDATOR_WARNING))
                else:
                    formatted_exclude = "%s%s" % (PREFIX_LINK_UUID, check_id)
                    formatted_excludes.append(formatted_exclude)
            else:
                validation.append(self._create_message(self.app._('validator.service_excludes_no_check_with_id') % exclude, level=Validator._LEVEL_VALIDATOR_WARNING))
    
        if formatted_excludes:
            item[SERVICE_EXCLUDES_BY_ID] = ",".join(formatted_excludes)
        else:
            item.pop(SERVICE_EXCLUDES_BY_ID)
    
        return validation
