#!/usr/bin/python
# -*- coding: utf-8 -*-


import time
from shinken.log import logger
from shinken.basemodule import SOURCE_STATE

from ...dao.def_items import ITEM_STATE, ITEM_TYPE, METADATA
import shinkensolutions.shinkenjson as json

### Will be populated by the UI with it's own value
app = None


def launch_analyze_batch():
    app.response.content_type = 'application/json'
    analyzer_name = app.request.forms.get('analyzer_name').decode('utf-8', 'ignore')
    hosts_to_analyze = json.loads(app.request.forms.get('hosts_to_analyze').decode('utf-8', 'ignore'))
    overload_parameters = app.request.forms.get('overload_parameters', '{}').decode('utf-8', 'ignore')
    launch_batch_uuid = app.request.forms.get('launch_batch_uuid').decode('utf-8', 'ignore')
    logger.debug('[analyzer] launch_analyze_batch:: %s %s %s' % (hosts_to_analyze, analyzer_name, overload_parameters))
    r = app.launch_analyze_batch(analyzer_name, hosts_to_analyze, overload_parameters, launch_batch_uuid)
    return json.dumps(r)


def stop_analyze_batch():
    app.response.content_type = 'application/json'
    launch_batch_uuid = app.request.forms.get('launch_batch_uuid').decode('utf-8', 'ignore')
    if not launch_batch_uuid:
        return json.dumps(False)
    r = app.stop_analyze_batch(launch_batch_uuid)
    return json.dumps(r)


def get_analyze_jobs_result():
    app.response.content_type = 'application/json'
    launch_batch_uuid = app.request.forms.get('launch_batch_uuid').decode('utf-8', 'ignore')
    source_enabled = app.get_api_sources_from_backend().get('server-analyzer', {}).get('enabled', False)
    result = {'source_is_enabled': source_enabled, 'data': []}
    if not source_enabled or not launch_batch_uuid:
        return json.dumps(result)
    result['data'] = app.get_analyze_jobs_result(launch_batch_uuid)
    return json.dumps(result)


def get_analyze_diffs():
    app.response.content_type = 'application/json'
    job_results = json.loads(app.request.forms.get('job_results').decode('utf-8', 'ignore'))
    analyzer_name = app.request.forms.get('analyzer_name').decode('utf-8', 'ignore')
    
    if not job_results:
        return json.dumps({})
    
    id_by_states = {}
    # get all id's by state
    for result in job_results:
        state = result['object_state']
        if not state in id_by_states:
            id_by_states[state] = []
        id_by_states[state].append(result['object_uuid'])
    
    source = app.source_controller.get_source(analyzer_name)
    if source.enabled and source.state != SOURCE_STATE.CRITICAL:
        # we can search in new and changes with id only for already existing items (aka from stagging state)
        app.api_force_source(analyzer_name)
        iter_count = 0
        max_iter_count = 1200  # 1200 * 0.5 seconds is 10mn
        while app.get_api_source_controller_importing_from_backend() and iter_count < max_iter_count:
            logger.debug('[analyzer] get_analyze_diffs waiting for import and merge to be done')
            iter_count += 1
            time.sleep(0.5)
    stagging_objs = []
    discovery_objs = []
    
    if 'stagging' in id_by_states:
        stagging_objs = app.datamanagerV2.find_merge_state_items(ITEM_TYPE.HOSTS, item_states=[ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING], where={'_id': {'$in': id_by_states['stagging']}}, lookup={'_id': 1})
        stagging_objs = [{'_id': i['_id'], 'private_state': METADATA.get_metadata(i, METADATA.SOURCE_STATE)} for i in stagging_objs]
    if 'discovery' in id_by_states:
        # we need to find the 'real' id's of the items with the given ip
        items = app.datamanagerV2.find_items(ITEM_TYPE.HOSTS, ITEM_STATE.MERGE_SOURCES, where={"_SYNC_KEYS": {'$in': id_by_states['discovery']}})
        # create a mapping id->ip
        ip_for_id = {}
        for item in items:
            for ip in id_by_states['discovery']:
                if ip in item.get('_SYNC_KEYS', ()):
                    ip_for_id[item['_id']] = ip
        # now try to find if there are changes or new for the given id
        changes_objs = app.datamanagerV2.find_merge_state_items(ITEM_TYPE.HOSTS, item_states=[ITEM_STATE.NEW, ITEM_STATE.CHANGES], where={'_id': {'$in': ip_for_id.keys()}}, lookup={'_id': 1})
        # now we need to glue the private_state with the ip and return that
        for change_info in changes_objs:
            ip = ip_for_id[change_info['_id']]
            
            discovery_objs.append({
                '_id'          : ip,
                'private_state': METADATA.get_metadata(change_info, METADATA.SOURCE_STATE)
            })
    
    objs = stagging_objs + discovery_objs
    return json.dumps(objs)


pages = {
    launch_analyze_batch   : {'routes': ['/api/analyzers/launch_analyze_batch'], 'method': 'POST', 'wrappers': ['auth']},
    stop_analyze_batch     : {'routes': ['/api/analyzers/stop_analyze_batch'], 'method': 'POST', 'wrappers': ['auth']},
    get_analyze_jobs_result: {'routes': ['/api/analyzers/get_analyze_jobs_result'], 'method': 'POST', 'wrappers': ['auth']},
    get_analyze_diffs      : {'routes': ['/api/analyzers/get_analyze_diffs'], 'method': 'POST', 'wrappers': ['auth']},
}
