#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import json
from datetime import datetime

from shinkensolutions.lib_checks.common import Result, RaiseOnExitOptionParser, ParseOptionError, EXIT_STATUS, BREAK_LINE, ShinkenUtils, HTMLTag, HTMLList, HTMLTable

VERSION = '0.1'
DAEMON_TYPE = 'broker'

TAG_FOR_STATE = {
    EXIT_STATUS.OK      : HTMLTag.OK,
    EXIT_STATUS.WARNING : HTMLTag.WARNING,
    EXIT_STATUS.CRITICAL: HTMLTag.CRITICAL
}

DATE_FORMAT = '%Y/%m/%d'

result = Result()

parser = RaiseOnExitOptionParser('%prog [options] [--help]', version='%prog ' + VERSION)
parser.add_option('-H', '--hostname', dest='hostname', help='The hostname of the shinken daemon')
parser.add_option('-p', '--port', dest='port', type='int', help='The port of the shinken daemon')
parser.add_option('-t', '--timeout', dest='timeout', type='int', default=3, help='timeout to connect to the shinken daemon. Default : 3')
parser.add_option('-m', '--minutes', dest='minutes_of_stats', type='int', default=1, help='Number of minutes worth of stats displayed. Default : 1 minute')
parser.add_option('-w', '--webui', dest='webui', default='', help='Name of the webui to check. Default : the first one')
parser.add_option('--shinkenversion', dest='shinken_supervisor_version', default='', help='The shinken version number used to compare with the monitored shinken. Mandatory if in shinken mode.')


def _parse_args():
    opts = None
    try:
        opts, args = parser.parse_args()
        if args:
            parser.error('Does not accept arguments.')
        if not opts.hostname:
            parser.error('Missing parameter hostname (-H/--hostname)')
        if not opts.port:
            parser.error('Missing parameter port (-p/--port)')
    except ParseOptionError as e:
        if e.msg:
            result.hard_exit(EXIT_STATUS.CRITICAL, 'Fail to parse command argument: %s %s' % (BREAK_LINE, BREAK_LINE.join(e.msg.split('\n'))))
        exit(0)
    return opts


def get_date_string_from_timestamp(timestamp, date_stirng_format):
    return datetime.fromtimestamp(timestamp).strftime(date_stirng_format)


def check_modules(raw_stats, minutes_of_stats, webui_name):
    webui_module = next((module for module in raw_stats.get('modules_info', []) if module['type'] == 'webui' and (not webui_name or webui_name == module['name'])), None)
    if not webui_module:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - There is no WebUI module on the Broker daemon.' % TAG_FOR_STATE[EXIT_STATUS.CRITICAL])
        return
    
    webui_modules = webui_module.get('modules', None)
    if not webui_modules:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - The WebUI module does not have any submodules.' % TAG_FOR_STATE[EXIT_STATUS.CRITICAL])
        return
    
    sla_module = next((module for module in webui_modules if module['type'] == 'sla'), None)
    if not sla_module:
        result.add_check(EXIT_STATUS.CRITICAL, '%s - There is no SLA submodule on the WebUI module.' % TAG_FOR_STATE[EXIT_STATUS.CRITICAL])
        return
    
    reader_status = EXIT_STATUS.OK
    output = []
    lines = []
    
    module_name = sla_module['name']
    module_stats = sla_module['raw_stats']
    
    database_status = module_stats.get('database_status', None)
    if not database_status:
        result.add_check(EXIT_STATUS.OK if database_status else EXIT_STATUS.CRITICAL, 'Database status: %s' % TAG_FOR_STATE[EXIT_STATUS.OK if database_status else EXIT_STATUS.CRITICAL])
    
    stored_sla_stats = module_stats.get('stored_sla_stats', [])
    # Time, Number, Execution time
    delta_stored_sla_stats = (0, 0, 0)
    if len(stored_sla_stats) >= 2:
        max_index = minutes_of_stats * 6 if len(stored_sla_stats) >= minutes_of_stats * 6 + 1 else len(stored_sla_stats) - 1
        delta_stored_sla_stats = (stored_sla_stats[0][0] - stored_sla_stats[max_index][0], stored_sla_stats[0][1] - stored_sla_stats[max_index][1], stored_sla_stats[0][2] - stored_sla_stats[max_index][2])
    
    raw_sla_stats = module_stats.get('raw_sla_stats', [])
    # Time, Number, Execution time
    delta_raw_sla_stats = (0, 0, 0)
    if len(raw_sla_stats) >= 2:
        max_index = minutes_of_stats * 6 if len(raw_sla_stats) >= minutes_of_stats * 6 + 1 else len(raw_sla_stats) - 1
        delta_raw_sla_stats = (raw_sla_stats[0][0] - raw_sla_stats[max_index][0], raw_sla_stats[0][1] - raw_sla_stats[max_index][1], raw_sla_stats[0][2] - raw_sla_stats[max_index][2])
    
    history_requests = module_stats.get('history_requests', [])
    # Time, Number, Execution time
    delta_history_requests = (0, 0, 0)
    if len(history_requests) >= 2:
        max_index = minutes_of_stats * 6 if len(history_requests) >= minutes_of_stats * 6 + 1 else len(history_requests) - 1
        delta_history_requests = (history_requests[0][0] - history_requests[max_index][0], history_requests[0][1] - history_requests[max_index][1], history_requests[0][2] - history_requests[max_index][2])
    
    widget_requests = module_stats.get('widget_requests', [])
    # Time, Number, Execution time
    delta_widget_requests = (0, 0, 0)
    if len(widget_requests) >= 2:
        max_index = minutes_of_stats * 6 if len(widget_requests) >= minutes_of_stats * 6 + 1 else len(widget_requests) - 1
        delta_widget_requests = (widget_requests[0][0] - widget_requests[max_index][0], widget_requests[0][1] - widget_requests[max_index][1], widget_requests[0][2] - widget_requests[max_index][2])
    
    report_requests = module_stats.get('report_requests', [])
    # Time, Number, Execution time
    delta_report_requests = (0, 0, 0)
    if len(report_requests) >= 2:
        max_index = minutes_of_stats * 6 if len(report_requests) >= minutes_of_stats * 6 + 1 else len(report_requests) - 1
        delta_report_requests = (report_requests[0][0] - report_requests[max_index][0], report_requests[0][1] - report_requests[max_index][1], report_requests[0][2] - report_requests[max_index][2])
    
    # This will be used to get the top 5 requests
    all_requests = module_stats.get('all_requests', [])
    # Time, All requests
    delta_all_requests = (0, [])
    if len(all_requests) >= 2:
        max_index = minutes_of_stats * 6 if len(all_requests) >= minutes_of_stats * 6 + 1 else len(all_requests) - 1
        request_list = []
        for request in all_requests[:max_index]:
            request_list.extend(request[1])
        delta_all_requests = (report_requests[0][0] - report_requests[max_index][0], request_list)
    top_requests = sorted(delta_all_requests[1], key=lambda request: request[2], reverse=True)[:5]
    
    if delta_stored_sla_stats[0] > 0:
        lines.append('Stats for the last %g seconds' % round(delta_stored_sla_stats[0], 2))
    else:
        lines.append('Not enough data, please try again in a minute')
    
    if delta_stored_sla_stats[0] > 0:
        sla_read_lines = [('Stored', '%d' % delta_stored_sla_stats[1] if delta_stored_sla_stats[1] > 0 else '-', '%g seconds' % round(delta_stored_sla_stats[2], 3) if delta_stored_sla_stats[2] > 0 else '-'),
                          ('Raw', '%d' % delta_raw_sla_stats[1] if delta_raw_sla_stats[1] > 0 else '-', '%g seconds' % round(delta_raw_sla_stats[2], 3) if delta_raw_sla_stats[2] > 0 else '-')]
        
        sla_read_table = HTMLTable.table(['SLA type', 'Number read', 'Execution time'], sla_read_lines, 'SLAs read', compact_title=True)
        lines.append(sla_read_table)
    
    if delta_history_requests[0] > 0:
        sla_requests_lines = [('History', '%d' % delta_history_requests[1] if delta_history_requests[1] > 0 else '-', '%g seconds' % round(delta_history_requests[2], 3) if delta_history_requests[2] > 0 else '-'),
                              ('Widget', '%d' % delta_widget_requests[1] if delta_widget_requests[1] > 0 else '-', '%g seconds' % round(delta_widget_requests[2], 3) if delta_widget_requests[2] > 0 else '-'),
                              ('Report', '%d' % delta_report_requests[1] if delta_report_requests[1] > 0 else '-', '%g seconds' % round(delta_report_requests[2], 3) if delta_report_requests[2] > 0 else '-')]
        
        sla_requests_table = HTMLTable.table(['Request type', 'Number of calls', 'Execution time'], sla_requests_lines, 'SLA requests from WebUI module(s)', compact_title=True)
        lines.append(sla_requests_table)
    
    table_lines = []
    for data in top_requests:
        table_lines.append(('<span style="margin-left:10px;">' + data[0] + '<br /><div style="font-size: 10px;margin-left:10px">endpoint: [<div style="font-style:italic;color:gray;margin-left:10px;">' + data[1] + '</div>]</div></span>',
                            '<span style="margin-left:10px;">%gms</span>' % round(data[2] * 1000, 2)))
    if len(table_lines) > 0:
        sla_build_times_table = HTMLTable.table(['Request type', 'Time taken'], table_lines, 'Top %d requests' % len(table_lines), compact_title=True)
        sla_build_times_table = sla_build_times_table.replace("skn-ict", "skn-ict ctm-tbl")
        lines.append(sla_build_times_table)

    oldest_sla_date = module_stats.get('oldest_sla_date', None)
    if oldest_sla_date:
        oldest_sla_date = get_date_string_from_timestamp(oldest_sla_date, DATE_FORMAT)
        lines.append('Date the oldest SLA\'s data was stored: %s' % oldest_sla_date)
    
    short_output_lines = [('Stored SLAs', '%d' % delta_stored_sla_stats[1] if delta_stored_sla_stats[1] > 0 else '-', '%g seconds' % round(delta_stored_sla_stats[2], 3) if delta_stored_sla_stats[2] > 0 else '-'),
                          ('Today\'s RAW SLAs', delta_raw_sla_stats[1] if delta_raw_sla_stats[1] > 0 else '-', '%g seconds' % round(delta_raw_sla_stats[2], 3) if delta_raw_sla_stats[2] > 0 else '-'),
                          ('History', delta_history_requests[1] if delta_history_requests[1] > 0 else '-', '%g seconds' % round(delta_history_requests[2], 3) if delta_history_requests[2] > 0 else '-'),
                          ('Widget', delta_widget_requests[1] if delta_widget_requests[1] > 0 else '-', '%g seconds' % round(delta_widget_requests[2], 3) if delta_widget_requests[2] > 0 else '-'),
                          ('Report', delta_report_requests[1] if delta_report_requests[1] > 0 else '-', '%g seconds' % round(delta_report_requests[2], 3) if delta_report_requests[2] > 0 else '-')]
    output.append(HTMLTable.table(['Request type', 'Count', 'Execution time'], short_output_lines, '<br /><span style="font-weight: bold;">Request times in the last %g seconds</span>' % round(delta_stored_sla_stats[0], 2), compact_title=True,
                                  extra_tags='style="max-width: 360px"'))
    
    output.append('<style type="text/css">.ctm-tbl td:nth-child(2),.ctm-tbl th:nth-child(2){width:125px !important}.ctm-tbl th:first-child{width:auto !important}</style>')
    long_output = HTMLTable.table([], [[HTMLList.simple_list(lines)]], left_headers=['%s - Reader:<br>%s' % (module_name.upper(), TAG_FOR_STATE[reader_status])], all_col_same_width=False)
    
    if reader_status == EXIT_STATUS.CRITICAL:
        output.insert(0, 'Module is in a critical state.')
    elif reader_status == EXIT_STATUS.WARNING:
        output.insert(0, 'Module has some anomalies')
    elif reader_status == EXIT_STATUS.OK:
        output.insert(0, 'Module is working as intended.')
    result.add_check(status=reader_status, output=''.join(output), long_output=long_output, title=True)


def main():
    opts = _parse_args()
    daemon_adr = opts.hostname
    daemon_port = opts.port
    timeout = opts.timeout
    minutes_of_stats = opts.minutes_of_stats
    shinken_supervisor_version = opts.shinken_supervisor_version
    webui_name = opts.webui
    
    html, connection_time = ShinkenUtils.request_get_daemon(result, DAEMON_TYPE, '%s:%s' % (daemon_adr, daemon_port), '/get_raw_stats', timeout=timeout)
    raw_stats = json.loads(html)
    ShinkenUtils.minimal_check(result, raw_stats, DAEMON_TYPE, shinken_supervisor_version)
    
    check_modules(raw_stats, minutes_of_stats, webui_name)
    result.exit()


if __name__ == '__main__':
    main()
