#!/usr/bin/env python
# -*-coding:utf-8-*-
# Copyright (C) 2012:
#    Romain Forlot, rforlot@yahoo.com
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import traceback
import time
import sys
import logging
import smtplib
from datetime import datetime, timedelta
import socket
from optparse import OptionParser, OptionGroup
from email.mime.image import MIMEImage
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from shinkensolutions.shinken_time_helper import print_human_readable_period, DisplayFormat
import jinja2
import cgi
import urllib

# Global var
shinken_icon_dir = '/var/lib/shinken/libexec/notifications/assets/img/mail'
host_states = ["UP", "DOWN", "DOWN", "UNKNOWN"]
check_states = ["OK", "WARNING", "CRITICAL", "UNKNOWN"]

EXECUTION_ERROR_PATH = '/var/log/shinken/failed_notifications.log'


# Set up root logging
def setup_logging():
    log_level = logging.INFO
    if opts.debug:
        log_level = logging.DEBUG
    if opts.logfile:
        logging.basicConfig(filename=opts.logfile, level=log_level, format='%(asctime)s:%(levelname)s: %(message)s')
    else:
        logging.basicConfig(level=log_level, format='%(asctime)s:%(levelname)s: %(message)s')


mail_welcome = 'Monitoring Notification'
mail_format = {'html': MIMEMultipart('related')}


# Translate a comma separated list of mail recipient into a python list
def make_receivers_list(receivers):
    if ',' in receivers:
        ret = receivers.split(',')
    else:
        ret = [receivers]
    
    return ret


# This just create mail skeleton and doesn't have any content.
def create_mail():
    # Fill SMTP header and body.
    msg = mail_format['html']
    logging.debug('From: %s' % opts.sender)
    msg['From'] = opts.sender
    logging.debug('To: %s' % (opts.receivers))
    msg['To'] = opts.receivers
    with open(opts.title_tpl) as f:
        text_tpl = f.read()
        tpl = jinja2.Template(text_tpl)
        subject = tpl.render(shinken_var=shinken_var)
        logging.debug('Subject: %s' % subject)
        msg['Subject'] = subject
    
    return msg


def add_image(path, mail, cid):
    with open(path, 'rb') as f:
        msgImage = MIMEImage(f.read())
        msgImage.add_header('Content-ID', '<%s>' % cid)
        mail.attach(msgImage)


#############################################################################
# Html creation lair
#############################################################################

def create_html_message(msg, template):
    # Get url and add it in footer
    with open(template) as f:
        text_tpl = f.read()
        tpl = jinja2.Template(text_tpl)
        html_content = tpl.render(shinken_var=shinken_var, mail_welcome=mail_welcome)
    
    # Make final string var to send and encode it to stdout encoding
    # avoiding decoding error.
    
    try:
        html_msg = html_content.encode('utf-8')
    except UnicodeDecodeError:
        logging.debug('Content is Unicode encoded.')
        html_msg = html_content
    
    logging.debug('HTML string: %s' % html_msg)
    
    msgAlternative = MIMEMultipart('alternative')
    msg.attach(msgAlternative)
    
    msgText = MIMEText(html_msg, 'html', 'utf-8')
    msgAlternative.attach(msgText)
    logging.debug('Mail object: %s' % msg)
    
    return msg


def dump_error(exp):
    err = '%s(%s)' % (type(exp), str(exp))
    with open(EXECUTION_ERROR_PATH, 'a') as f:
        f.write('\n******\nNew notification error:(%s)\n' % datetime.fromtimestamp(time.time()).strftime('%Y-%m-%d %H:%M:%S'))
        f.write('%s\n' % err)
        f.write('%s\n' % traceback.format_exc())
        f.write('\n******\n')
    return err


if __name__ == "__main__":
    try:
        parser = OptionParser(description='Notify by email receivers of Shinken alerts. Message will be formatted in html and can embed customer logo. To included customer logo, just load png image named customer_logo.png in ' + shinken_icon_dir)
        
        group_debug = OptionGroup(parser, 'Debugging and test options', 'Useful to debug script under shinken processes. Useful to just make a standalone test of script to see what it looks like.')
        group_general = OptionGroup(parser, 'General options', 'Default options to setup')
        group_shinken = OptionGroup(parser, 'Shinken macros to specify.',
                                    'Used to specify usual shinken macros in notifications, if not specified then it will try to get them from environment variable. You need to enable_environment_macros in shinken.cfg if you want to used them. It isn\'t recommended to use environment macros for large environments. You \'d better use options -c and -s or -h depending on which object you\'ll notify for.')
        
        # Debug and test options
        group_debug.add_option('-D', '--debug', dest='debug', default=False, action='store_true', help='Generate a test mail message')
        group_debug.add_option('-l', '--logfile', dest='logfile', help='Specify a log file. Default: log to stdout.')
        
        # General options
        group_general.add_option('-r', '--receivers', dest='receivers', help='Mail recipients comma-separated list')
        group_general.add_option('-F', '--sender', dest='sender', help='Sender email address, default is system user')
        group_general.add_option('-S', '--SMTP', dest='smtp', default='localhost', help='Target SMTP hostname. None for just a sendmail lanch. Default: localhost')
        group_general.add_option('--title-tpl', dest='title_tpl', help='Mail subject template')
        group_general.add_option('--content-tpl', dest='content_tpl', help='Template for email body')
        
        # Shinken options
        
        group_debug.add_option('--with-images', dest='images', default=False, action='store_true', help='Activates logo and status images in mail. Disabled by default')
        group_shinken.add_option('--address', dest='address', help='Host network address')
        group_shinken.add_option('-n', '--notif', dest='notification', help='Notification type, PROBLEM or RECOVERY')
        group_shinken.add_option('-H', '--hostname', dest='hostname', help='The name of the host')
        group_shinken.add_option('--url', dest='url', help='The url of the web interface')
        group_shinken.add_option('--huuid', dest='hostuuid', help='Host UUID')
        group_shinken.add_option('--serviceuuid', dest='serviceuuid', help='Service UUID')
        group_shinken.add_option('--last-check', dest='date', help='Date and time of the check')
        group_shinken.add_option('--state', dest='state', help='Check or host state')
        group_shinken.add_option('--last-state', dest='laststate', help='Check or host previous state')
        group_shinken.add_option('--last-change', dest='lastchange', help='Last state change timestamp')
        group_shinken.add_option('--check', dest='desc', help='Check description')
        group_shinken.add_option('--output', dest='output', help='Check output')
        group_shinken.add_option('--long-output', dest='long_output', help='Check long output, if available')
        group_shinken.add_option('--ack-author', dest='ackauthor', help='Author of the acknowledge')
        group_shinken.add_option('--ack-data', dest='ackdata', help='Acknowledge comment')
        group_shinken.add_option('--downtime-comment', dest='downtimecomment', help='Downtime comment')
        group_shinken.add_option('--downtime-author', dest='downtimeauthor', help='Downtime author')
        group_shinken.add_option('--first-notification-delay', dest='first_notif_delay', help='Number of minutes waited before first notification')
        group_shinken.add_option('--notification-number', dest='notif_number', help='Current number of notifications for this event')
        group_shinken.add_option('--realm', dest='realm', help='Realm of the host/check to notify')
        
        # Shinken WebUI options
        
        parser.add_option_group(group_debug)
        parser.add_option_group(group_general)
        parser.add_option_group(group_shinken)
        
        (opts, args) = parser.parse_args()
        
        setup_logging()
        
        # Check and process arguments
        #
        # Retrieve and setup shinken macros that make the mail content
        template = opts.content_tpl
        serviceuuid = opts.serviceuuid or ""
        if serviceuuid:
            list_filter = """[{"service_description":"%s"}]""" % opts.desc
            element_link = "%s/static/ui/index.html#/hosts/%s/checks/detail/%s?listDetailFilter=%s" % (opts.url, opts.hostuuid, serviceuuid, urllib.quote_plus(list_filter))
        else:
            element_link = "%s/static/ui/index.html#/hosts/%s/summary" % (opts.url, opts.hostuuid)
        state_duration = ''
        if opts.lastchange:
            _offset = timedelta(seconds=int(float(time.time())) - int(float(opts.lastchange)))
            _date_last_change = datetime.fromtimestamp(int(float(opts.lastchange))).strftime("%d/%m/%Y %H:%M:%S")
            _offset_hours_minute = print_human_readable_period(_offset.seconds, time_format='auto', display_format=DisplayFormat('', '%02ds', '%02dm', '%02dh', '%s days '), separator='')
            _offset_hours_minute = _offset_hours_minute if _offset_hours_minute else 'now'
            if _offset.days:
                state_duration = '%s days %s (%s) ' % (_offset.days, _offset_hours_minute, _date_last_change)
            else:
                state_duration = '%s (%s)' % (_offset_hours_minute, _date_last_change)
        
        shinken_var = {"Notification type"  : opts.notification,
                       "Host address"       : opts.address,
                       "Hostname"           : opts.hostname,
                       "Check date"         : opts.date,
                       "State"              : opts.state,
                       "State duration"     : state_duration,
                       "Output"             : opts.output,
                       "Long output"        : opts.long_output,
                       "Check name"         : opts.desc,
                       "View"               : element_link,
                       "Last state"         : "",
                       "Logo"               : "",
                       "Arrow"              : "",
                       "State logo"         : "",
                       "Title notification" : opts.notification if opts.notification not in ('PROBLEM', 'RECOVERY') else opts.state,
                       "Last state logo"    : "",
                       "Acknowledge author" : opts.ackauthor,
                       "Acknowledge data"   : opts.ackdata,
                       "downtime_author"    : opts.downtimeauthor,
                       "downtime_comment"   : opts.downtimecomment,
                       "Notification delay" : int(opts.first_notif_delay or 0),
                       "Notification number": int(opts.notif_number or 0),
                       "Realm"              : opts.realm or ""
                       }
        
        for (input, v) in shinken_var.items():
            if isinstance(v, basestring):
                v = v.decode('string_escape')
                if isinstance(v, str):
                    v = v.decode('utf8', 'ignore')
                shinken_var[input] = v.replace('\#', '#')
                if input not in ("Output", "Long output") and shinken_var[input]:
                    shinken_var[input] = cgi.escape(shinken_var[input])
        
        shinken_var["Images enabled"] = opts.images
        if opts.receivers == None:
            logging.error('You must define at least one mail recipient using -r')
            sys.exit(5)
        else:
            contactemail = opts.receivers
        receivers = make_receivers_list(opts.receivers)
        
        logging.debug('Create mail skeleton')
        mail = create_mail()
        
        for state in ('State', 'Last state'):
            if shinken_var[state] in ('CRITICAL', 'DOWN'):
                shinken_var['state_html'] = u"<span style='color:#dc2020;'>[CRITICAL]</span>"
            elif shinken_var[state] in ('OK', 'UP'):
                shinken_var['state_html'] = u"<span style='color:#2a9a3d;'>[OK]</span>"
            elif shinken_var[state] == 'WARNING':
                shinken_var['state_html'] = u"<span style='color:#e48c19;'>[WARNING]</span>"
        
        if not opts.images:
            if shinken_var['Check name']:
                shinken_var['Last state'] = check_states[int(opts.laststate)]
            else:
                shinken_var['Last state'] = host_states[int(opts.laststate)]
        
        logging.debug('Create mail content')
        
        mail = create_html_message(mail, template)
        
        # ADD IMAGES
        if opts.images:
            add_image(shinken_icon_dir + '/logos/enterprise_black_154x66.png', mail, 'logo')
            add_image(shinken_icon_dir + '/icons/arrow.png', mail, 'arrow')
            if shinken_var['State'] in ('OK', 'UP'):
                add_image(shinken_icon_dir + '/icons/state_ok_32x32.png', mail, 'state_logo')
            
            elif shinken_var['State'] == 'WARNING':
                add_image(shinken_icon_dir + '/icons/state_warning_32x32.png', mail, 'state_logo')
            
            else:
                add_image(shinken_icon_dir + '/icons/state_critical_32x32.png', mail, 'state_logo')
            
            if opts.laststate == '0':
                add_image(shinken_icon_dir + '/icons/state_ok_32x32.png', mail, 'last_state_logo')
            
            elif opts.laststate == '1' and shinken_var['Check name']:
                add_image(shinken_icon_dir + '/icons/state_warning_32x32.png', mail, 'last_state_logo')
            
            elif opts.laststate == '3' and shinken_var['Check name']:
                add_image(shinken_icon_dir + '/icons/state_unknown_32x32.png', mail, 'last_state_logo')
            
            else:
                add_image(shinken_icon_dir + '/icons/state_critical_32x32.png', mail, 'last_state_logo')
    except Exception as exp:
        err = dump_error(exp)
        print "ERROR: cannot send the email: %s (see more information in the file /var/log/shinken/failed_notifications.log on the reactionner server)" % (err)
        sys.exit(2)
    
    # Use SMTP or sendmail to send the mail ...
    logging.debug('Connect to %s smtp server' % (opts.smtp))
    try:
        smtp = smtplib.SMTP(opts.smtp)
        logging.debug('Send the mail')
        smtp.sendmail(opts.sender, receivers, mail.as_string())
        logging.info("Mail sent successfully")
    
    except smtplib.SMTPHeloError:
        print "ERROR : The server didn't reply properly to the helo greeting"
        sys.exit(2)
    except smtplib.SMTPRecipientsRefused:
        print "ERROR : The server rejected ALL recipients (no mail was sent)"
        sys.exit(2)
    except smtplib.SMTPSenderRefused:
        print "ERROR : The server didn't accept the from_addr"
        sys.exit(2)
    except smtplib.SMTPDataError:
        print "ERROR : The server replied with an unexpected error code (other than a refusal of a recipient)"
        sys.exit(2)
    except socket.timeout as e:
        print 'ERROR : Can not connect to stmp server at "%s" : Timeout' % opts.smtp
        sys.exit(2)
    except socket.error:
        print 'ERROR : Can not connect to stmp server at "%s"' % opts.smtp
        sys.exit(2)
    except Exception as exp:
        err = dump_error(exp)
        print "ERROR: cannot send the email to smtp server \"%s\" : %s (see more information in the file /var/log/shinken/failed_notifications.log on the reactionner server)" % (opts.smtp, err)
        sys.exit(2)
