#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import logging

from component.sla_archive import SLAArchive
from component.sla_common import shared_data
from component.sla_component_manager import ComponentManager
from component.sla_compute_percent_sla import ComputePercentSla
from component.sla_database import SLADatabase
from component.sla_database_connection import SLADatabaseConnection
from component.sla_info import SLAInfo
from component.sla_migration import SLAMigration
from component.sla_reader import SLAReader
from component.sla_reader_stats import SLAReaderStats
from shinken.basemodule import BaseModule
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinkensolutions.date_helper import date_now, get_previous_date


class SLAModuleWebUI(BaseModule):
    
    def __init__(self, configuration, display_info=True):
        # type: (ShinkenModuleDefinition, bool) -> None
        BaseModule.__init__(self, configuration)
        if not display_info:
            self.logger.set_level(logging.ERROR)
        
        self.logger.info('Creating new SLA module name : %s for webui' % (configuration.get_name()))
        
        self.component_manager = ComponentManager(self.logger)
        self.sla_database_connection = SLADatabaseConnection(configuration, self.component_manager)
        self.compute_percent_sla = ComputePercentSla(configuration, self.component_manager)
        self.sla_info = SLAInfo(configuration, self.component_manager, self.sla_database_connection)
        self.sla_database = SLADatabase(configuration, self.component_manager, self.sla_database_connection, self.sla_info)
        self.sla_migration = SLAMigration(configuration, self.component_manager, self.sla_info)
        self.sla_archive = SLAArchive(configuration, self.component_manager, self.sla_info, self.compute_percent_sla, self.sla_database)
        self.sla_reader = SLAReader(configuration, self.component_manager, self.compute_percent_sla, self.sla_archive, self.sla_migration, self.sla_database, self.sla_info)
        self.sla_reader_stats = SLAReaderStats(configuration, self.component_manager, self.sla_info, self.sla_reader)
        
        shared_data.set_default_values(getattr(configuration, 'configuration_default_value', {}))
    
    
    def init(self):
        self.component_manager.init()
    
    
    def get_state(self):
        status = BaseModule.get_state(self)
        status['raw_stats'] = self.sla_reader_stats.get_raw_stats()
        return status
    
    
    def compute_sla_week_info(self, last_date, item_uuid, day_ask=7):
        now_date = date_now()
        cursor_date = now_date
        current_sal_thresholds = self.sla_info.get_sla_thresholds(item_uuid)
        week_info = {'by_days': [], 'thresholds': current_sal_thresholds}
        week_sum = 0
        week_sum_day = 0
        
        for day_count in range(0, day_ask):
            archive = self.sla_reader.read_archive(item_uuid, cursor_date)
            
            _total = archive['sla_total']
            if archive.get('missing', False) and archive.get('in_past', False):
                _percent = None
            elif _total == 0:
                # If the day is not in past and it not have a sla_total it mean that the day don't count
                _percent = None
            else:
                _percent = float(archive.get('sla_ok', 0)) / _total
                week_sum += _percent
                week_sum_day += 1
            
            if day_count == 1:
                week_info['yesterday'] = _percent
            if day_count == 0:
                week_info['today'] = _percent
            week_info['by_days'].append({'yday': cursor_date.yday, 'year': cursor_date.year, 'percent': _percent, 'thresholds': archive.get('thresholds', current_sal_thresholds)})
            
            cursor_date = get_previous_date(cursor_date)
        
        if week_sum_day:
            week_info['week'] = week_sum / week_sum_day
        else:
            week_info['week'] = None
        
        # self.logger.debug('week_info:[%s]' % (week_info))
        return week_info
    
    
    def get_first_monitoring_start_time(self):
        return self.sla_info.get_first_monitoring_start_time()
    
    
    def get_monitoring_start_time(self, item_uuid):
        return self.sla_info.get_monitoring_start_time(item_uuid)
    
    
    def read_archive(self, item_uuid, date=None, date_ranges=None, limit_ranges=None, where=None, lookup=None, only_existing_day=False):
        return self.sla_reader.read_archive(item_uuid, date=date, date_ranges=date_ranges, limit_ranges=limit_ranges, where=where, lookup=lookup, only_existing_day=only_existing_day)
    
    
    @staticmethod
    def get_default_sla_thresholds():
        return shared_data.get_default_sla_thresholds()
    
    
    def get_sla_thresholds(self, item_uuid):
        return self.sla_info.get_sla_thresholds(item_uuid)
    
    
    def find_downtimes(self, downtime_ids):
        return self.sla_database.find_downtimes(downtime_ids)
    
    
    def find_acknowledge(self, acknowledge_id):
        return self.sla_database.find_acknowledge(acknowledge_id)
    
    
    def send_stat_to_reader_stats(self, data):
        self.sla_reader_stats.add_stat_to_request_list(data)
