#!/usr/bin/python

# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import inspect
import re
import urllib
from shinken.log import logger
from shinken.misc.filter import only_related_to, is_authorized

### Will be populated by the UI with it's own value
app = None

try:
    import json
except ImportError:
    # For old Python version, load
    # simple json (it can be hard json?! It's 2 functions guy!)
    try:
        import simplejson as json
    except ImportError:
        print "Error: you need the json or simplejson module"
        raise


# Sort function use to sort files in a natural order, by lowering
# characters, and manage multi levels of integers (tome 1/ page 1.jpg, etc etc)
def natural_key(string_):
    string_ = string_[0]
    l = []
    for s in re.split(r'(\d+)', string_):
        if s.isdigit():
            l.append(int(s))
        else:
            l.append(s.lower())
    return l


def lookup(filter=''):
    user = app.get_user_auth()
    
    name_8 = filter.lower().decode('utf8', 'ignore')
    with_checks_only = (app.request.GET.get('with_checks_only', '0') == '1')
    
    filtered_hosts = only_related_to(app.datamgr.get_hosts(), user, app)
    hnames = ((h.host_name, h.uuid) for h in filtered_hosts if not with_checks_only or len(h.services) > 0)
    r = [n for n in hnames if name_8 in n[0].lower()]
    r.sort(key=natural_key)
    
    return json.dumps(r)


def lookup_service(uuid=''):
    user = app.get_user_auth()
    
    h = app.datamgr.get_host_by_uuid(uuid)
    if not h:
        return json.dumps([])
    
    if not is_authorized(h, user, app):
        return json.dumps([])
    
    r = [(s.service_description, "%s-%s" % (h.uuid, s.uuid)) for s in h.services]
    r.sort(key=natural_key)
    
    return json.dumps(r)


def lookup_post():
    user = app.get_user_auth()
    name = app.request.forms.get('value').lower()
    if not name or len(name) < 3:
        print "Lookup POST %s too short, bail out" % name
        return []
    
    filtered_hosts = only_related_to(app.datamgr.get_hosts(), user, app)
    hnames = ((h.host_name, h.uuid) for h in filtered_hosts)
    r = [n for n in hnames if name in n[0].lower()]
    r.sort(key=natural_key)
    return json.dumps(r)


def get_uuid():
    app.response.content_type = 'application/json'
    
    hname = app.request.GET.get('hname', '')
    sdesc = app.request.GET.get('sdesc', '')
    if not hname:
        return app.abort(400, '[get_uuid] Please give a hname')
    
    error, uuid = _get_uuid(hname, sdesc)
    if error:
        return app.abort(error['code'], error['msg'])
    
    return json.dumps({'uuid': uuid})


def _get_uuid(hname, sdesc):
    uuid = ''
    error = {}
    caller_name = inspect.stack()[1][3]
    if sdesc:
        item_ask = app.datamgr.get_service(hname, sdesc)
    else:
        item_ask = app.datamgr.get_host(hname)
    
    if item_ask:
        uuid = item_ask.get_instance_uuid()
    else:
        logger.info('[%s] No item found with param (hname:%s,sdesc:%s)' % (caller_name, hname, sdesc))
        error = {'code': 404, 'msg': '[%s] No item found with param (hname:%s,sdesc:%s)' % (caller_name, hname, sdesc)}
    return error, uuid


def get_page_detail(name):
    if '/checks/' in name:
        elts = name.split('/checks/', 1)
        hname = elts[0]
        sdesc = elts[1]
    else:
        hname = name.split('/')[0]
        sdesc = ''
    
    logger.debug('[get_page_detail] get for [%s-%s]' % (hname, sdesc))
    
    if not hname:
        return app.abort(400, '[get_page_detail] Please give a hname')
    
    error, uuid = _get_uuid(hname, sdesc)
    if error:
        uuid = 'UNKNOWN_HOST'
    
    url = ''
    if '-' in uuid:
        uuid_host = uuid.split('-')[0]
        uuid_check = uuid
        
        if hasattr(sdesc, 'decode') and type(sdesc) != unicode:
            sdesc = sdesc.decode('utf-8')
        check_filter = u'[{"service_description": "%s"}]' % sdesc
        check_filter = urllib.quote_plus(check_filter.encode('utf-8'))
        url = "/static/ui/index.html#/hosts/%s/checks/detail/%s?listDetailFilter=%s" % (uuid_host, uuid_check, check_filter)
    else:
        url = "/static/ui/index.html#/hosts/%s/summary" % uuid
    
    app.bottle.redirect(url)


pages = {
    lookup         : {'routes': ['/lookup/:filter', '/lookup/'], 'wrappers': ['auth', 'json']},
    lookup_service : {'routes': ['/lookup-service/:uuid'], 'wrappers': ['auth', 'json']},
    lookup_post    : {'routes': ['/lookup'], 'method': 'POST', 'wrappers': ['auth', 'json']},
    get_uuid       : {'routes': ['/get_uuid'], 'wrappers': ['json']},
    get_page_detail: {'routes': ['/detail-by-name/:name#.+#'], 'method': 'GET', 'wrappers': []}
}
