#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-end of time:
#
# This file is part of Shinken Enterprise, all rights reserved.

import os
import os.path
import shutil
import json
import traceback
import time

from django.shortcuts import HttpResponse
from django.conf import settings
from django.utils.log import getLogger

from graphite.storage import CACHE_INVALIDATION_FILE

logger = getLogger()


# Runs the migration between names and uids.
# HTTP Body expected : a json array, whose each element is a sub-array [src path, dst_path]
def do_migration(request):
    error_count = 0
    ok_count = 0
    nb_migrated = 0
    
    # As we got a new configuration we should reset the cache invalidation file to let all other
    # workers know that they must drop/reset their caches
    try:
        f = open(CACHE_INVALIDATION_FILE, 'w')
        f.write('# graphite cache invalidation file. Do not edit it.\nLast update at %s.\n' % int(time.time()))
        f.close()
        logger.log('The cache invalidation file %s is updated' % CACHE_INVALIDATION_FILE)
    except Exception as exp:
        retval = {
            'error_count': 1,
            'ok_count'   : 0,
            'nb_migrated': 0,
            'last_error' : 'cannot update the cache invalidation file: %s: %s' % (CACHE_INVALIDATION_FILE, exp),
        }
        HttpResponse(status=200, content=json.dumps(retval), mimetype='application/json')
    
    migration_request = json.loads(request.body)
    last_error = ''
    for migration_data in migration_request:
        src_path = migration_data[0].replace('.', os.path.sep)
        dst_path = migration_data[1].replace('.', os.path.sep)
        try:
            path_prefix = settings.WHISPER_DIR
            for src_path_part, dst_path_part in zip(os.path.split(src_path), os.path.split(dst_path)):
                current_src_path = os.path.join(path_prefix, src_path_part)
                current_dst_path = os.path.join(path_prefix, dst_path_part)
                
                if not os.path.exists(current_dst_path):
                    if os.path.isdir(current_src_path) and src_path_part != dst_path_part:
                        # print "Move from %s to %s" % (current_src_path, current_dst_path)
                        shutil.move(current_src_path, current_dst_path)
                        nb_migrated += 1
                path_prefix = current_dst_path
            
            ok_count += 1
        except OSError:
            error_count += 1
            last_error = traceback.format_exc()
    # Give back restults with stats about what we did, if it fail or such things
    retval = {
        'error_count': error_count,
        'ok_count'   : ok_count,
        'nb_migrated': nb_migrated,
        'last_error' : last_error,
    }
    return HttpResponse(status=200, content=json.dumps(retval), mimetype='application/json')
