#!/usr/bin/python

# Copyright (C) 2013:
#    Gabes Jean, j.gabes@shinken-solutions.com
#
# This file is part of Shinken Enterprise, all rights reserved.

import json
import uuid
import optparse
import os
import sys
import tarfile
import traceback
from cStringIO import StringIO
from pymongo.connection import Connection
import logging
import datetime
import re

try:
    import shinken
except ImportError:
    print 'Cannot import shinken lib, please install it before launching this tool'
    sys.exit(2)

from shinken.log import logger
from shinken.synchronizer.dao.datamanagerV2 import add_history_info

LOG_FILENAME = '/var/log/shinken/focus-backup.log'
CONTEXT_PATH = '/var/lib/shinken/context.json'
VERSION = '2.03.02'

if os.path.exists(CONTEXT_PATH):
    context = json.loads(open(CONTEXT_PATH, 'r').read())
    VERSION = context.get('current_version', VERSION).split('-')[0]

logger.setLevel('INFO')

if not os.getuid() == 0:
    print "ERROR: this script must be run as root"
    sys.exit(2)

# ENV variable to force unset at all cost
unset_variables = ['http_proxy', 'https_proxy', 'ftp_proxy']
for v in os.environ:
    if v.lower() in unset_variables:
        os.environ[v] = ''

success = {}
warnings = {}
errors = {}

parts = set()  # list of all parts possible

licence_data = None

# Please put them in the GOOD order for application if need!
ALL_FIX = []


def add_success(s, part='unknown'):
    parts.add(part)
    if part not in success:
        success[part] = []
    success[part].append(s)


def add_warning(s, part='unknown'):
    parts.add(part)
    if part not in warnings:
        warnings[part] = []
    warnings[part].append(s)


def get_local_daemons():
    if not os.path.exists('/var/lib/shinken/context.json'):
        return ['central', 'poller', 'synchronizer', 'scheduler', 'reactionner', 'receiver', 'broker']
    f = open('/var/lib/shinken/context.json', 'r')
    context = json.loads(f.read())
    local_daemons = [k for (k, installed) in context['daemons'].iteritems() if installed]
    return local_daemons


def is_daemon_node(dtype):
    local_daemons = get_local_daemons()
    if 'central' in local_daemons or dtype in local_daemons:
        return True
    return False


def cut_line(s):
    size = 80
    lines = []
    words = s.split(' ')
    cline = ''
    for word in words:
        # Maybe this is too large already, put the line
        if len(cline) + len(word) > size:
            lines.append(cline)
            cline = ''
        cline += ' ' + word
    lines.append(cline)
    return lines


def colorize(s, color):
    c = {'OK': 32, 'WARNING': 35, 'FOCUS': 35, 'ERROR': 31}.get(color, 0)
    return '\033[%dm%s\033[0m' % (c, s)


def print_line(title, color, s):
    lines = cut_line(s)
    title = '%-10s' % title
    print ' ' * 4 + '\033[%dm%s\033[0m %s' % (color, title, lines[0])
    if len(lines) > 1:
        for line in lines[1:]:
            print ' ' * 5 + ' ' * len(title) + line


def show_warnings(part):
    for (p, l) in warnings.iteritems():
        if p != part:
            continue
        title = 'AT RISK: '
        color = 35
        for s in l:
            print_line(title, color, s)


def add_error(s, part='unknown'):
    parts.add(part)
    if part not in errors:
        errors[part] = []
    errors[part].append(s)


def show_errors(part):
    for (p, l) in errors.iteritems():
        if p != part:
            continue
        title = 'ERROR: '
        color = 31
        for s in l:
            print_line(title, color, s)


def show_success(part):
    for (p, l) in success.iteritems():
        if p != part:
            continue
        title = 'OK: '
        color = 32
        for s in l:
            print_line(title, color, s)


orig_stdout = sys.stdout


class LogFile(object):
    def __init__(self, name=None):
        self.logger = logging.getLogger(name)
        self.logger.setLevel(logging.DEBUG)
        handler = logging.handlers.RotatingFileHandler(LOG_FILENAME, maxBytes=1024 * 1024, backupCount=5)
        self.logger.addHandler(handler)
    
    
    def write(self, msg, level=logging.DEBUG):
        orig_stdout.write(msg)
        msg = msg.strip()
        s_time = datetime.datetime.now().strftime('[%Y-%b-%d %H:%M]')
        s = '%s %s' % (s_time, msg)
        self.logger.log(level, s)
    
    
    def flush(self):
        for handler in self.logger.handlers:
            handler.flush()


sys.stdout = LogFile('stdout')
sys.stderr = LogFile('stderr')


# Utility function for printing
def show_avance(pct):
    print '\r[',
    print '.' * pct, ' ' * (100 - pct),
    print '] %d%%' % pct,
    sys.stdout.flush()
    if pct == 100:
        print ''


def natural_key(string_):
    l = []
    for s in re.split(r'(\d+)', string_):
        if s.isdigit():
            l.append(int(s))
        else:
            l.append(s.lower())
    return l


def natural_version(f):
    return natural_key(f.version)


# Look if a function is ok to be launch on this server
def do_match_daemons(f):
    l_daemons = get_local_daemons()
    f_daemons = f.context_daemons
    if not f_daemons:
        return True
    for fd in f_daemons:
        if fd in l_daemons:
            return True
    return False


# TODO: look at real db path
def get_synchronizer_db():
    con = Connection()
    db = con.synchronizer
    return db


TABLES = {
    'businessimpactmodulation': {},
    'command'                 : {},
    'contact'                 : {},
    'contact-template'        : {'table': 'contact'},
    'contactgroup'            : {},
    'escalation'              : {},
    'host'                    : {},
    'host-template'           : {'table': 'host'},
    'hostgroup'               : {},
    'macromodulation'         : {},
    'notificationway'         : {},
    'check'                   : {'table': 'service', 'key': 'service_description'},
    'check-template'          : {'table': 'service'},
    'timeperiod'              : {},
}


def get_objects_to_save(type_match, name_match):
    db = get_synchronizer_db()
    colname = TABLES[type_match].get('table', type_match)
    col = getattr(db, 'configuration-stagging-' + colname)
    key = ''
    # If template, will match the name
    if type_match.endswith("-template"):
        key = 'name'
    else:  # ok classic object, maybe there is a exception to the key to get
        key = TABLES[type_match].get('key', '%s_name' % type_match)
    
    search = {key: re.compile('%s' % name_match)}
    objs = list(col.find(search))
    # clean all last_modification from backups
    for o in objs:
        try:
            del o['last_modification']
        except:
            pass
    return objs


class mock_datamanagerV2:
    def get_raw_item(self, i):
        return i


def restore_input(input_file):
    # 
    try:
        f = open(input_file, 'r')
        r = json.loads(f.read())
        f.close()
    except Exception, exp:
        print "Error: cannot load backup file %s : %s" % (input_file, exp)
        sys.exit(2)
    what = r.get('what', None)
    if what != 'focus-backup':
        print "Error: the backup file %s is not a focus backup" % input_file
        sys.exit(2)
    
    type_match = r.get("type", None)
    if type_match is None or type_match not in TABLES:
        print "Error: the backup file %s have an unknown data type %s" % (input_file, type_match)
        sys.exit(2)
    
    name_match = r.get('name', '')
    
    db = get_synchronizer_db()
    colname = TABLES[type_match].get('table', type_match)
    col = getattr(db, 'configuration-stagging-' + colname)
    objs = r.get('data', [])
    # set the history info on the elements
    # by simulating our user shinken-core with _id 16f36b30185f11e69025f8bc126497d6
    _user = {'_id': '16f36b30185f11e69025f8bc126497d6'}
    for o in objs:
        add_history_info(mock_datamanagerV2(), o, type_match, 'stagging', _user, 'ELEMENT_RESTORE', {})
        logger.debug('Element to import %s' % o)
        col.save(o)
    print "We did restore %s elements of type %s that match the backup filter '%s' into database" % (colorize('%d' % len(objs), 'FOCUS'), colorize(type_match, 'FOCUS'), colorize(name_match, 'FOCUS'))


if __name__ == '__main__':
    parser = optparse.OptionParser("%prog ", version="%prog: " + VERSION, description='This tool  is used to backup specific elements in your configuration database')
    parser.add_option('-l', '--list', dest='list_tables', action='store_true',
                      help="Only available types")
    parser.add_option('-n', '--name', dest='name_match',
                      help="[backup] Element name to match")
    parser.add_option('-t', '--type', dest='type_match',
                      help="[backup] Type of element to match")
    parser.add_option('-o', '--output', dest='output',
                      help="[backup] File to save backup")
    parser.add_option('-v', '--verbose', dest='verbose', action='store_true',
                      help="Show verbose output")
    parser.add_option('-a', '--action', dest='action',
                      help="Which action to do: backup/restore")
    parser.add_option('-i', '--input', dest='input',
                      help="[restore] Backup file to restore")
    
    opts, args = parser.parse_args()
    
    # Look if the user ask for local or global, and if not, guess
    list_tables = opts.list_tables
    name_match = opts.name_match
    output = opts.output
    type_match = opts.type_match
    input_ = opts.input
    action = opts.action
    
    if opts.verbose:
        logger.setLevel('DEBUG')
    
    local_daemons = get_local_daemons()
    if not is_daemon_node('synchronizer'):
        print "ERROR: must be run on the synchronizer server"
        sys.exit(2)
    
    if list_tables:
        print "Available types to backup:"
        _types = TABLES.keys()
        _types.sort()
        for k in _types:
            print "\t%s" % k
        sys.exit(0)
    
    if action not in ['backup', 'restore']:
        print "ERROR: the action parameter (-a/--action) is mandatory and must be backup or restore"
        sys.exit(2)
    
    is_backup = (action == 'backup')
    is_restore = (action == 'restore')
    
    # Do the backup:
    if is_backup:
        if not type_match:
            print "Error: missing type filtering (-t/--type) parameter"
            sys.exit(1)
        
        # First check for parameters fo backup
        if name_match is None:
            print "Error: missing name filtering (-n/--name) parameter"
            sys.exit(1)
        
        if type_match not in TABLES:
            print "Error: unknown type selected. Please use the (-l/--list) parameter to list them"
            sys.exit(1)
        
        if output is None:
            print "Error: missing output file parameter. Please use the (-o/--ouput) parameter to set it"
            sys.exit(1)
        
        if os.path.isdir(output):
            print "Error: output must be a filename, not a directory"
            sys.exit(1)
        
        # All is ok, we can get
        objs = get_objects_to_save(type_match, name_match)
        logger.debug("OBJETS BACKUP: %s" % objs)
        if len(objs) == 0:
            print "Error: there is no matching objects for your selection"
            sys.exit(2)
        
        # Ok there is something to really restore
        r = {'what': 'focus-backup',
             'type': type_match,
             'name': name_match,
             'data': objs,
             }
        
        try:
            f = open(output, 'w')
            buf = json.dumps(r, indent=4)
            f.write(buf)
            f.close()
        except Exception, exp:
            print "Error: cannot write backup file %s: %s" % (output, exp)
            sys.exit(2)
        
        print "Did backup %s elements of type %s that matches the name filter '%s' into the file %s : OK" % (colorize('%d' % len(objs), 'FOCUS'), colorize(type_match, 'FOCUS'), colorize(name_match, 'FOCUS'), colorize(output, 'FOCUS'))
        sys.exit(0)
    
    # Do restore
    if is_restore:
        if input_ is None:
            print "Error: missing input file parameter. Please use the (-i/--input) parameter to set it"
            sys.exit(1)
        
        if not os.path.exists(input_):
            print "Error: input (-i/--input) is missing."
            sys.exit(1)
        
        restore_input(input_)
