import threading
import traceback
import time
import cPickle
from Queue import Empty

from .moduleworker import ModuleWorker
from shinken.load import AvgForFixSizeCall

ONE_MINUTE = 60
SAMPLING_TIME = 10

# Send broks by stack of X or at least once a second
MAX_BROKS_TO_SEND_QUEUE_SIZE = 100


# The Broker module Worker is a classic worker but with new capabilities so:
# * it does inherit from the inventory features
# * it add methods to send broks to the sub-process
# * it add a thread that get the broks from the main process and call the user manage
class BrokerModuleWorker(ModuleWorker):
    def __init__(self, worker_id, mod_conf, name, from_module_to_main_daemon_queue, queue_factory, daemon_display_name):
        super(BrokerModuleWorker, self).__init__(worker_id, mod_conf, name, from_module_to_main_daemon_queue, queue_factory, daemon_display_name)
        
        # We should create a new queue __init__ is in the main process, that the sub-process worker will inherit
        self._main_process_to_worker_broks_queue = self._queue_factory('_main_process_to_worker_broks_queue')
        
        self._last_brok_push = 0.0
        self._to_send_broks_stack = []
        
        self._in_worker_to_manage_queue_lock = None
        self._in_worker_to_manage_queue = []
        
        # STATS:
        # If compute_stats is at False only cumulative time will be set
        self.compute_stats = True
        
        self._manage_work_time = AvgForFixSizeCall(time_limit=ONE_MINUTE)
        self._manage_work_time_sampling = self._get_time_sampling_struct()  # [-1 for _ in xrange(int(ONE_MINUTE / SAMPLING_TIME))]
        self._manage_work_time_last_print = time.time()
        self._manage_work_time_cumulative = (0.0, 0.0, 0)
        
        self._get_broks_work_time = AvgForFixSizeCall(time_limit=ONE_MINUTE)
        self._get_broks_work_time_sampling = self._get_time_sampling_struct()  # [-1 for _ in xrange(int(ONE_MINUTE / SAMPLING_TIME))]
        self._get_broks_work_time_last_print = time.time()
        self._get_broks_work_time_cumulative = (0.0, 0.0, 0)
    
    
    @staticmethod
    def _get_time_sampling_struct():
        return [-1 for _ in xrange(int(ONE_MINUTE / SAMPLING_TIME))]
    
    
    def _force_push_to_worker(self):
        if len(self._to_send_broks_stack) == 0:
            return
        queue = self._main_process_to_worker_broks_queue
        try:
            queue.put(self._to_send_broks_stack)
            self._to_send_broks_stack = []
            self._last_brok_push = time.time()
        except Exception as exp:
            raise Exception('[%s] Cannot send the brok to the worker %s: %s' % (self.get_name(), self._worker_id, exp))
    
    
    def _send_brok_to_worker(self, brok):
        # Stack
        # NOTE: we do pre-serialize the broks because the multiprocessing lib seems to be using pickle and not cPickle.
        serialized_brok = cPickle.dumps(brok, cPickle.HIGHEST_PROTOCOL)
        self._to_send_broks_stack.append(serialized_brok)
        
        # and check if we should send it now or not
        now = time.time()
        if now > self._last_brok_push + 1 or len(self._to_send_broks_stack) > MAX_BROKS_TO_SEND_QUEUE_SIZE:
            self._force_push_to_worker()
    
    
    # In the raw stats, we will have to sum the main two threads, the get_broks & manage broks one
    def get_raw_stats(self):
        data = super(BrokerModuleWorker, self).get_raw_stats()
        
        total_cumulative = (
            max(self._manage_work_time_cumulative[0], self._get_broks_work_time_cumulative[0]),
            self._manage_work_time_cumulative[1] + self._get_broks_work_time_cumulative[1],
            self._manage_work_time_cumulative[2]
        )
        data['work_time_cumulative'] = {
            'manage_work_time'   : self._manage_work_time_cumulative,
            'get_broks_work_time': self._get_broks_work_time_cumulative,
            'total'              : total_cumulative,
        }
        if not self.compute_stats:
            return data
        # Times
        manage_work_times = self._manage_work_time.get_sum_in_range(avg_on_time=True, with_range_size=True)
        get_broks_work_times = self._get_broks_work_time.get_sum_in_range(avg_on_time=True, with_range_size=True)
        # Beware: sum value, but do not sum the range size
        work_times = (manage_work_times[0] + get_broks_work_times[0], manage_work_times[1])
        data['work_time'] = work_times
        
        # Sampling
        work_time_sampling = self._get_time_sampling_struct()
        for idx, v in enumerate(self._manage_work_time_sampling):
            if v != -1:
                work_time_sampling[idx] = v
        for idx, v in enumerate(self._get_broks_work_time_sampling):
            if v != -1:
                work_time_sampling[idx] += v
        data['work_time_sampling'] = work_time_sampling
        return data
    
    
    # Every second, we do force the broks in queue to be send, to not have out of time broks
    def in_main_process_tick(self):
        self._force_push_to_worker()
    
    
    def _do_in_worker_get_broks_thread(self):
        while True:
            try:
                self._do_in_worker_get_broks_thread_loop()
            except Exception as exp:
                self._send_exception_error_to_main_process_and_exit(traceback.format_exc(), exp, 'broks reading')
    
    
    def _do_in_worker_get_broks_thread_loop(self):
        # We get from the Queue update from the receiver.
        # NOTE: as we are blocking, we are not using CPU and so no need for sleep
        try:
            serialized_broks = self._main_process_to_worker_broks_queue.get(block=True, timeout=1)
        except Empty:
            return
        
        start_time = time.time()
        for serialized_brok in serialized_broks:
            brok = cPickle.loads(serialized_brok)
            with self._in_worker_to_manage_queue_lock:
                self._in_worker_to_manage_queue.append(brok)
        
        # Compute stats
        end_time = time.time()
        time_taken = end_time - start_time
        self._get_broks_work_time_cumulative = (end_time, self._get_broks_work_time_cumulative[1] + time_taken, self._get_broks_work_time_cumulative[2] + len(serialized_broks))
        if self.compute_stats:
            self._get_broks_work_time.update_avg(time_taken)
            last_print_period = end_time - self._get_broks_work_time_last_print
            if last_print_period > SAMPLING_TIME:
                self._get_broks_work_time_sampling.pop(0)
                self._get_broks_work_time_sampling.append(self._get_broks_work_time.get_sum_in_range(avg_on_time=False, with_range_size=False, time_limit_overload=last_print_period))
                self._get_broks_work_time_last_print = end_time
    
    
    def _do_in_worker_manage_broks_thread(self):
        # Important: we cannot manage broks until the worker is fully init
        self._wait_for_worker_init_to_be_done()
        
        while True:
            try:
                self._do_in_worker_manage_broks_thread_loop()
            except Exception as exp:
                self._send_exception_error_to_main_process_and_exit(traceback.format_exc(), exp, 'broks managing')
    
    
    def _get_brok_from_manage_queue(self):
        brok = None
        start = time.time()
        while brok is None:
            with self._in_worker_to_manage_queue_lock:
                if len(self._in_worker_to_manage_queue) != 0:
                    brok = self._in_worker_to_manage_queue.pop(0)
                    return brok
            # Do not exceed 1s loop
            if time.time() > start + 1:
                return None
            time.sleep(0.01)
    
    
    def _do_in_worker_manage_broks_thread_loop(self):
        brok = self._get_brok_from_manage_queue()
        if brok is None:
            return
        
        start_time = time.time()
        manage = getattr(self, 'manage_' + brok.type + '_brok', None)
        if manage:
            # Be sure the brok is prepared before call it
            brok.prepare()
            manage(brok)
        
        # Compute stats
        end_time = time.time()
        time_taken = end_time - start_time
        self._manage_work_time_cumulative = (end_time, self._manage_work_time_cumulative[1] + time_taken, self._manage_work_time_cumulative[2] + 1)
        if self.compute_stats:
            self._manage_work_time.update_avg(time_taken)
            last_print_period = end_time - self._manage_work_time_last_print
            if last_print_period > SAMPLING_TIME:
                self._manage_work_time_sampling.pop(0)
                self._manage_work_time_sampling.append(self._manage_work_time.get_sum_in_range(avg_on_time=False, with_range_size=False, time_limit_overload=last_print_period))
                self._manage_work_time_last_print = end_time
    
    
    def start_worker_specific_treads(self):
        # We are now in the worker process, we can create the locks
        self._in_worker_to_manage_queue_lock = threading.RLock()
        
        # Also start a thread that will manage broks
        thr = threading.Thread(target=self._do_in_worker_get_broks_thread, name='do-in-worker-get-broks')
        thr.daemon = True
        thr.start()
        
        # Also start a thread that will manage broks
        thr = threading.Thread(target=self._do_in_worker_manage_broks_thread, name='do-in-worker-manage-broks')
        thr.daemon = True
        thr.start()
    
    
    def _get_main_process_to_worker_broks_queue_size(self):
        try:
            qsize = self._main_process_to_worker_broks_queue.qsize()
        except Exception as exp:
            self.logger.warning('Cannot get the size of the main process to workers queue: %s' % (exp))
            qsize = -1
        return qsize
