import uuid
import time

from .log import logger

# Purely random uuid4 for regenerator that do not have currently a configuration to manage
DEFAULT_CONFIGURATION_UUID = '6dbc00f706ca4a3b9a60a843a370a634'


class ConfigurationIncarnation(object):
    def __init__(self):
        self._uuid = uuid.uuid4().hex
        self._creation_date = int(time.time())
        self._author = '(unknown)'
        
        # logger.info('The configuration is loaded with a unique uuid: %s' % self._uuid)
    
    
    def set_author(self, author):
        self._author = author
    
    
    def get_uuid(self):
        return self._uuid
    
    
    def get_creation_date(self):
        return self._creation_date
    
    
    def reset_creation_date(self):
        self._creation_date = int(time.time())
    
    
    def __str__(self):
        return '[configuration_uuid=%s, arbiter=%s, date=%s]' % (self._uuid, self._author, time.strftime('%d-%m-%Y %H:%M:%S', time.localtime(self._creation_date)))
    
    
    def __repr__(self):
        return self.__str__()


class PartConfigurationIncarnation(object):
    def __init__(self, configuration_incarnation, part_id, scheduler_name):
        # type: (ConfigurationIncarnation, int, str) -> None
        self._uuid = configuration_incarnation._uuid
        self._creation_date = configuration_incarnation._creation_date
        self._author = configuration_incarnation._author
        self._part_id = part_id
        self.scheduler_name = scheduler_name
    
    
    def __str__(self):
        return '[configuration_part_id=%s, scheduler=%s, configuration_uuid=%s, arbiter=%s, date=%s]' % (self._part_id, self.scheduler_name, self._uuid, self._author, time.strftime('%d-%m-%Y %H:%M:%S', time.localtime(self._creation_date)))
    
    
    def __repr__(self):
        return self.__str__()
    
    
    def get_uuid(self):
        return self._uuid
    
    
    def get_creation_date(self):
        return self._creation_date
    
    
    def get_author(self):
        return self._author
    
    
    def get_part_id(self):
        return self._part_id
    
    
    def is_newer_or_equal_than(self, _date):
        # type: (int) -> bool
        return self._creation_date >= _date
    
    
    def is_newer_than(self, _date):
        # type: (int) -> bool
        return self._creation_date > _date


class PartConfigurationIncarnationContainer(object):
    
    def __init__(self):
        self._data = {}
        self._configuration_handle_uuid = DEFAULT_CONFIGURATION_UUID
        self._configuration_handle_date = 0
        self.last_part_configuration_incarnation = None
    
    
    def add(self, part_configuration_incarnation):
        # type: (PartConfigurationIncarnation) -> None
        self._data[part_configuration_incarnation.get_part_id()] = part_configuration_incarnation
        self.last_part_configuration_incarnation = part_configuration_incarnation
        self._configuration_handle_uuid = part_configuration_incarnation.get_uuid()
        self._configuration_handle_date = part_configuration_incarnation.get_creation_date()
    
    
    def check(self, part_configuration_incarnation):
        # type: (PartConfigurationIncarnation) -> (bool, str)
        current_part_configuration_incarnation = self._data.get(part_configuration_incarnation.get_part_id(), None)
        
        i_have_this_part = current_part_configuration_incarnation is not None
        not_already_handle = not i_have_this_part or (i_have_this_part and part_configuration_incarnation.get_uuid() != current_part_configuration_incarnation.get_uuid())
        if_already_have_this_part__new_conf_must_be_newer_than_the_one_i_have = i_have_this_part and part_configuration_incarnation.is_newer_than(current_part_configuration_incarnation.get_creation_date())
        if_do_not_have_this_part__new_conf_must_be_newer_or_same_date = not i_have_this_part and part_configuration_incarnation.is_newer_or_equal_than(self._configuration_handle_date)
        
        if not not_already_handle:
            cause = 'I already handle it %s' % part_configuration_incarnation
        elif i_have_this_part:
            cause = 'The configuration send is older than the one we have. Current handle configuration %s. Send configuration : %s' % (current_part_configuration_incarnation, part_configuration_incarnation)
        else:
            cause = 'The configuration send is too old. Current handle configuration %s. Send configuration : %s' % (time.strftime('%d-%m-%Y %H:%M:%S', time.localtime(self._configuration_handle_date)), part_configuration_incarnation)
        
        return not_already_handle and (if_already_have_this_part__new_conf_must_be_newer_than_the_one_i_have or if_do_not_have_this_part__new_conf_must_be_newer_or_same_date), cause
    
    
    def get_configuration_handle_uuid(self):
        return self._configuration_handle_uuid
    
    
    def get_part_configuration_incarnation(self, part_id):
        return self._data.get(part_id, None)
        
    def get_last_part_configuration_incarnation(self):
        return self.last_part_configuration_incarnation
