#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import time
import copy
import threading

from item import Item
from itemgroup import Itemgroup, Itemgroups
from shinken.property import BoolProp, IntegerProp, StringProp, RawProp
from shinken.log import logger
from shinken.util import alive_then_spare_then_deads
from shard import Shard
from .inventory import ElementsInventory

MAX_REALMS_LEVEL = 10


# It change from hostgroup Class because there is no members
# properties, just the realm_members that we rewrite on it.

class Realm(Itemgroup):
    id = 1  # zero is always a little bit special... like in database
    my_type = 'realm'
    
    satellite_kinds = ('reactionner', 'poller', 'broker', 'receiver')
    
    properties = Itemgroup.properties.copy()
    properties.update({
        'id'                   : IntegerProp(default=0, fill_brok=['full_status']),
        'realm_name'           : StringProp(fill_brok=['full_status']),
        'realm_members'        : StringProp(default=''),  # No status_broker_name because it put hosts, not host_name
        'higher_realms'        : StringProp(default=''),
        'default'              : BoolProp(default='0'),
        'broker_complete_links': BoolProp(default='0'),
        'only_schedulers'      : BoolProp(default='0'),
    })
    
    running_properties = Item.running_properties.copy()
    running_properties.update({
        'shards'                : RawProp(default={}),
        'sharded_configurations': RawProp(default={}),
        'higher_realms'         : StringProp(default=[]),
        
    })
    
    macros = {
        'REALMNAME'   : 'realm_name',
        'REALMMEMBERS': 'members',
    }
    
    static_macros = set()
    
    
    def __init__(self, params={}, skip_useless_in_configuration=False):
        super(Realm, self).__init__(params=params, skip_useless_in_configuration=skip_useless_in_configuration)
        # We will keep a list of all the hosts and checks we have with basic information
        # mainly for receivers & brokers
        self._host_and_check_receiver_inventory = ElementsInventory()
        self._host_and_check_broker_inventory = ElementsInventory()
        self._got_loop = False
    
    
    # Call by pickle for dataify the realm, but we do not want the lock (not picklelisable)
    def __getstate__(self):
        res = self.__dict__.copy()
        del res['structure_lock']
        return res
    
    
    # Inverted function of getstate
    def __setstate__(self, state):
        self.__dict__.update(state)
        # Recreate the lock
        self.structure_lock = threading.RLock()
    
    
    def get_name(self):
        return self.realm_name
    
    
    def get_realms(self):
        return self.realm_members
    
    
    def add_string_member(self, member):
        if member:
            self.realm_members += ',' + member
    
    
    def remove_string_member(self, member):
        members_list = self.get_realm_members()
        if member in members_list:
            members_list.remove(member)
            self.realm_members = ",".join(members_list)
    
    
    def get_realm_members(self):
        if self.has('realm_members'):
            return [r.strip() for r in self.realm_members.split(',') if r.strip()]
        else:
            return []
    
    
    # This realm can have special issues:
    # * multiples active schedulers (potential retentions issues):
    #   * if no retention
    #   * if retention module that is not mongodb (distributed)  TODO: how to detect this?
    #   * if retention mongodb BUT not the same address
    #   * if retention mongodb BUT still localhost address (need public IP)
    # * a scheduler without a broker is an error
    def is_correct(self):
        r = True
        
        active_schedulers = [s for s in self.schedulers if not s.spare]
        nb_active_sched = len(active_schedulers)
        
        # Allowed retention modules are in the list:
        # * pickle flat file : pickle_retention_file (or pickle-retention-file)
        # * mongodb : mongodb_retention (or mongodb-retention)
        is_distributed = (nb_active_sched >= 2)
        
        total_pickle = 0
        total_mongo = 0
        all_mongodb_modules = []
        for sched in self.schedulers:
            modules = sched.modules
            # if no modules at all, get out, because no retention is set
            if len(modules) == 0:
                self.configuration_errors.append("The scheduler %s does not have any retention module and so it won't be able to save and load its retention data. Please add a scheduler retention module." % sched.get_name())
                continue
            for mod in modules:
                if mod.module_type == 'mongodb_retention':
                    all_mongodb_modules.append(mod)
            mod_types = [mod.module_type.replace('-', '_') for mod in modules]  # beware: framework 2.4 module are with - instead of _
            nb_pickle = mod_types.count('pickle_retention_file')
            total_pickle += 1
            nb_mongo = mod_types.count('mongodb_retention')
            total_mongo += 1
            if is_distributed and nb_pickle >= 1:
                self.configuration_errors.append(
                    'The scheduler %s has a pickle_retention_file module but is in a distributed realm with several schedulers. This module is incompatible for those environments as the retention file is not shared across servers. Please use a mongodb_retention type module instead.' % (
                        sched.get_name()))
                continue
            if is_distributed and nb_mongo < 1:
                self.configuration_errors.append('The scheduler %s is in a distributed realm (%d active schedulers) but do not have any mongodb retention module. It will break the retention data.' % (sched.get_name(), nb_active_sched))
                continue
        # Now all individual cases for schedulers are done, look at if the mongodb retention modules conf is ok
        # in a distributed env, should not be with localhost
        if is_distributed:
            for module in all_mongodb_modules:
                uri = getattr(module, 'uri', '')
                if uri == '':
                    self.configuration_errors.append('The %s module is a mongodb retention module but without a URI parameter. Please add one.' % module.get_name())
                    continue
                if 'localhost' in uri:
                    self.configuration_errors.append(
                        'The MongodbRetention module %s is configured with localhost URI. In a distributed realm with several schedulers, all retention modules in that realm must be set to the same server. Please specify the IP address of the mongodb retention server.' % (
                            module.get_name()))
                    continue
        r |= Itemgroup.is_correct(self)
        if self.configuration_errors:
            r = False
        return r
    
    
    # We want to see if non void schedulers (with hosts) are reacheable
    def is_correct_schedulers_with_hosts_and_satellites(self):
        if self.only_schedulers:
            return True
        
        nb_hosts_in_realm = 0
        for shard in self.shards.values():
            nb_hosts_in_realm += len(shard.hosts)
        
        _errors = []
        if nb_hosts_in_realm != 0:
            _no_daemon_err = "The scheduler %s (in the realm %s) have elements to monitor but has no %s in its realm (or upper realm with the manage_sub_realms option enabled)"
        else:
            _no_daemon_err = "The scheduler %s (in the realm %s) has no %s in its realm (or upper realm with the manage_sub_realms option enabled). This is just a warning because it does not have elements to monitor but it will be an error if you add hosts or clusters in this realm"
        # Look if we have at least a broker for schedulers
        for sched in self.schedulers:
            sched_name = sched.get_name()
            realm_name = self.get_name()
            if len(self.potential_brokers) == 0:
                _errors.append(_no_daemon_err % (sched_name, realm_name, 'brokers'))
            
            if len(self.potential_pollers) == 0:
                _errors.append(_no_daemon_err % (sched_name, realm_name, 'pollers'))
            
            if len(self.potential_reactionners) == 0:
                _errors.append(_no_daemon_err % (sched_name, realm_name, 'reactionners'))
        
        # All is great
        if len(_errors) == 0:
            return True
        
        # Oups, can be a realm error if the scheduler have hosts (becuase the realm have some), or
        # just a warning if the realm is without hosts (like the top level one when the admin
        # forget to remove the default scheduler)
        if nb_hosts_in_realm == 0:  # no hosts, WARNING
            for _err in _errors:
                logger.warning(_err)
            return True  # is valid
        else:  # have hosts, will be ERROR
            for _err in _errors:
                logger.error(_err)
            return False
    
    
    # Use to make python properties
    # TODO: change itemgroup function pythonize?
    def pythonize(self):
        cls = self.__class__
        for prop, tab in cls.properties.iteritems():
            try:
                old_val = getattr(self, prop)
                new_val = tab.pythonize(old_val)
                # print "Changing ", old_val, "by", new_val
                setattr(self, prop, new_val)
            except AttributeError as exp:
                pass  # Will be catch at the is_correct moment
    
    
    def add_shard(self, shard_id, serialized_shard, host_names):
        self.shards[shard_id] = Shard(serialized_shard, shard_id, host_names)
    
    
    # We are receiving a partial (one shard only) host & checks inventory (name, uuid, etc) so we are merging it
    # as they are host_uuid indexed, and one host can only be in one shard
    def update_inventory_from_sharded_inventories(self, sharded_inventories):
        for (receiver_sharded_inventory, broker_sharded_inventory) in sharded_inventories:
            self._host_and_check_receiver_inventory.update_from_sharded_inventory(receiver_sharded_inventory)
            self._host_and_check_broker_inventory.update_from_sharded_inventory(broker_sharded_inventory)
        self._host_and_check_receiver_inventory.compute_hash()
        self._host_and_check_broker_inventory.compute_hash()
        
        logger.info('[INVENTORY] [%s] The realm hosts & checks inventory for receivers is now updated from shard inventories. We now have an inventory of %d hosts. The new realm inventory hash is %s.' % (
            self.get_name(), self._host_and_check_receiver_inventory.get_len(), self._host_and_check_receiver_inventory.get_hash()))
        logger.info('[INVENTORY] [%s] The realm hosts & checks inventory for brokers is now updated from shard inventories. We now have an inventory of %d hosts. The new realm inventory hash is %s.' % (
            self.get_name(), self._host_and_check_broker_inventory.get_len(), self._host_and_check_broker_inventory.get_hash()))
    
    
    # We should give an inventory to the receiver, in a sharded way
    # * if we have only one shard, we can directly give our inventory
    # * if not, we should compute another inventory with only the needs hosts
    def _get_inventory_for_receiver(self, previous_elements_weight_sum, receiver_weight, sum_of_receiver_weights, receiver_index, receiver_weights):
        # One shard means that our weight is the sum
        if receiver_weight == sum_of_receiver_weights:
            return self._host_and_check_receiver_inventory
        receiver_inventory = self._host_and_check_receiver_inventory.get_new_inventory_from_us_and_take_only_a_shard_part(previous_elements_weight_sum, receiver_weight, sum_of_receiver_weights, receiver_index, receiver_weights)
        return receiver_inventory
    
    
    # We should give an inventory to the broker.
    # Not the same as receiver, as the broker need EVERY hosts/checks
    # so we can give the whole inventory
    def _get_inventory_for_broker(self):
        return self._host_and_check_broker_inventory
    
    
    def get_realms_by_explosion(self, realms, level=0):
        
        if level > MAX_REALMS_LEVEL:
            err = "Error: %s got too many level for realms" % self.get_name()
            self.configuration_errors.append(err)
            return
        
        if getattr(self, '_already_explode', False):
            return
        
        self._already_explode = True
        
        all_my_sons_members = set()
        realm_mbrs = self.get_realm_members()
        
        for sub_realm_name in realm_mbrs:
            sub_realm = realms.find_by_name(sub_realm_name.strip())
            if sub_realm is None:
                err = "Error: the realm '%s' have the sub realm '%s' but it does not exists" % (self.get_name(), sub_realm_name)
                self.configuration_errors.append(err)
                continue
            # my son is one of my members
            all_my_sons_members.add(sub_realm_name)
            # Ask the sub realm to fill itself
            sub_realm.get_realms_by_explosion(realms, level + 1)
            for sub_realm_son in sub_realm.get_realm_members():
                if sub_realm_son == self.get_name():  # do not set ourselve as sons if ther is a loop
                    err = "Error: The realm %s got a loop with the other realm %s" % (self.get_name(), sub_realm_name)
                    self.configuration_errors.append(err)
                    self._got_loop = True
                    continue
                all_my_sons_members.add(sub_realm_son)
        self.realm_members = ','.join(all_my_sons_members)
    
    
    def get_all_subs_pollers(self):
        r = copy.copy(self.pollers)
        for p in self.realm_members:
            tmps = p.get_all_subs_pollers()
            for s in tmps:
                r.append(s)
        return r
    
    
    def get_all_subs_reactionners(self):
        r = copy.copy(self.reactionners)
        for p in self.realm_members:
            tmps = p.get_all_subs_reactionners()
            for s in tmps:
                r.append(s)
        return r
    
    
    def get_all_subs_receivers(self):
        r = copy.copy(self.receivers)
        for p in self.realm_members:
            tmps = p.get_all_subs_receivers()
            for s in tmps:
                r.append(s)
        return r
    
    
    def count_reactionners(self):
        self.nb_reactionners = 0
        for reactionner in self.reactionners:
            if not reactionner.spare:
                self.nb_reactionners += 1
        for realm in self.higher_realms:
            for reactionner in realm.reactionners:
                if not reactionner.spare and reactionner.manage_sub_realms:
                    self.nb_reactionners += 1
    
    
    def fill_potential_reactionners(self):
        self.potential_reactionners = []
        for reactionner in self.reactionners:
            self.potential_reactionners.append(reactionner)
        for realm in self.higher_realms:
            for reactionner in realm.reactionners:
                if reactionner.manage_sub_realms:
                    self.potential_reactionners.append(reactionner)
    
    
    def count_pollers(self):
        self.nb_pollers = 0
        for poller in self.pollers:
            if not poller.spare:
                self.nb_pollers += 1
        for realm in self.higher_realms:
            for poller in realm.pollers:
                if not poller.spare and poller.manage_sub_realms:
                    self.nb_pollers += 1
    
    
    def fill_potential_pollers(self):
        self.potential_pollers = []
        for poller in self.pollers:
            self.potential_pollers.append(poller)
        for realm in self.higher_realms:
            for poller in realm.pollers:
                if poller.manage_sub_realms:
                    self.potential_pollers.append(poller)
    
    
    def count_brokers(self):
        self.nb_brokers = 0
        for broker in self.brokers:
            if not broker.spare:
                self.nb_brokers += 1
        for realm in self.higher_realms:
            for broker in realm.brokers:
                if not broker.spare and broker.manage_sub_realms:
                    self.nb_brokers += 1
    
    
    def fill_potential_brokers(self):
        self.potential_brokers = []
        for broker in self.brokers:
            self.potential_brokers.append(broker)
        for realm in self.higher_realms:
            for broker in realm.brokers:
                if broker.manage_sub_realms:
                    self.potential_brokers.append(broker)
    
    
    def count_providers(self):
        self.nb_providers = 0
        for provider in self.providers:
            if not provider.spare:
                self.nb_providers += 1
        for realm in self.higher_realms:
            for provider in realm.providers:
                if not provider.spare and provider.manage_sub_realms:
                    self.nb_providers += 1
    
    
    def fill_potential_providers(self):
        self.potential_providers = []
        for provider in self.providers:
            self.potential_providers.append(provider)
        for realm in self.higher_realms:
            for provider in realm.providers:
                if provider.manage_sub_realms:
                    self.potential_providers.append(provider)
    
    
    def count_receivers(self):
        self.nb_receivers = 0
        for receiver in self.receivers:
            if not receiver.spare:
                self.nb_receivers += 1
        for realm in self.higher_realms:
            for receiver in realm.receivers:
                if not receiver.spare and receiver.manage_sub_realms:
                    self.nb_receivers += 1
    
    
    def fill_potential_receivers(self):
        self.potential_receivers = []
        for receiver in self.receivers:
            self.potential_receivers.append(receiver)
        for realm in self.higher_realms:
            for receiver in realm.receivers:
                if receiver.manage_sub_realms:
                    self.potential_receivers.append(receiver)
    
    
    # Return the list of satellites of a certain type
    # like reactionner -> self.reactionners
    def get_satellites_by_type(self, type):
        if hasattr(self, type + 's'):
            return getattr(self, type + 's')
        else:
            logger.debug("[realm] do not have this kind of satellites: %s" % type)
            return []
    
    
    # Return the list of potentials satellites of a certain type
    # like reactionner -> self.potential_reactionners
    def get_potential_satellites_by_type(self, type):
        if hasattr(self, 'potential_' + type + 's'):
            return getattr(self, 'potential_' + type + 's')
        else:
            logger.debug("[realm] do not have this kind of satellites: %s" % type)
            return []
    
    
    # Return the list of potentials satellites of a certain type
    # like reactionner -> self.nb_reactionners
    def get_nb_of_must_have_satellites(self, type):
        if hasattr(self, 'nb_' + type + 's'):
            return getattr(self, 'nb_' + type + 's')
        else:
            logger.debug("[realm] do not have this kind of satellites: %s" % type)
            return 0
    
    
    def declare_shard_sent_to_a_scheduler(self, shard, scheduler):
        with self.structure_lock:
            scheduler_configuration_for_satellites = scheduler.give_satellite_cfg()
            for kind in self.satellite_kinds:
                self.to_satellites[kind][shard.id] = scheduler_configuration_for_satellites
                self.to_satellites_need_dispatch[kind][shard.id] = True
                self.to_satellites_managed_by[kind][shard.id] = []
    
    
    def reset_satellite_association_for_shard_id(self, kind, shard_id):
        with self.structure_lock:
            self.to_satellites[kind][shard_id] = None
            self.to_satellites_need_dispatch[kind][shard_id] = False
            self.to_satellites_managed_by[kind][shard_id] = []
    
    
    def reset_all_satellite_association_for_shard_id(self, shard_id):
        with self.structure_lock:
            for kind in self.satellite_kinds:
                self.reset_satellite_association_for_shard_id(kind, shard_id)
    
    
    def reset_all_satellites_associations(self):
        with self.structure_lock:
            for kind in self.satellite_kinds:
                for shard_id in self.shards.iterkeys():
                    self.reset_satellite_association_for_shard_id(kind, shard_id)
    
    
    def _get_all_satellites(self):
        return self.pollers + self.reactionners + self.brokers + self.receivers
    
    
    # Fill dict of realms for managing the satellites confs
    def prepare_for_satellites_conf(self):
        self.structure_lock = threading.RLock()
        self.to_satellites = {}
        self.to_satellites_need_dispatch = {}
        self.to_satellites_managed_by = {}
        for kind in self.satellite_kinds:
            self.to_satellites[kind] = {}
            self.to_satellites_need_dispatch[kind] = {}
            self.to_satellites_managed_by[kind] = {}
        
        self.count_reactionners()
        self.fill_potential_reactionners()
        self.count_pollers()
        self.fill_potential_pollers()
        self.count_brokers()
        self.fill_potential_brokers()
        self.count_providers()
        self.fill_potential_providers()
        self.count_receivers()
        self.fill_potential_receivers()
        
        s = "Realm %-15s: (in realm / in realm + higher realms) (schedulers:%d) (pollers:%d/%d) (reactionners:%d/%d) (brokers:%d/%d) (receivers:%d/%d)" % \
            (self.get_name(),
             len(self.schedulers),
             self.nb_pollers, len(self.potential_pollers),
             self.nb_reactionners, len(self.potential_reactionners),
             self.nb_brokers, len(self.potential_brokers),
             self.nb_receivers, len(self.potential_receivers)
             )
        logger.info(s)
    
    
    # Get a conf package of satellites links that can be useful for
    # a scheduler
    def get_satellites_links_for_scheduler(self):
        with self.structure_lock:
            cfg = {}
            
            # First we create/void theses links
            cfg['pollers'] = {}
            cfg['reactionners'] = {}
            cfg['schedulers'] = {}
            
            # First our own level
            for p in self.potential_pollers:
                c = p.give_satellite_cfg()
                cfg['pollers'][p.id] = c
            
            for r in self.potential_reactionners:
                c = r.give_satellite_cfg()
                cfg['reactionners'][r.id] = c
            
            for r in self.schedulers:
                s = r.give_satellite_cfg()
                cfg['schedulers'][r.id] = s
            
            # print "***** Preparing a satellites conf for a scheduler", cfg
            return cfg
    
    
    # Make an ORDERED list of schedulers so we can
    # send them conf in this order for a specific realm
    def get_scheduler_ordered_list(self):
        # get scheds, alive and no spare first
        scheds = set()
        for s in self.schedulers:
            scheds.add(s)
        
        # now the spare scheds of higher realms
        # they are after the sched of realm, so
        # they will be used after the spare of
        # the realm
        for higher_r in self.higher_realms:
            for s in higher_r.schedulers:
                if s.spare:
                    scheds.add(s)
        scheds = list(scheds)
        # Now we sort the scheds so we take master, then spare
        # the dead, but we do not care about them
        scheds.sort(alive_then_spare_then_deads)
        scheds.reverse()  # pop is last, I need first
        return scheds
    
    
    def get_not_assigned_shards(self):
        with self.structure_lock:
            return [shard for shard in self.shards.itervalues() if not shard.is_assigned]
    
    
    def _unset_scheduler_conf(self, sched):
        with self.structure_lock:
            self.dispatch_ok = False  # so we ask a new dispatching
            if sched.conf:
                sched.conf.assigned_to = None
                sched.conf.is_assigned = False
                sched.conf.push_flavor = 0
            sched.push_flavor = 0
            sched.conf = None
    
    
    def check_schedulers_dispatch(self, first_dispatch_done, arbiter_trace, configuration_incarnation):
        dispatch_ok = True
        potential_scheds = set(self.get_scheduler_ordered_list())
        managed_scheds = set()
        realm_name = self.get_name()
        for shard_id in self.shards:
            push_flavor = self.shards[shard_id].push_flavor
            sched = self.shards[shard_id].assigned_to
            if sched is None:
                # On the very first run, we do not raise an error, it's normal to not be assigned
                if first_dispatch_done:
                    logger.error("[DISPATCH][%s] UNMANAGED SHARD   shard [%d] is unmanaged. We will sent it to a new scheduler." % (realm_name, shard_id))
                dispatch_ok = False
            else:
                managed_scheds.add(sched)
                if not sched.alive:
                    logger.error("[DISPATCH][%s] DEAD SCHEDULER   Scheduler %s had the shard [%d] but is dead. We will sent its shard to another scheduler." % (realm_name, sched.get_name(), shard_id))
                    self._unset_scheduler_conf(sched)
                # Maybe the scheduler restarts, so is alive but without the conf we think it was managing
                # so ask it what it is really managing, and if not, put the conf unassigned
                elif not sched.do_i_manage(shard_id, push_flavor):
                    logger.error(
                        "[DISPATCH][%s] MISMATCHED SCHEDULER    Scheduler %s did not managed the shard [%d-%s] as expected. Managed shard by the scheduler [%s]" % (realm_name, sched.get_name(), shard_id, push_flavor, sched.managed_confs))
                    self._unset_scheduler_conf(sched)
                    sched.need_conf = True
                else:  # ok seem to be good but maybe there is a old/bad satellites
                    allowed_brokers_for_this_scheduler = [broker.get_name() for broker in self.to_satellites_managed_by['broker'][shard_id]]
                    logger.debug('[DISPATCH][%s] Checking if scheduler %s brokers are in this allowed list: "%s"' % (realm_name, sched.get_name(), ','.join(allowed_brokers_for_this_scheduler)))
                    sched.assert_only_allowed_brokers(allowed_brokers_for_this_scheduler)
        
        unmanaged_scheds = potential_scheds - managed_scheds
        unmanaged_scheds = [sched for sched in unmanaged_scheds if sched.alive and sched.reachable]
        unmanaged_non_spare_scheds = [sched for sched in unmanaged_scheds if not sched.spare]
        spare_scheds_with_conf = [sched for sched in managed_scheds if sched.spare]
        # if spare schedulers manage conf and non spare are waiting, we need to ask a redispatch for this conf
        if any(unmanaged_non_spare_scheds) and any(spare_scheds_with_conf):
            for index, sched in enumerate(spare_scheds_with_conf):
                logger.error("[DISPATCH][%s] IDLE SCHEDULER WITH SHARD   The scheduler %s have a shard but should not have one. We are asking it to go in sleep mode." % (realm_name, sched.get_name()))
                self._unset_scheduler_conf(sched)
                sched.need_conf = True
                # LONG CALL
                sched.create_and_put_inactive_scheduler_configuration(arbiter_trace, configuration_incarnation)
                # ask the redispatch only for the number of available non spare scheds. avoid redispatch every scheds if only one master come up
                if index >= len(unmanaged_non_spare_scheds):
                    break
        
        # Some schedulers are waiting for a new conf but they all have been sent, send them a spare conf
        for sched in unmanaged_scheds:
            if not sched.already_have_conf:
                logger.info('[DISPATCH][%s] IDLE SATELLITE   The %s scheduler seems to be waiting for a conf.' % (realm_name, sched.get_name()))
                self._unset_scheduler_conf(sched)
                sched.need_conf = True
                # only send a real spare conf if it's not the very first loop, at least let a dispatch
                # make all assignation
                if first_dispatch_done:
                    logger.info('[DISPATCH][%s] SET AS SPARE   The %s scheduler do not have any shard to manage, so it will be set as a spare until it is assigned a shard.' % (realm_name, sched.get_name()))
                    # LONG CALL
                    sched.create_and_put_inactive_scheduler_configuration(arbiter_trace, configuration_incarnation)
        
        return dispatch_ok
    
    
    # Maybe satellites are alive, but do not have a cfg yet.
    # I think so. It is not good. I ask a global redispatch for the shard_id I think is not correctly dispatched.
    def check_satellites_dispatch(self, first_dispatch_done):
        dispatch_ok = True
        realm_name = self.get_name()
        for shard_id in self.shards:
            push_flavor = self.shards[shard_id].push_flavor
            for kind in ('reactionner', 'poller', 'broker', 'receiver'):
                # We must have the good number of satellite or we are not happy
                # So we are sure to raise a dispatch every loop a satellite is missing
                _nb_managed = len(self.to_satellites_managed_by[kind][shard_id])
                _nb_must_have = self.get_nb_of_must_have_satellites(kind)
                if _nb_managed < _nb_must_have:
                    # Only raise a WARNING if we did go with a dispatch at least one
                    if first_dispatch_done:
                        logger.warning("[DISPATCH][%s] MISSING SATELLITE   Missing satellite %s for shard %d:" % (realm_name, kind, shard_id))
                    # find only the dead satellites and do the work only for them.
                    # others already have the right conf
                    current_sat_set = set(self.to_satellites_managed_by[kind][shard_id])
                    potential_sat_set = set(self.get_potential_satellites_by_type(kind))
                    dead_sats = potential_sat_set - current_sat_set
                    dispatch_ok = False  # so we will redispatch all
                    self.to_satellites_need_dispatch[kind][shard_id] = True
                    for sat in dead_sats:
                        if sat in self.to_satellites_managed_by[kind][shard_id]:
                            self.to_satellites_managed_by[kind][shard_id].remove(sat)
                # A satellite has coming back, we now have too many sat for this kind,
                # we need to remove some sat from to_satellites_managed_by and ask a redispatch
                elif len(self.to_satellites_managed_by[kind][shard_id]) > self.get_nb_of_must_have_satellites(kind):
                    satellites = self.to_satellites_managed_by[kind][shard_id]
                    satellites.sort(alive_then_spare_then_deads)
                    dispatch_ok = False  # so we will redispatch all
                    self.to_satellites_need_dispatch[kind][shard_id] = True
                    to_remove_sat = satellites[-self.get_nb_of_must_have_satellites(kind):]
                    edited = False
                    for sat in to_remove_sat:
                        if sat in self.to_satellites_managed_by[kind][shard_id]:
                            self.to_satellites_managed_by[kind][shard_id].remove(sat)
                            edited = True
                    if edited:
                        logger.warning("[DISPATCH][%s] TOO MANY SATELLITES   too many %s for shard %d:, we don't need some spare anymore" % (realm_name, kind, shard_id))
                for satellite in self.to_satellites_managed_by[kind][shard_id]:
                    # Maybe the sat was marked as not alive, but still in to_satellites_managed_by. That means that a new dispatch  is needed
                    # Or maybe it is alive but I thought that this reactionner managed the conf
                    # and it doesn't. I ask a full redispatch of these cfg for both cases
                    
                    if push_flavor == 0 and satellite.alive:
                        logger.error('[DISPATCH][%s] UNMATCHED SHARD   The %s %s manage a unmanaged shard' % (realm_name, kind, satellite.get_name()))
                        continue
                    if not satellite.alive:
                        logger.error('[DISPATCH][%s] DEAD SATELLITE    The %s %s seems to be down, I must re-dispatch its role to someone else.' % (realm_name, kind, satellite.get_name()))
                        dispatch_ok = False  # so we will redispatch all
                        self.to_satellites_need_dispatch[kind][shard_id] = True
                        self.to_satellites_managed_by[kind][shard_id].remove(satellite)
                    if satellite.reachable and not satellite.do_i_manage(shard_id, push_flavor):
                        logger.error('[DISPATCH][%s] BAD CONF FOR SATELLITE    The %s %s seems to manage an old conf, I must re-dispatch.' % (realm_name, kind, satellite.get_name()))
                        dispatch_ok = False  # so we will redispatch all
                        self.to_satellites_need_dispatch[kind][shard_id] = True
                        self.to_satellites_managed_by[kind][shard_id].remove(satellite)
                # find in the potential satellites the spares that start late and are waiting for a conf
                for satellite in self.get_potential_satellites_by_type(kind):
                    if satellite.alive and satellite.reachable and not satellite.already_have_conf:
                        logger.info('[DISPATCH][%s] A SATELLITE    The %s %s seems to be waiting for a conf' % (realm_name, kind, satellite.get_name()))
                        dispatch_ok = False  # so we will redispatch all
                        self.to_satellites_need_dispatch[kind][shard_id] = True
                        if satellite not in self.to_satellites_managed_by[kind][shard_id]:
                            self.to_satellites_managed_by[kind][shard_id].append(satellite)
        return dispatch_ok
    
    
    # Imagine a world where... oh no, wait...
    # Imagine a master got the conf and the network is down a spare takes it (good :) ). Like the Empire, the master
    # strikes back! It was still alive! (like Elvis). It still got conf and is running! not good!
    # Bad dispatch: a link that has a conf but I do not allow this so I ask it to wait a new conf and stop kidding.
    def check_bad_dispatch(self, other_realms, first_dispatch_done, arbiter_trace):
        for scheduler in self.schedulers:
            # If element has a conf, I do not care, it's a good dispatch
            # If dead: I do not ask it something, it won't respond..        # LONG CALL HERE
            if scheduler.conf is None and scheduler.reachable and scheduler.have_conf():
                txt = "[DISPATCH] IDLE DAEMON WITH SHARD   The element %s have a shard but should not have one. We are asking it to go in sleep mode." % scheduler.get_name()
                if not first_dispatch_done:
                    logger.info(txt)
                else:
                    logger.error(txt)
                scheduler.active = False
                scheduler.wait_new_conf()  # LONG CALL
                # I do not care about order not send or not. If not, The next loop will resent it
        
        # I ask satellites which sched_id they manage. If I do not agree, I ask them to remove it
        for satellite in self._get_all_satellites():
            if not satellite.reachable:
                continue
            
            satellite_name = satellite.get_name()
            kind = satellite.get_my_type()
            
            shard_ids = satellite.managed_confs
            # I do not care about satellites that do nothing, they already do what I want :)
            if len(shard_ids) == 0:
                continue
            
            ids_to_delete = []
            for shard_id in shard_ids:
                # Ok, we search for realms that have the conf
                for realm in other_realms:
                    if shard_id in realm.shards:
                        # Ok we've got the realm, we check its to_satellites_managed_by
                        # to see if satellite is in. If not, we remove the sched_id for it
                        if satellite not in realm.to_satellites_managed_by[kind][shard_id]:
                            ids_to_delete.append(shard_id)
            
            # Let's clean
            for _id in ids_to_delete:
                if first_dispatch_done:
                    logger.info("[DISPATH] REMOVE SHARD FROM SATELLITE     I ask to remove shard %d from satellite %s" % (_id, satellite_name))
                else:  # oh it's the first turn, so let know it's normal
                    logger.info("[DISPATH] CLEAN AT START      I ask to remove shard %d from satellite %s so we can send a new one" % (_id, satellite_name))
                satellite.remove_from_conf(_id)
            
            # If a spare came back to void, set as inactive
            is_void_now = len(ids_to_delete) == len(shard_ids)
            if first_dispatch_done and satellite.spare and is_void_now:
                self._set_satellite_as_inactive(satellite, arbiter_trace)
                logger.info('[DISPATCH][%s][SPARE/IDLE] The %s %s was set to INACTIVE as it do not manage any more shards/schedulers' % (self.get_name(), kind, satellite.get_name()))
    
    
    def dispatch_schedulers(self, arbiter_trace, configuration_incarnation):
        shards_to_dispatch = self.get_not_assigned_shards()
        realm_name = self.get_name()
        nb_shards_to_dispatch = len(shards_to_dispatch)
        if nb_shards_to_dispatch > 0:
            logger.info("[DISPATCH][%s]  Dispatching shards and satellites " % realm_name)
            logger.info('[DISPATCH][%s]     Dispatching %d/%d shards to schedulers' % (realm_name, nb_shards_to_dispatch, len(self.shards)))
        
        # Now we get in scheds all scheduler of this realm and upper so
        # we will send them conf (in this order)
        potential_schedulers = [scheduler for scheduler in self.get_scheduler_ordered_list() if scheduler.alive]
        if nb_shards_to_dispatch > 0:
            print_string = "[DISPATCH][%s]     Shards will be dispatched to schedulers in this order: " % realm_name
            print_string += ', '.join(['%s (realm=%s, spare=%s), ' % (scheduler.get_name(), scheduler.realm.get_name(), str(scheduler.spare)) for scheduler in reversed(potential_schedulers)])
            logger.info(print_string)
        
        # Now we do the real job
        for shard in shards_to_dispatch:
            shard_id = shard.id
            logger.info('[DISPATCH][%s] DISPATCH SHARD    Dispatching shard [%s]' % (realm_name, shard_id))
            
            # If there is no alive schedulers, not good...
            if len(potential_schedulers) == 0:
                logger.error('[DISPATCH][%s] NO SCHEDULERS    but there a no alive schedulers in this realm!' % realm_name)
            
            # we need to loop until the conf is assigned or when there are no more schedulers available
            while True:
                # No more schedulers.. not good, no loop
                # => the conf does not need to be dispatch
                if len(potential_schedulers) == 0:
                    self.reset_all_satellite_association_for_shard_id(shard_id)
                    break
                
                # Ok there are still some schedulers
                scheduler = potential_schedulers.pop()
                
                is_sent = scheduler.create_and_put_active_scheduler_configuration(shard, arbiter_trace, configuration_incarnation)
                # Maybe the scheduler do not even need a shard, or maybe the send fail.In all cases, just loop to the next scheduler
                if not is_sent:
                    continue
                
                # Let the realm know that satellites need to talk to the scheduler now
                self.declare_shard_sent_to_a_scheduler(shard, scheduler)
                
                # Ok, the conf is dispatched, no more loop for this configuration
                break
    
    
    # Get a list of satellite that are available for this realm and
    # * are alive
    # * are reacheable
    # * sorted by nearest and not spare first
    def _get_dispatchable_satellite(self, kind):
        # make copies of potential_react list for sort
        valid_realm_satellites = []
        for satellite in self.get_potential_satellites_by_type(kind):
            # skip not reacheable ones
            if not satellite.alive or not satellite.reachable:
                continue
            valid_realm_satellites.append(satellite)
        valid_realm_satellites.sort(alive_then_spare_then_deads)
        return valid_realm_satellites
    
    
    def _dispatch_shard_to_satellite(self, satellite, kind, shard_id, shard_for_satellite_part, shard, flavor, arbiters_cfg, arbiter_trace):
        satellite.cfg['schedulers'][shard_id] = shard_for_satellite_part
        satellite.cfg['arbiters'] = arbiters_cfg
        satellite.cfg['activated'] = True
        
        is_sent = False
        # Maybe this satellite already got this configuration, so skip it
        if satellite.do_i_manage(shard_id, flavor):
            logger.info('[DISPATCH][%s] ALREADY MANAGED SHARD Skipping shard [%d] send to the %s %s: it already manage it' % (self.get_name(), shard_id, kind, satellite.get_name()))
            is_sent = True
        else:  # ok, it really need it :)
            logger.info('[DISPATCH][%s] SATELLITE SENT START Trying to send shard [%s] to %s %s' % (self.get_name(), shard_id, kind, satellite.get_name()))
            
            # see bottom of page for extended_conf explanation
            extended_conf = satellite.cfg
            arbiter_trace['arbiter_time'] = time.time()
            extended_conf['arbiter_trace'] = arbiter_trace
            
            # LONG CALL
            is_sent = satellite.put_conf(extended_conf)
        
        if is_sent:
            satellite.active = True
            logger.info('[DISPATCH][%s] SATELLITE SENT OK  Dispatch OK of shard [%s] to %s %s' % (self.get_name(), shard_id, kind, satellite.get_name()))
            # We change the satellite configuration, update our data
            satellite.known_conf_managed_push(shard_id, flavor)
            
            if satellite not in self.to_satellites_managed_by[kind][shard_id]:
                self.to_satellites_managed_by[kind][shard_id].append(satellite)
            
            # If receiver, we must send:
            # * the hostnames of this configuration
            # * the host inventory (name, uuid and such things)
            if kind == 'receiver':
                shard_host_names = shard.get_managed_host_names()  # get the list of the host names (not uuid) that this shard manage
                logger.info("[DISPATCH][%s] RECEIVER INVENTORY UPDATE Sending %s hostnames to the receiver %s" % (self.get_name(), len(shard_host_names), satellite.get_name()))
                satellite.push_host_names(shard_id, shard_host_names)
        
        return is_sent
    
    
    def _dispatch_set_satellite_as_spare_for_a_shard(self, satellite, kind, arbiters_cfg, shard_id, flavor, arbiter_trace):
        # I've got enough satellite, the next ones are considered spares
        # send config to them but remove the scheduler
        satellite.cfg['schedulers'] = {}
        satellite.cfg['activated'] = False
        satellite.cfg['arbiters'] = arbiters_cfg
        
        is_sent = False
        # Maybe this satellite already got this configuration, so skip it
        if satellite.do_i_manage(shard_id, flavor):
            logger.info('[DISPATCH][%s][SPARE/IDLE] ALREADY MANAGED SHARD Skipping shard %d send to the %s %s: it already manage it' % (self.get_name(), shard_id, kind, satellite.get_name()))
            is_sent = True
        else:  # ok, it really need it :)
            logger.info('[DISPATCH][%s][SPARE/IDLE] SATELLITE SENT START  Trying to send shard to %s %s' % (self.get_name(), kind, satellite.get_name()))
            # LONG CALL
            is_sent = self._set_satellite_as_inactive(satellite, arbiter_trace)
        
        if is_sent:
            # satellite.active = False
            logger.info('[DISPATCH][%s][SPARE/IDLE] SATELLITE SENT OK  Dispatch OK of shard %s to %s %s' % (self.get_name(), shard_id, kind, satellite.get_name()))
        return is_sent
    
    
    def _set_satellite_as_inactive(self, satellite, arbiter_trace):
        satellite.cfg['schedulers'] = {}
        satellite.cfg['activated'] = False
        arbiter_trace['arbiter_time'] = time.time()
        satellite.cfg['arbiter_trace'] = arbiter_trace
        # LONG CALL
        is_sent = satellite.put_conf(satellite.cfg)
        return is_sent
    
    
    def assert_receivers_inventories(self):
        ########
        # Assert that the receivers have the good inventories
        ########
        # make copies of potential_react list for sort
        all_valid_realm_receivers = self._get_dispatchable_satellite('receiver')
        # Only push inventory to the receiver that need it
        only_shardable_receivers = [receiver for receiver in all_valid_realm_receivers if receiver.elements_sharding_enabled]
        
        # must be sorted to be sure we always have the same dispatch
        sorted_shardable_receivers = sorted(only_shardable_receivers, key=lambda x: x.get_name())
        
        # number_of_receiver_shards = len(sorted_shardable_receivers)
        receiver_weights = [receiver.elements_sharding_weight for receiver in sorted_shardable_receivers]
        sum_of_receiver_weights = sum(receiver_weights)
        previous_elements_weight_sum = 0
        
        for (receiver_index, receiver) in enumerate(sorted_shardable_receivers):
            # realm inventory
            inventory_for_receiver = self._get_inventory_for_receiver(previous_elements_weight_sum, receiver.elements_sharding_weight, sum_of_receiver_weights, receiver_index, receiver_weights)
            
            # Be sure that this receiver have this realm inventory up-to-date
            receiver.assert_have_realm_inventory(self.get_name(), inventory_for_receiver)
            
            # Increate the weight sum for the next receiver
            previous_elements_weight_sum += receiver.elements_sharding_weight
    
    
    def assert_brokers_inventories(self):
        # make copies of potential_react list for sort
        all_valid_realm_brokers = self._get_dispatchable_satellite('broker')
        
        # All brokers are taking the same inventory (full with all hosts/checks)
        inventory_for_broker = self._get_inventory_for_broker()
        for broker in all_valid_realm_brokers:
            # Be sure that this receiver have this realm inventory up-to-date
            broker.assert_have_realm_inventory(self.get_name(), inventory_for_broker)
    
    
    def dispatch_satellites(self, arbiters_cfg, arbiter_trace):
        # General dispatch: let each satellite know about a shard place (on which scheduler) and other nodes too
        for shard in self.shards.itervalues():
            shard_id = shard.id
            # flavor if the push number of this configuration send to a scheduler
            flavor = shard.push_flavor
            for kind in ('reactionner', 'poller', 'broker', 'receiver'):
                # Maybe we have nothing to do
                if not self.to_satellites_need_dispatch[kind][shard_id]:
                    continue
                
                shard_for_satellite_part = self.to_satellites[kind][shard_id]
                
                # make copies of potential_react list for sort
                valid_realm_satellites = self._get_dispatchable_satellite(kind)
                
                if len(valid_realm_satellites) == 0:
                    logger.info("[DISPATCH][%s] SATELLITES CANNOT SENT  There are no available %s satellites to manage shard %s" % (self.get_name(), kind, shard_id))
                    continue
                
                # Dump the order where we will send conf
                satellite_string = "[DISPATCH][%s] SATELLITE ORDER  Dispatching %s satellite with be done in this order: " % (self.get_name(), kind)
                for satellite in valid_realm_satellites:
                    satellite_string += '%s (spare:%s), ' % (satellite.get_name(), str(satellite.spare))
                logger.info(satellite_string)
                
                # Now we dispatch cfg to every one ask for it
                nb_cfg_sent = 0
                for satellite in valid_realm_satellites:
                    if not satellite.alive:  # we set to dead for a reason
                        logger.error('[DISPATCH][%s] the %s named %s seems to be down and will not get any conf' % (self.get_name(), kind, satellite.get_name()))
                        continue
                    
                    # Send only if we need, and if we can
                    if nb_cfg_sent < self.get_nb_of_must_have_satellites(kind):
                        is_sent = self._dispatch_shard_to_satellite(satellite, kind, shard_id, shard_for_satellite_part, shard, flavor, arbiters_cfg, arbiter_trace)
                        if is_sent:
                            nb_cfg_sent += 1
                    else:
                        is_sent = self._dispatch_set_satellite_as_spare_for_a_shard(satellite, kind, arbiters_cfg, shard_id, flavor, arbiter_trace)
                        if is_sent:
                            nb_cfg_sent += 1
                
                if nb_cfg_sent >= self.get_nb_of_must_have_satellites(kind):
                    logger.info("[DISPATCH][%s] SATELLITES ALL SENT  No more %s sent need for this realm" % (self.get_name(), kind))
                    self.to_satellites_need_dispatch[kind][shard_id] = False
    
    
    # We will fill all higher_realms from our sons with ourselve
    def fill_sub_realms_with_myself_as_higher_realm(self, level=0):
        for realm in self.realm_members:
            realm.add_higher_realm(self)
    
    
    def add_higher_realm(self, higher_realm, level=0):
        if level > MAX_REALMS_LEVEL:  # Loop?
            return
        if higher_realm not in self.higher_realms:
            self.higher_realms.append(higher_realm)
        
        for realm in self.realm_members:
            realm.add_higher_realm(higher_realm, level + 1)


class Realms(Itemgroups):
    name_property = "realm_name"  # is used for finding hostgroups
    inner_class = Realm
    
    
    def get_members_by_name(self, pname):
        realm = self.find_by_name(pname)
        if realm is None:
            return []
        return realm.get_realms()
    
    
    def linkify(self, only_schedulers=False):
        self.linkify_p_by_p()
        
        # prepare list of satellites and confs
        for p in self:
            p.pollers = []
            p.schedulers = []
            p.reactionners = []
            p.brokers = []
            p.providers = []
            p.receivers = []
            p.packs = []
            p.only_schedulers = only_schedulers
    
    
    # We just search for each realm the others realms
    # and replace the name by the realm
    def linkify_p_by_p(self):
        for p in self.items.values():
            mbrs = p.get_realm_members()
            # The new member list, in id
            new_mbrs = []
            for mbr in mbrs:
                new_mbr = self.find_by_name(mbr)
                if new_mbr is not None:
                    new_mbrs.append(new_mbr)
            # We find the id, we replace the names
            p.realm_members = new_mbrs
        
        # Also put in each realm the list of its higher realms
        # Need to set higher_realms as we are not in the pythonize part
        for r in self.items.values():
            r.higher_realms = []
        
        # Only fill higher realms if there is no loop in th realm definition
        realm_with_loops = [realm for realm in self if realm._got_loop]
        if realm_with_loops:
            logger.error('There are realms with loop, cannot compute realm tree')
        else:
            for realm in self.items.values():
                realm.fill_sub_realms_with_myself_as_higher_realm()  # DO NOT CALL THIS ONE IF THERE ARE LOOPS!
    
    
    # Use to fill members with hostgroup_members
    def explode(self):
        default_realm = self.get_default()
        if not default_realm:
            return None
        
        for p in self:
            p.get_realms_by_explosion(self)
        
        # We clean the tags
        for tmp_p in self.items.values():
            del tmp_p._already_explode
    
    
    def get_parent_realm_for(self, realm_name):
        """Returns the parent realm corresponding to the string realm_name, or None if not
        found. Returns a Realm class or None."""
        for r in self:
            members_lst = r.get_realm_members()
            if realm_name in members_lst:
                return r
    
    
    def get_path_to_default(self, realm_name):
        """Return a list of nested realms to traverse to join the default (root) realm"""
        realms = [realm_name]
        current_realm = realm_name
        iter_max = 15  # catch loop in realm config
        iter_current = 0
        while iter_current < iter_max:
            parent_realm = self.get_parent_realm_for(current_realm)
            if parent_realm is None:
                break
            parent_realm_name = parent_realm.get_name()
            # detect loop
            if parent_realm_name == current_realm:
                break
            realms.append(parent_realm_name)
            current_realm = parent_realm_name
            iter_current += 1
        
        return reversed(realms)
    
    
    def get_default(self):
        default_realm = None
        for r in self:
            if getattr(r, 'default', False) in ("1", True):
                # There can be only one default realm
                if default_realm:
                    return None
                
                default_realm = r
        return default_realm
    
    
    def prepare_for_satellites_conf(self):
        for r in self:
            r.prepare_for_satellites_conf()
    
    
    def check_default_realm(self):
        default_realms = []
        for r in self:
            if getattr(r, 'default', False) in ("1", True):
                default_realms.append(r.get_name())
        
        if len(default_realms) == 1:
            self.conf_is_correct = True
            return True
        elif len(default_realms) == 0:
            self.configuration_errors.append("There must be one default realm defined. Please set one by setting the 'default' property to '1' for one realm.")
            self.conf_is_correct = False
        elif len(default_realms) > 1:
            self.configuration_errors.append("There can be only one default realm ; the following realms are set as default : %s" % ", ".join(default_realms))
            self.conf_is_correct = False
    
    
    def is_correct(self):
        is_realms_conf_correct = super(Realms, self).is_correct()
        
        self.conf_is_correct = is_realms_conf_correct and self.check_default_realm()
        return self.conf_is_correct
