#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import shlex
import subprocess
import threading
import time

from shinken.log import logger

from ..dao.def_items import NAGIOS_TABLE_KEYS

# If the Arbiter is reloading for more than 1200 seconds, consider him as dead
ARBITER_RELOADING_TIMEOUT = 1200
# If the Arbiter arbiter restart for more than RESTART_TIMEOUT sec, return rc 0 to force the apply reload
RESTART_TIMEOUT = 30

FORBIDDEN_TYPES = ['servicedependency', 'hostdependency', 'serviceescalation', 'hostescalation', 'hostextinfo', 'serviceextinfo', 'checkmodulation']
FORBIDDEN_TYPES.extend(NAGIOS_TABLE_KEYS.keys())
FORBIDDEN_TYPES = ' '.join(FORBIDDEN_TYPES)


class ARBITER_AREA(object):
    CHECK_PROPOSE = 'CHECK_PROPOSE'
    CHECK_STAGGING = 'CHECK_STAGGING'
    CHECK_PREPROD = 'CHECK_PREPROD'
    
    KEYS = (CHECK_PROPOSE, CHECK_STAGGING, CHECK_PREPROD)


class ArbiterController(object):
    def __init__(self, app):
        self.app = app
        
        # used to know if arbiter is restarting, value can be False or starting date
        self.arbiter_reloading = False
        self.arbiter_reloading_lock = threading.RLock()
    
    
    @staticmethod
    def _build_environment_variable_for_arbiter(arbiter_load):
        # NOTE: both LANGUAGE and LANG even if only LANGUAGE is need, because newcomers know only about the LANG=
        env = {'FORBIDDEN_TYPES': FORBIDDEN_TYPES, 'LANG': 'en_US.utf-8', 'LANGUAGE': 'en_US.utf-8'}
        for k in ARBITER_AREA.KEYS:
            env[k] = '1' if k == arbiter_load else '0'
        return env
    
    
    def launch_arbiter_check(self, area_check):
        cmd = shlex.split('sudo /usr/bin/env python -u /usr/sbin/shinken-arbiter -v -c /etc/shinken/shinken.cfg')
        env = self._build_environment_variable_for_arbiter(area_check)
        p = subprocess.Popen(cmd, stdin=None, stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=env)
        stdoutdata, stderrdata = p.communicate()
        rc = p.returncode
        output = stderrdata + stdoutdata
        return rc, output
    
    
    def reload_arbiter(self):
        # set the arbiter_reloading at
        with self.arbiter_reloading_lock:
            self.arbiter_reloading = time.time()
        
        cmd = shlex.split('sudo /etc/init.d/shinken-arbiter restart')
        p = subprocess.Popen(cmd, stdin=None, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        stdoutdata, stderrdata = p.communicate()
        rc = p.returncode
        
        # If arbiter don't restart release the reloading lock
        if rc != 0:
            with self.arbiter_reloading_lock:
                self.arbiter_reloading = False
        
        output = stderrdata + stdoutdata
        return rc, output
    
    
    # check if the master is alive and if the spare must_run
    def get_arbiters_aliveness(self):
        enabled_arbiters = self.app.conf.arbiters.enabled()
        
        if not enabled_arbiters:
            return {'master': None}
        arbiter_master = None
        arbiter_spare = None
        for arb in enabled_arbiters:
            if not arb.spare:
                arbiter_master = arb
            else:
                arbiter_spare = arb
        original_arbiter_timeout = arbiter_master.timeout
        arbiter_master.reset_connection()
        arbiter_master.timeout = 1
        try:
            arbiter_master.ping()
        except Exception:
            arbiter_master.reachable = False
        arbiter_master.timeout = original_arbiter_timeout
        # create a dict with master and spare as key and a boolean representing the reachability
        result = {'master': arbiter_master.reachable}
        # someone reload the master
        if self.arbiter_reloading:
            result["master_reloading"] = self.arbiter_reloading
        # if arbiter master is dead
        if not arbiter_master.reachable:
            # if the arbiter is marked as reloading for more than ARBITER_RELOADING_TIMEOUT seconds, remove the reloading
            if self.arbiter_reloading and (time.time() - self.arbiter_reloading) > ARBITER_RELOADING_TIMEOUT:
                with self.arbiter_reloading_lock:
                    self.arbiter_reloading = False
                    result.pop("master_reloading", None)
        # if arbiter master is alive but was reloading previously
        if not arbiter_master.reachable and arbiter_spare:
            # the master is not reachable, try to ping the spare
            try:
                arbiter_spare.ping()
            except Exception:
                arbiter_spare.reachable = False
            result["spare"] = arbiter_spare.reachable
        return result
    
    
    def wait_arbiter_have_reload(self):
        # wait for the arbiter to be ready
        arbiter_ready = False
        while not arbiter_ready:
            if (time.time() - self.arbiter_reloading) > RESTART_TIMEOUT:
                logger.error('arbiter restarting for more than %s sec, return rc 0 to force the apply reload' % RESTART_TIMEOUT)
                break
            arbiters_aliveness = self.get_arbiters_aliveness()
            arbiter_ready = arbiters_aliveness.get('master', False)  # and not app.arbiter_reloading
        
        # arbiter is now reachable and ready to work, remove the arbiter_reloading
        with self.arbiter_reloading_lock:
            logger.debug('[apply] reload arbiter done')
            self.arbiter_reloading = False
