#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.
from collections import namedtuple

from shinken.log import logger

from shinken.objects.service import PREFIX_LINK_UUID

from ..dao.def_items import DEF_ITEMS, ITEM_TYPE, METADATA, LINKIFY_MANAGE_STATES, SERVICE_OVERRIDE, ITEM_STATE, SERVICE_OVERRIDE_UUID_SEP, SERVICE_EXCLUDES_BY_ID
from ..dao.helpers import get_item_property, set_item_property, split_and_strip_list, get_name_from_type, ShinkenDatabaseConsistencyError
from ..dao.parsers.bp_rule_parser import BpRuleParser
from ..dao.parsers.complex_exp_parser import parse_complex_exp, OPERAND, do_match
from shinkensolutions.service_override_parser import parse_service_override_property


# get a link value and return True if the links was (before 02.06.xx) made with id ( not name)
# replace the 'item_links_with_id' in DEF_ITEM
def _is_item_linked_by_name(link_val):
    return link_val not in ('last_modification.contact', 'work_area_info.user')


def _build_command_link(item, find_by_name_func, link_prop_name, link_prop_item_type):
    # find_by_name_func is a pointer on the function tu use (datamanagerv2.find_item or dataprovider._find..) needed because of read/write lock
    if not link_prop_item_type == ITEM_TYPE.COMMANDS:
        return False
    raw_value = item.get(link_prop_name, None)
    if not raw_value or not isinstance(raw_value, basestring):
        return False
    command_link = _build_command_link_for_value(raw_value, find_by_name_func)
    set_item_property(item, link_prop_name, command_link)
    return True


def _build_link_bp_rule(item, find_by_name_func):
    raw_value = item.get('bp_rule', None)
    if not raw_value or not isinstance(raw_value, basestring):
        return False
    try:
        bp_rule_link = _build_link_bp_rule_for_value(raw_value, find_by_name_func)
    except SyntaxError as e:
        logger.warning('The bp_rule [%s] of item [%s] cannot be parse : %s' % (raw_value, item['_id'], e))
        bp_rule_link = {
            'raw_value': raw_value,
            'node'     : None
        }
    if hasattr(item, 'set_value'):
        item.set_value('bp_rule', bp_rule_link)
    else:
        item['bp_rule'] = bp_rule_link
    return True


def _build_link_service_overrides_for_properties(override, item_type, datamanagerV2, add_link=False):
    # Compute links for overridden property if necessary
    property_name = override['key']
    property_value = override['value']
    if isinstance(property_value, dict):
        return
    if isinstance(property_value, list):
        property_value = ",".join(property_value)
    
    check_type = DEF_ITEMS[item_type]['check_type']
    property_type = DEF_ITEMS[check_type]['props_links'].get(property_name, '')
    if len(property_type) > 1:
        raise ShinkenDatabaseConsistencyError(override['check_link'])
    elif len(property_type) == 1:
        property_type = property_type[0]
        linked_value = _build_prop_link_for_value(property_value, property_type, datamanagerV2.find_item_by_name, add_link=add_link)
    else:
        linked_value = property_value
    
    override['value'] = linked_value


CheckIndex = namedtuple("CheckIndex", ['check', 'check_type', 'check_index', 'host_id', 'host_type'])


def _compute_check_index(check, check_type, host, host_type, host_state, host_templates):
    """ this method compute check_index which is use for select between 2 check with same name attache on one host which check is active
return : check, check_type, check_index, host_id, host_type
 * for a check on host or on cluster is 0
 * for a check on host template or cluster template it is the index + 1 in use of tpls in check host_name
 *   index + 1 -> so check on host is higher than check on host template
    :rtype:  CheckIndex
    """
    from ..dao.items.mixins import HostNameMixin
    from ..dao.items import get_item_instance
    
    if check_type in (ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESCLUSTERS):
        host_id_on_check = [i['_id'] for i in check.get('host_name', {}).get('links', []) if i['exists']]
        host_name_on_check = [i['name'] for i in check.get('host_name', {}).get('links', []) if not i['exists']]
        host_id = host['_id']
        if host_id in host_id_on_check or host['host_name'] in host_name_on_check:
            if host_state == ITEM_STATE.NEW:
                # New items are not link see SEF-4508
                host_id = None
                host_type = None
            return CheckIndex(check, check_type, 0 + 10000 * (0 if check.get('enabled', '1') == '1' else 1), host_id, host_type)
        else:
            return CheckIndex(check, check_type, -1, None, None)
    else:
        host_name_node = check.get('host_name', {}).get('node')
        if host_name_node and do_match(host_name_node, host_templates):
            check_item = get_item_instance(check_type, check)
            tpl_on_check = HostNameMixin._get_links_host_name(check_item, 'host_name')
            tpl_on_check_ids = [i['_id'] for i in tpl_on_check if i['exists']]
            tpl_on_check_names = [i['name'] for i in tpl_on_check if not i['exists']]
            
            for index, host_template in enumerate(host_templates):
                if host_template['_id'] in tpl_on_check_ids or host_template['name'] in tpl_on_check_names:
                    check_index = index + 1 + 10000 * (0 if check.get('enabled', '1') == '1' and host_template.get('enabled', '1') == '1' else 1)
                    return CheckIndex(check, check_type, check_index, host_template['_id'], DEF_ITEMS[host_type]['template'])
            # Hard crash -> check do_match but host don't have any template which match the host_name of the check
            raise Exception('For compute check_index the host must have the check')
        else:
            return CheckIndex(check, check_type, -1, None, None)


def _find_all_templates(item, item_type, datamanagerV2, known_templates=None):
    templates = []
    inherited_templates = []
    template_names = item.get('use', [])
    if not known_templates:
        known_templates = []
    
    if not template_names:
        return templates
    
    if isinstance(template_names, dict):
        template_names = datamanagerV2.flatten_prop(item, item_type, 'use')
    
    if not template_names:
        return templates
    
    if isinstance(template_names, basestring):
        template_names = template_names.split(',')
    
    template_names = [name for name in template_names if name not in known_templates]
    _item_type = DEF_ITEMS[item_type]['template']
    for name in template_names:
        template = datamanagerV2.find_item_by_name(item_name=name, item_type=_item_type, item_state=ITEM_STATE.STAGGING)
        if template:
            templates.append(template)
            known_templates.append(get_name_from_type(item_type, template))
            inherited_templates.extend(_find_all_templates(template, _item_type, datamanagerV2, known_templates=known_templates))
    
    templates.extend(inherited_templates)
    
    return templates


def _compute_check_id_from_check_name_for_service_excludes_by_id(all_templates, check_link_name, check_types, datamanagerV2, item, item_type, item_state, service_excludes_link):
    checks = []
    check_link_id = ''
    if check_link_name.startswith(PREFIX_LINK_UUID):
        (check_link_name, check_link_id) = check_link_name.split(PREFIX_LINK_UUID)
    
    for check_type in check_types:
        check_key_name = DEF_ITEMS[check_type]['key_name']
        _select = {'_id': check_link_id} if check_link_id else {check_key_name: check_link_name}
        _checks = datamanagerV2.find_merge_state_items(where=_select, item_type=check_type, item_states=LINKIFY_MANAGE_STATES)
        checks.extend([_compute_check_index(c, check_type, item, item_type, item_state, all_templates) for c in _checks])
    
    checks.sort(key=lambda x: x.check_index)
    
    # Remove checks with check indexes < 0 from *SORTED* list
    while checks and checks[0].check_index < 0:
        checks.pop(0)
    
    # A check has match and is enabled
    if checks and 0 <= checks[0].check_index < 10000:  # and (get_name_from_type(checks[0].check_type, checks[0].check), service_excludes_link['key']) not in override_already_take:
        service_excludes_link.clear()
        service_excludes_link['item_type'] = checks[0].check_type
        service_excludes_link['_id'] = checks[0].check['_id']
        service_excludes_link['exists'] = True
    elif check_link_name:
        service_excludes_link.clear()


def _update_check_host_link_with_check_name_for_override(all_templates, check_link_name, datamanagerV2, item, item_type, item_state, override_link, override_already_take):
    host_link = override_link['host_link']
    check_link = override_link['check_link']
    def_item = DEF_ITEMS[item_type]
    check_types = set([def_item['check_type'], DEF_ITEMS[def_item['template']]['check_type']])
    checks = []
    check_link_id = ''
    if SERVICE_OVERRIDE_UUID_SEP in check_link_name:
        _split = check_link_name.split(SERVICE_OVERRIDE_UUID_SEP)
        check_link_name = _split[0]
        check_link_id = _split[1]
    
    for check_type in check_types:
        check_key_name = DEF_ITEMS[check_type]['key_name']
        _select = {'_id': check_link_id} if check_link_id else {check_key_name: check_link_name}
        _checks = datamanagerV2.find_merge_state_items(where=_select, item_type=check_type, item_states=LINKIFY_MANAGE_STATES)
        checks.extend([_compute_check_index(c, check_type, item, item_type, item_state, all_templates) for c in _checks])
    checks.sort(key=lambda x: x.check_index)
    # A check has match and is enabled
    if checks and 0 <= checks[0].check_index < 10000 and (get_name_from_type(checks[0].check_type, checks[0].check), override_link['key']) not in override_already_take:
        check = checks[0].check
        check_type = checks[0].check_type
        host_id = checks[0].host_id
        host_type = checks[0].host_type
        override_link['has_already_been_linked'] = True
        
        check_link.clear()
        check_link['item_type'] = check_type
        check_link['_id'] = check['_id']
        check_link['exists'] = True
        
        host_link.clear()
        if host_id is not None:
            host_link['item_type'] = host_type
            host_link['_id'] = host_id
            host_link['exists'] = True
        override_already_take.append((get_name_from_type(checks[0].check_type, checks[0].check), override_link['key']))
    else:
        override_link['has_already_been_linked'] = False
        check_link.clear()
        check_link['name'] = check_link_name
        check_link['exists'] = False
        
        host_link.clear()


def _build_service_override_cache(service_override):
    links_cache = []
    for override in service_override['links']:
        override_check_link = override['check_link']
        if override_check_link['exists']:
            links_cache.append(override_check_link['item_type'])
        else:
            links_cache.append('__ALL_SERVICES__')
        
        if override.get('host_link', None):
            override_host_link = override['host_link']
            if override_host_link['exists']:
                links_cache.append(override['host_link']['item_type'])
            else:
                links_cache.append('__ALL_HOSTS__')
        
        value = override.get('value', None)
        if value and isinstance(value, dict):
            property_type = DEF_ITEMS[ITEM_TYPE.SERVICETPLS]['props_links'].get(override['key'], [])
            if len(property_type) == 1:
                links_cache.append(property_type[0])
            # links_cache.extend([prop_link['item_type'] for prop_link in value['links'] if prop_link['exists']])
    
    service_override['all_linked_types'] = list(set(links_cache))


def update_frontend_service_override_links_into_links(item, item_type, item_state, datamanagerV2):
    if not item[SERVICE_OVERRIDE]:
        return
    
    if isinstance(item[SERVICE_OVERRIDE], basestring):
        _build_link_service_overrides(item, item_type, item_state, datamanagerV2)
        return
    
    all_templates = _find_all_templates(item, item_type, datamanagerV2)
    if ITEM_TYPE.is_template(item_type):
        item['can_match_by_name'] = True
        all_templates.insert(0, item)
    
    override_already_take = []
    override_to_update = []
    for override in item[SERVICE_OVERRIDE]['links']:
        check_link = override['check_link']
        host_link = override.get('host_link', {})
        override['host_link'] = host_link
        
        check_link_name = check_link['name']
        
        if override.get('has_already_been_linked', False) and not check_link['exists']:
            check_link.clear()
            check_link['name'] = check_link_name
            check_link['exists'] = False
            
            host_link.clear()
            continue
        
        # First we get all the checks which might match
        if check_link['exists']:
            override['has_already_been_linked'] = True
            check_link_id = check_link['_id']
            check_link_type = check_link['item_type']
            host_link_id = host_link['_id']
            host_link_type = host_link['item_type']
            checks = datamanagerV2.find_merge_state_items(where={'_id': check_link_id}, item_type=check_link_type, item_states=LINKIFY_MANAGE_STATES)
            checks = [_compute_check_index(c, check_link_type, item, item_type, item_state, all_templates) for c in checks]
            
            # The host have a check on him (event disable)
            if [i for i in checks if i.check_index != -1]:
                check_link.clear()
                check_link['item_type'] = check_link_type
                check_link['_id'] = check_link_id
                check_link['exists'] = True
                
                host_link.clear()
                if host_link_id is not None:
                    host_link['item_type'] = host_link_type
                    host_link['_id'] = host_link_id
                    host_link['exists'] = True
                
                override_already_take.append((check_link_name, override['key']))
            else:
                override['has_already_been_linked'] = False
                check_link.clear()
                check_link['name'] = check_link_name
                check_link['exists'] = False
                
                host_link.clear()
        else:
            override_to_update.append(override)
    
    for override in override_to_update:
        check_link_name = override['check_link']['name']
        if override.get('has_already_been_linked', False) and not override['check_link']['exists'] or override['check_link']['exists']:
            continue
        
        _update_check_host_link_with_check_name_for_override(all_templates, check_link_name, datamanagerV2, item, item_type, item_state, override, override_already_take)
    
    item.pop('can_match_by_name', None)


def _build_link_service_excludes_by_id(item, item_type, item_state, datamanagerV2, add_link=False):
    value_to_build_link = item.get(SERVICE_EXCLUDES_BY_ID, '')
    if not value_to_build_link:
        return False
    
    if item_type == ITEM_TYPE.HOSTS:
        ordered_list_of_types_to_check = (ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESHOSTTPLS)
    elif item_type == ITEM_TYPE.HOSTTPLS:
        ordered_list_of_types_to_check = (ITEM_TYPE.SERVICESHOSTTPLS,)
    elif item_type == ITEM_TYPE.CLUSTERS:
        ordered_list_of_types_to_check = (ITEM_TYPE.SERVICESCLUSTERS, ITEM_TYPE.SERVICESCLUSTERTPLS)
    elif item_type == ITEM_TYPE.CLUSTERTPLS:
        ordered_list_of_types_to_check = (ITEM_TYPE.SERVICESCLUSTERTPLS,)
    else:
        return False
    
    _edited = False
    
    if not isinstance(value_to_build_link, basestring):
        value_to_build_link['has_error'] = value_to_build_link.get('has_error', False)
        value_to_build_link['error'] = value_to_build_link.get('error', '')
        
        if add_link:
            for link in value_to_build_link['links']:
                if link['exists'] and not link.get('@link'):
                    linked_item = datamanagerV2.find_item_by_id(link['_id'], item_type=link['item_type'], item_state=ITEM_STATE.STAGGING)
                    link['@link'] = {linked_item.get_state(): linked_item}
        
        return True
    
    all_templates = _find_all_templates(item, item_type, datamanagerV2)
    new_links = []
    for check_link_name in split_and_strip_list(value_to_build_link):
        service_excludes_link = {}
        _compute_check_id_from_check_name_for_service_excludes_by_id(all_templates, check_link_name, ordered_list_of_types_to_check, datamanagerV2, item, item_type, item_state, service_excludes_link)
        if service_excludes_link:
            new_links.append(service_excludes_link)
            if service_excludes_link['exists']:
                if add_link:
                    linked_item = datamanagerV2.find_item_by_id(service_excludes_link['_id'], item_type=service_excludes_link['item_type'], item_state=ITEM_STATE.STAGGING)
                    service_excludes_link['@link'] = {linked_item.get_state(): linked_item}
    
    if new_links:
        excludes_value = {
            'raw_value' : item[SERVICE_EXCLUDES_BY_ID],
            'links'     : new_links,
            'has_errors': False,
            'has_plus'  : False
        }
        if hasattr(item, 'set_value'):
            item.set_value(SERVICE_EXCLUDES_BY_ID, excludes_value)
        else:
            item[SERVICE_EXCLUDES_BY_ID] = excludes_value
        _edited = True
    else:
        item.pop(SERVICE_EXCLUDES_BY_ID)
        _edited = True
    
    return _edited


def _build_link_service_overrides(item, item_type, item_state, datamanagerV2, add_link=False):
    value_to_build_link = item.get(SERVICE_OVERRIDE, '')
    if not value_to_build_link:
        return False
    
    if not isinstance(value_to_build_link, basestring):
        value_to_build_link['has_error'] = value_to_build_link.get('has_error', False)
        value_to_build_link['error'] = value_to_build_link.get('error', '')
        _build_service_override_cache(item[SERVICE_OVERRIDE])
        
        if add_link:
            links_to_linkify = []
            for override_link in value_to_build_link['links']:
                override_check_link = override_link['check_link']
                if override_check_link['exists'] and not override_check_link.get('@link'):
                    links_to_linkify.append(override_check_link)
                
                if override_link.get('host_link', None):
                    override_host_link = override_link['host_link']
                    if override_host_link['exists'] and not override_host_link.get('@link'):
                        links_to_linkify.append(override_host_link)
                
                value = override_link.get('value', None)
                if value and isinstance(value, dict):
                    links_to_linkify.extend([prop_link for prop_link in value['links'] if prop_link['exists']])
                
                for link in links_to_linkify:
                    linked_item = datamanagerV2.find_item_by_id(link['_id'], item_type=link['item_type'], item_state=ITEM_STATE.WORKING_AREA) or \
                                  datamanagerV2.find_item_by_id(link['_id'], item_type=link['item_type'], item_state=ITEM_STATE.STAGGING)
                    link['@link'] = {linked_item.get_state(): linked_item}
        return True
    
    parsed_overrides = {}
    error_message = ""
    has_error = False
    try:
        parsed_overrides = parse_service_override_property(value_to_build_link)
    except SyntaxError as e:
        has_error = True
        error_message = 'The service_overrides [%s] of item [%s] cannot be parsed. %s' % (value_to_build_link, item['_id'], e)
    except KeyError:
        has_error = True
        error_message = "The service_overrides [%s] of item [%s] defines several overrides for the same property of the same check " % (value_to_build_link, item['_id'])
    
    override_links = []
    for check_name, parsed_override in parsed_overrides.iteritems():
        for property_name, property_value in parsed_override.iteritems():
            override_link = {
                'key'       : property_name,
                'value'     : property_value,
                'check_link': {
                    'name': check_name
                },
                'host_link' : {}
            }
            
            _build_link_service_overrides_for_properties(override_link, item_type, datamanagerV2, add_link=False)
            override_links.append(override_link)
    
    all_templates = _find_all_templates(item, item_type, datamanagerV2)
    if ITEM_TYPE.is_template(item_type):
        item['can_match_by_name'] = True
        all_templates.insert(0, item)
    
    override_already_take = []
    for override_link in override_links:
        check_link_name = override_link['check_link']['name']
        _update_check_host_link_with_check_name_for_override(all_templates, check_link_name, datamanagerV2, item, item_type, item_state, override_link, override_already_take)
    
    edited_override = {
        'has_error': has_error,
        'error'    : error_message,
        'raw_value': value_to_build_link,
        'links'    : override_links,
    }
    _build_service_override_cache(edited_override)
    
    if hasattr(item, 'set_value'):
        item.set_value(SERVICE_OVERRIDE, edited_override)
    else:
        item[SERVICE_OVERRIDE] = edited_override
    
    item.pop('can_match_by_name', None)
    return True


def _build_link_bp_rule_for_value(property_value, find_by_name_func):
    node = BpRuleParser.compute_internal_bp_rule(property_value)
    all_linked_types = _set_id_bprule_node(node, find_by_name_func)
    bp_rule_link = {
        'raw_value'       : property_value,
        'node'            : node,
        'all_linked_types': all_linked_types
    }
    return bp_rule_link


def _get_link_for_check_name(key, value, item, item_type, check_name, checks, datamanagerv2):
    # type: (basestring, basestring, dict, ITEM_TYPE, basestring, list, DataManagerV2 ) -> dict
    
    content = {
        'exists': False,
        'key'   : key,
        'value' : value
    }
    
    for tpl_name, check_list in checks.iteritems():
        matching_checks = [check for check in check_list if check.get('service_description') == check_name]
        if len(matching_checks) == 1:
            if get_name_from_type(item_type, item) == tpl_name:
                tpl_id = item['_id']
                tpl_type = item_type
            else:
                tpl_type = DEF_ITEMS[item_type]['template']
                tpl_items = datamanagerv2.find_merge_state_items(item_type=tpl_type, item_states=LINKIFY_MANAGE_STATES, where={'name': tpl_name})
                if len(tpl_items) == 0:
                    continue
                tpl_id = tpl_items[0]['_id']
            content['check_link'] = {
                '_id'      : matching_checks[0].get('_id', None),
                'item_type': METADATA.get_metadata(matching_checks[0], METADATA.ITEM_TYPE),
                'exists'   : True
            }
            content['host_link'] = {
                '_id'      : tpl_id,
                'item_type': tpl_type,
                'exists'   : True
            }
            break
    if 'host_link' not in content:
        content['name'] = check_name
    
    return content


def _set_id_bprule_node(bp_rule, find_by_name_func):
    found_type = set()
    if not bp_rule or isinstance(bp_rule, basestring):
        return list(found_type)
    for token in bp_rule:
        if BpRuleParser._is_linkable_token(token):
            content = {
                'exists': False,
                'name'  : token['content']
            }
            item_type = token.get('item_type', [])
            for _type in item_type:
                for item_state in LINKIFY_MANAGE_STATES:
                    
                    item = find_by_name_func(token['content'], item_type=_type, item_state=item_state)
                    if item:
                        content = {
                            '_id'      : item.get('_id', None),
                            'item_type': _type,
                            'exists'   : True
                        }
                        token['item_type'] = _type
                        found_type.add(_type)
                        break
            if not content.get('_id'):
                for available_item_type in token['item_type']:
                    found_type.add(available_item_type)
            token['content'] = content
    return list(found_type)


def _build_link_complex_exp(item, item_type, item_property, find_by_name_func):
    if item_property == 'host_name':
        item_type_to_link_with = ITEM_TYPE.HOSTTPLS if ITEM_TYPE.SERVICESHOSTTPLS == item_type else ITEM_TYPE.CLUSTERTPLS
    else:
        item_type_to_link_with = ITEM_TYPE.HOSTGROUPS
    
    raw_value = item.get(item_property, None)
    if not raw_value or not isinstance(raw_value, basestring):
        return False
    try:
        host_name_link = _build_link_complex_exp_for_value(raw_value, item_type, find_by_name_func, item_type_to_link_with)
    except SyntaxError as e:
        logger.warning('The complex exp [%s] of item [%s] cannot be parse.%s' % (raw_value, item['_id'], e))
        host_name_link = {
            'raw_value': raw_value,
            'node'     : None
        }
    
    if hasattr(item, 'set_value'):
        item.set_value(item_property, host_name_link)
    else:
        item[item_property] = host_name_link
    return True


def _build_link_complex_exp_for_value(property_value, item_type, find_by_name_func, item_type_to_link_with):
    host_name_link = {
        'raw_value': property_value,
        'node'     : parse_complex_exp(property_value)
    }
    _set_id_complexexp(item_type, host_name_link['node'], find_by_name_func, item_type_to_link_with)
    return host_name_link


def _set_id_complexexp(item_type, node, find_by_name_func, item_type_to_link_with):
    if not node:
        return
    
    for node_sons in node['sons']:
        _set_id_complexexp(item_type, node_sons, find_by_name_func, item_type_to_link_with)
    
    template_name = node.get('content', '')
    if node['operand'] == OPERAND.TEMPLATE_OP and template_name:
        node['content'] = {
            'exists': False,
            'name'  : template_name
        }
        for item_state in LINKIFY_MANAGE_STATES:
            link_item = find_by_name_func(template_name, item_type_to_link_with, item_state)
            if link_item:
                node['content']['exists'] = True
                node['content']['item_type'] = item_type_to_link_with
                node['content']['_id'] = link_item['_id']
                del node['content']['name']
                break


def _build_command_link_for_value(property_value, find_by_name_func):
    command_link = {
        'raw_value': property_value,
    }
    
    splited_res = property_value.split('!', 1)
    if len(splited_res) > 1:
        command_name = splited_res[0]
        args = splited_res[1]
    else:
        command_name = splited_res[0]
        args = ''
    
    if not command_name:
        return command_link
    
    for item_state in LINKIFY_MANAGE_STATES:
        command_item = find_by_name_func(command_name, item_type=ITEM_TYPE.COMMANDS, item_state=item_state)
        if command_item:
            content = {
                '_id'      : command_item.get('_id', None),
                'item_type': ITEM_TYPE.COMMANDS,
                'exists'   : True
            }
            command_link['node'] = {
                'link': content,
                'args': args,
            }
            break
    if not 'node' in command_link and command_name:
        content = {
            'name'  : command_name,
            'exists': False
        }
        command_link['node'] = {
            'link': content,
            'args': args,
        }
    return command_link


def _build_prop_link_for_value(property_value, item_type, find_by_name_func, add_link=False):
    has_plus = False
    if property_value.startswith('+'):
        has_plus = True
        property_value = property_value[1:]
    item_values = split_and_strip_list(property_value)
    new_style_link = []
    
    for item_value in item_values:
        link_value = {
            u'name'  : item_value,
            u'exists': False
        }
        
        for linkify_state in LINKIFY_MANAGE_STATES:
            link_value_item = find_by_name_func(item_value, item_type, linkify_state)
            if link_value_item:
                link_value[u'_id'] = link_value_item[u'_id']
                link_value[u'item_type'] = item_type
                link_value[u'exists'] = True
                if add_link:
                    link_value[u'@link'] = {METADATA.get_metadata(link_value_item, METADATA.STATE): link_value_item}
                del link_value[u'name']
                break
        new_style_link.append(link_value)
    
    new_value = {
        u'has_plus': has_plus,
        u'links'   : new_style_link,
    }
    return new_value


def _build_prop_link(item, datamanagerV2, link_prop_name, link_prop_item_type):
    if _is_item_linked_by_name(link_prop_name):
        item_values = get_item_property(item, link_prop_name)
        if not item_values or not isinstance(item_values, basestring):
            return False
        
        has_plus = False
        if item_values.startswith('+'):
            has_plus = True
            item_values = item_values[1:]
        item_values = split_and_strip_list(item_values)
        new_style_link = []
        
        for item_value in item_values:
            link_value = {
                u'name'  : item_value,
                u'exists': False
            }
            
            for linkify_state in LINKIFY_MANAGE_STATES:
                link_value_item = datamanagerV2.find_item_by_name(item_value, link_prop_item_type, linkify_state)
                if link_value_item:
                    link_value[u'_id'] = link_value_item[u'_id']
                    link_value[u'item_type'] = link_prop_item_type
                    link_value[u'exists'] = True
                    del link_value[u'name']
                    break
            new_style_link.append(link_value)
        
        new_value = {
            u'has_plus': has_plus,
            u'links'   : new_style_link,
        }
        set_item_property(item, link_prop_name, new_value)
    
    else:
        tmp = link_prop_name.split('.', 1)
        link_prop_dict = item.get(tmp[0], {})
        value_id = link_prop_dict.get(tmp[1], None)
        value_name = link_prop_dict.get(tmp[1] + '_name', 'shinken-core')
        
        if not value_id or not isinstance(value_id, basestring):
            return False
        
        new_value = {
            u'has_plus': False,
            u'links'   : [{
                u'name'  : value_name,
                u'exists': False
            }],
        }
        for linkify_state in LINKIFY_MANAGE_STATES:
            link_value_item = datamanagerV2.find_item_by_id(value_id, link_prop_item_type, linkify_state)
            
            if link_value_item:
                new_value[u'links'][0][u'_id'] = value_id
                new_value[u'links'][0][u'item_type'] = link_prop_item_type
                new_value[u'links'][0][u'exists'] = True
                del new_value[u'links'][0][u'name']
                break
        
        set_item_property(item, link_prop_name, new_value)
    
    return True


def build_link(item, item_type, item_state, datamanagerV2):
    edited = False
    find_by_name_func = datamanagerV2.find_item_by_name
    
    if item_type in (ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS):
        _edited = _build_link_bp_rule(item, find_by_name_func)
        edited = edited or _edited
    if item_type in (ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESCLUSTERTPLS):
        _edited = _build_link_complex_exp(item, item_type, 'host_name', find_by_name_func)
        edited = edited or _edited
    if item_type in (ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESHOSTS):
        _edited = _build_link_complex_exp(item, item_type, 'hostgroup_name', find_by_name_func)
        edited = edited or _edited
    if item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERTPLS):
        _edited = _build_link_service_overrides(item, item_type, item_state, datamanagerV2)
        edited = edited or _edited
        _edited = _build_link_service_excludes_by_id(item, item_type, item_state, datamanagerV2)
        edited = edited or _edited
    
    # /!\ BEWARE - if a property appears in several item_links entries in DEF_ITEMS, a specific build_link method is needed
    for link_prop_item_type, link_prop_names in DEF_ITEMS[item_type]['item_links'].iteritems():
        # Example when shinken_type='hostgroups' {link_key, link_values}
        # may be: {'hosts' : ['members']}
        for link_prop_name in link_prop_names:
            # all commands except from notificationways have args and are special links
            if link_prop_item_type == 'commands' and item_type not in (ITEM_TYPE.NOTIFICATIONWAYS, ITEM_TYPE.CONTACTS, ITEM_TYPE.CONTACTTPLS):
                _edited = _build_command_link(item, find_by_name_func, link_prop_name, link_prop_item_type)
                edited = edited or _edited
            else:
                # start the transformation from string to dict link
                _edited = _build_prop_link(item, datamanagerV2, link_prop_name, link_prop_item_type)
                edited = edited or _edited
    return edited
