#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import json
from ...business.item_controller.exceptions import MassiveChangeError
from ...business.item_controller.massive_change.massive_change_type import MASSIVE_CHANGE_TYPE
from ...business.item_controller.massive_change.massive_change_step import MASSIVE_CHANGE_STEP
from ...dao.def_items import DEF_ITEMS, ITEM_TYPE, ITEM_STATE, SERVICE_EXCLUDES_BY_ID

app = None
datamanagerV2 = None


# for Massive change specification see SEF-4712


def _add_step_to_title(title, step):
    title += ' > %s' % step
    return title


def __compute_massive_change_title(step, type_modification):
    title = "%s" % app._('mass-change.title')
    if step <= 1:
        return title
    if type_modification == MASSIVE_CHANGE_TYPE.EXCLUDES:
        title = _add_step_to_title(title, app._('mass-change.step-1.types.excludes'))
    elif type_modification == MASSIVE_CHANGE_TYPE.OVERRIDES:
        title = _add_step_to_title(title, app._('mass-change.step-1.types.overrides'))
        if step > 2:
            title = _add_step_to_title(title, app._('mass-change.step-1.types.properties'))
    elif type_modification == MASSIVE_CHANGE_TYPE.LOCAL_DATAS:
        title = _add_step_to_title(title, app._('mass-change.step-1.types.datas'))
        title = _add_step_to_title(title, app._('mass-change.step-1.types.local_datas'))
    elif type_modification == MASSIVE_CHANGE_TYPE.DFE_DATAS:
        title = _add_step_to_title(title, app._('mass-change.step-1.types.datas'))
        title = _add_step_to_title(title, app._('mass-change.step-1.types.dfe_datas'))
    else:
        title = _add_step_to_title(title, app._('mass-change.step-1.types.properties'))
    return title


def _parse_received_data():
    app.response.content_type = 'application/json'
    user = app.get_user_auth()
    raw_data = app.request.body.read()
    try:
        mc_form = json.loads(raw_data)
    except ValueError as e:
        return app.abort(400, 'Cannot parse received form to json : %s' % e.message)
    
    try:
        step = int(mc_form.get('step'))
    except ValueError as e:
        return app.abort(400, 'Step is not correct : %s' % e.message)
    
    return user, mc_form, step


def _massive_change_step_1(mc_form, item_type, data_for_tpl):
    data_for_tpl['has_check'] = "shinken-no-checks"
    data_for_tpl['has_datas'] = "shinken-no-datas"
    data_for_tpl['has_datas_dfe'] = "shinken-no-datas-dfe"
    if item_type == ITEM_TYPE.HOSTS:
        data_for_tpl['has_check'] = ""
        data_for_tpl['has_datas'] = ""
        data_for_tpl['has_datas_dfe'] = ""
    elif item_type == ITEM_TYPE.HOSTTPLS:
        data_for_tpl['has_check'] = ""
        data_for_tpl['has_datas'] = ""
        data_for_tpl['has_datas_dfe'] = ""
    elif item_type == ITEM_TYPE.CLUSTERS:
        data_for_tpl['has_datas'] = ""
        data_for_tpl['has_datas_dfe'] = ""
    elif item_type == ITEM_TYPE.CLUSTERTPLS:
        data_for_tpl['has_datas'] = ""
        data_for_tpl['has_datas_dfe'] = ""
    elif item_type in ITEM_TYPE.ALL_SERVICES:
        data_for_tpl['has_datas'] = ""
    elif item_type == ITEM_TYPE.CONTACTS:
        data_for_tpl['has_datas'] = ""
    elif item_type == ITEM_TYPE.CONTACTTPLS:
        data_for_tpl['has_datas'] = ""
    elif item_type == ITEM_TYPE.MACROMODULATIONS:
        data_for_tpl['has_datas'] = ""


def _massive_change_step_2(mc_form, item_type, data_for_tpl):
    # List check
    if item_type not in [ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS]:
        return app.abort(400, 'Cannot get checks for item type : %s.' % item_type)
    
    try:
        checks_list, ignored_items, missing_items, all_items = app.massive_change_controller.get_active_checks_from_item_dict(mc_form['items'], item_type)
    except MassiveChangeError as e:
        return app.abort(400, 'Error during apply changes on items : %s' % e.text)
    data_for_tpl['massive_change_step_2_checks'] = checks_list
    data_for_tpl['ignored_items'] = ignored_items
    data_for_tpl['missing_items'] = missing_items
    data_for_tpl['all_items'] = all_items


def _massive_change_step_3(mc_form, item_type, data_for_tpl, type_modification):
    # List data
    try:
        if type_modification == MASSIVE_CHANGE_TYPE.LOCAL_DATAS:
            data_list, ignored_items, missing_items, host_list, template_list = app.massive_change_controller.get_data_for_items(mc_form['items'], [item_type])
        elif type_modification == MASSIVE_CHANGE_TYPE.DFE_DATAS:
            data_list, ignored_items, missing_items, host_list, template_list = app.massive_change_controller.get_dfe_for_items(mc_form['items'], [item_type])
        else:
            tmp = {}
            for check in mc_form['checks']:
                tmp[check[2]['_id']] = check[2]
            data_list, ignored_items, missing_items, host_list, template_list = app.massive_change_controller.get_data_for_check_overrides(tmp)
    except MassiveChangeError as e:
        return app.abort(400, 'Error during apply changes on items : %s' % e.text)
    data_for_tpl['data_list'] = data_list
    data_for_tpl['ignored_items'] = ignored_items
    data_for_tpl['missing_items'] = missing_items
    data_for_tpl['host_list'] = host_list
    data_for_tpl['template_list'] = template_list


def _massive_change_step_4(mc_form, item_type, data_for_tpl, user):
    # Validate changes
    try:
        massive_change = app.massive_change_controller.apply_changes(mc_form, item_type, user=user)
    except MassiveChangeError as e:
        return app.abort(400, 'Error during apply changes on items : %s' % e.text)
    
    data_for_tpl.update(massive_change)
    data_for_tpl['nb_modif'] = len(mc_form['actions'])
    if data_for_tpl['type_modification'] == MASSIVE_CHANGE_TYPE.EXCLUDES:
        for host_uuid, excludes in data_for_tpl['update'].get(SERVICE_EXCLUDES_BY_ID, {}).iteritems():
            data_for_tpl['nb_modif'] += len(excludes)
    data_for_tpl['actions'] = mc_form['actions']


def _massive_change_step_5(mc_form, item_type, data_for_tpl, user):
    # Save changes
    try:
        massive_change = app.massive_change_controller.apply_changes(mc_form, item_type, user=user, dry_run=False)
    except MassiveChangeError as e:
        return app.abort(400, 'Error during apply changes on items : %s' % e.text)
    data_for_tpl.update(massive_change)
    data_for_tpl['nb_modif'] = len(mc_form['actions'])
    if data_for_tpl['type_modification'] == MASSIVE_CHANGE_TYPE.EXCLUDES:
        for host_uuid, excludes in data_for_tpl['update'].get(SERVICE_EXCLUDES_BY_ID, {}).iteritems():
            data_for_tpl['nb_modif'] += len(excludes)
    data_for_tpl['actions'] = mc_form['actions']


def massive_change_call(item_type):
    user, mc_form, step = _parse_received_data()
    
    if not mc_form.get('items', []):
        return app.abort(400, 'There is no items sent to massive_change_call')
    
    type_modification = mc_form.get('type_modification', None)
    
    data_for_tpl = {
        'helper'           : app.helper,
        'helper_object'    : app.helper.helper_object,
        'step'             : step,
        'type_modification': type_modification,
        'app'              : app,
        'user'             : user,
        'item_type'        : item_type,
        'protected_fields' : DEF_ITEMS[item_type].get('protected_fields', []),
        'items'            : mc_form['items'],
        'update'           : mc_form.get('update', {}),
    }
    
    if step == MASSIVE_CHANGE_STEP._1:
        _massive_change_step_1(mc_form, item_type, data_for_tpl)
    elif step == MASSIVE_CHANGE_STEP._2:
        # list_checks
        _massive_change_step_2(mc_form, item_type, data_for_tpl)
    elif step == MASSIVE_CHANGE_STEP._3 and type_modification in (MASSIVE_CHANGE_TYPE.LOCAL_DATAS, MASSIVE_CHANGE_TYPE.DFE_DATAS, MASSIVE_CHANGE_TYPE.OVERRIDES):
        # list_datas
        _massive_change_step_3(mc_form, item_type, data_for_tpl, type_modification)
    elif step == MASSIVE_CHANGE_STEP._3 and type_modification in (MASSIVE_CHANGE_TYPE.PROPERTIES) and item_type in [ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS]:
        user_list = []
        for contact in datamanagerV2.find_items(ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING):
            user_list.append({
                'contact_name': contact.get_name(),
                'uuid'        : contact['_id'],
                'item_type'   : contact.get_type(),
                'is_admin'    : contact.is_admin(),
                'is_expert'   : contact.is_expert()
            })
        
        data_for_tpl['user_list'] = user_list
    
    elif step == MASSIVE_CHANGE_STEP._4:
        # Validate
        _massive_change_step_4(mc_form, item_type, data_for_tpl, user)
    elif step == MASSIVE_CHANGE_STEP._5:
        # Edit
        _massive_change_step_5(mc_form, item_type, data_for_tpl, user)
    return data_for_tpl
