from .http_client import HTTPExceptions
from .satellitelink import SatelliteLink
from .log import logger


class WithInventorySatelliteLink(SatelliteLink):
    properties = SatelliteLink.properties.copy()
    
    
    def __init__(self, *args, **kwargs):
        super(WithInventorySatelliteLink, self).__init__(*args, **kwargs)
        # We will have the list of realms that this receiver can be contacted with
        self._accessible_realms = []
    
    
    def __getstate__(self):
        _link_as_dict = super(WithInventorySatelliteLink, self).__getstate__()
        _link_as_dict['_accessible_realms'] = self._accessible_realms
        return _link_as_dict
    
    
    def __setstate__(self, state):
        super(WithInventorySatelliteLink, self).__setstate__(state)
        self._accessible_realms = state['_accessible_realms']
    
    
    # A realm is giving it's name to know that this receiver can be contacted by it
    def set_accessible_realm(self, realm_name):
        self._accessible_realms.append(realm_name)
    
    
    # The receiver need to have a new entry in the configuration: 'known_realms'
    # that was prepared by the realms just before
    def prepare_for_conf(self, configuration_incarnation):
        super(WithInventorySatelliteLink, self).prepare_for_conf(configuration_incarnation)
        self.cfg['known_realms'] = self._accessible_realms
        logger.info('[%s] The %s %s can be reachable by the realms: %s' % (self.my_type, self.realm.get_name(), self.get_name(), ', '.join(self._accessible_realms)))
    
    
    # For a specific realm, we want to know if the receiver have the inventory up-to-date
    # * ask it for its current inventory hash. If matching, also check that the shard index&number of shards are OK too
    # * if not matching, push the new one (inventory + index + shard_size)
    def assert_have_realm_inventory(self, realm_name, realm_inventory):
        try:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                self.add_failed_check_attempt()
                return
            
            self.con.get('ping')
            inventory_hash = realm_inventory.get_hash()
            args = {
                'realm_name'    : realm_name,
                'inventory_hash': inventory_hash
            }
            already_have = self.con.get('already_have_realm_inventory', args=args)
            if already_have:
                logger.debug('[DISPATCH][%s] The %s %s does already have a valid inventory (%s elements)' % (realm_name, self.my_type, self.get_name(), realm_inventory.get_len()))
            else:
                logger.info('[DISPATCH][%s] Starting to send realm inventory to the %s %s' % (realm_name, self.my_type, self.get_name()))
                payload = {
                    'realm_name'     : realm_name,
                    'realm_inventory': realm_inventory
                }
                self.con.post('push_realm_inventory', payload, wait='long')
                logger.info('[DISPATCH][%s] The %s %s is now up-to-date with inventory %s (%s elements)' % (self.my_type, realm_name, self.get_name(), inventory_hash, realm_inventory.get_len()))
                # We did push both inventory and shard_index, it's sure to be OK
        except HTTPExceptions as exp:
            err = 'while calling /push_realm_inventory: %s' % str(exp)
            self.add_failed_check_attempt(reason=err)
