#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.basemodule import BaseModule, SOURCE_STATE
from shinken.misc.type_hint import NoReturn, Any, Optional

from shinkensolutions.api.synchronizer import ITEM_TYPE, component_manager, COMPONENT, SourceTranslatePart
from shinkensolutions.api.synchronizer.source.compute_sync_keys import SyncKeysManager
from shinkensolutions.api.synchronizer.source.display_origin_item.display_origin_item_format import DisplayOriginItemFormat
from shinkensolutions.api.synchronizer.source.item.item_container import ItemsContainers
from shinkensolutions.api.synchronizer.source.mapping.origin_to_source.mapper import MapperOriginToSource, ProxyDisableMapper, GenericDictMapperOriginToSource, ReloadMapperEvent
from shinkensolutions.api.synchronizer.source.mapping.source_to_conf.mapper import MapperSourceToConf
from shinkensolutions.api.synchronizer.source.origin_item_description import OriginItemDescription
from shinkensolutions.api.synchronizer.source.route.route_container import RouteContainer
from shinkensolutions.api.synchronizer.source.rules_application_template import SourceRulesManager, ReloadSourceRulesManagerEvent
from shinkensolutions.api.synchronizer.source.source_configuration import SourceConfiguration
from shinkensolutions.api.synchronizer.source.source_configuration_value import SERVICE_MODE, ServiceMode
from shinkensolutions.api.synchronizer.source.source_exception import SourceException
from shinkensolutions.api.synchronizer.source.tab.tab_container import TabContainer
from shinkensolutions.api.synchronizer.source.tab.tab_host_template_binding_rules import RouteReloadTemplateBindingRules
from shinkensolutions.api.synchronizer.source.tab.tab_host_template_binding_rules import TabHostTemplateBindingRules
from shinkensolutions.api.synchronizer.source.tab.tab_mapping_origin_to_source import RouteReloadMappingOriginToSource
from shinkensolutions.api.synchronizer.source.tab.tab_mapping_origin_to_source import TabMappingOriginToSource
from shinkensolutions.api.synchronizer.source.source_import_progression import SourceImportProgression
from shinkensolutions.api.synchronizer.source.file_loader import FileLoader
from shinkensolutions.api.synchronizer.source.api_item_properties import ApiItemProperties
from shinkensolutions.lib_modules.configuration_reader import read_string_in_configuration, read_list_in_configuration
from shinkensolutions.api.synchronizer.source.source_info import SourceInfo


class SOURCE_TYPES(object):
    COLLECTOR = 'collector'
    LISTENER = 'listener'
    ANALYZER = 'analyzer'


class SourceModule(BaseModule):
    
    def __init__(self, module_configuration, source_configuration=None):
        # type:(Any, Optional[SourceConfiguration])->NoReturn
        BaseModule.__init__(self, module_configuration)
        self.source_name = read_string_in_configuration(module_configuration, 'source_name', '')
        self.source_type = read_string_in_configuration(module_configuration, 'module_type', '')
        
        self._module_configuration = module_configuration
        self.route_container = RouteContainer()
        self.tab_container = TabContainer(self.route_container)
        self.type_to_import = []
        self.source_configuration = source_configuration or SourceConfiguration()
        self.translator = None  # type: Optional[SourceTranslatePart]
        self.sync_keys_manager = None  # type: Optional[SyncKeysManager]
        
        self.mapper_origin_to_source = None  # type: Optional[MapperOriginToSource]
        
        self._state_of_display_origin_item = ''  # type: ServiceMode
        
        self._state_of_mapping_origin_to_source = ''  # type: ServiceMode
        self._display_origin_item_format = None  # type: Optional[DisplayOriginItemFormat]
        self._mapper_source_to_conf = MapperSourceToConf(self.logger)
        self._reload_mapper = None  # type: Optional[ReloadMapperEvent]
        
        self._load_error_message = ''
        self.source_import_progression = SourceImportProgression()
        self.file_loader = None  # type: Optional[FileLoader]
        self.source_info = SourceInfo()
        
        self._state_of_host_template_binding_rule = ''  # type: ServiceMode
        self.host_template_binding_rule_manager = None  # type: Optional[SourceRulesManager]
        self._reload_source_rules_manager = None  # type: Optional[ReloadSourceRulesManagerEvent]
        
        self._state_of_origin_item_properties_description = ''  # type: ServiceMode
        self.origin_item_description = None  # type: Optional[OriginItemDescription]
        
        self._state_of_api_item_properties = ''  # type: ServiceMode
        self.api_item_properties = None  # type: Optional[ApiItemProperties]
    
    
    def load(self, _):
        try:
            self._load()
        except Exception as e:
            fail_to_load_source_message = 'Fail to load source with error : [%s]' % e
            self.logger.error(fail_to_load_source_message)
            self.logger.print_stack()
            self._load_error_message = fail_to_load_source_message
    
    
    def get_source_import_progression(self):
        return self.source_import_progression
    
    
    def _load(self):
        self.logger.info('module initialization start')
        self.type_to_import = read_list_in_configuration(self._module_configuration, 'type_to_import', '')
        self.translator = component_manager.get_component(COMPONENT.TRANSLATE).source_translator(self.source_type)
        self.file_loader = FileLoader(self.source_type, self.source_name)
        
        self._build_api_item_properties_filter()
        self._build_origin_item_description()
        self._build_sync_keys_manager()
        self._build_host_template_binding_rule()
        self._build_mapping_origin_to_source()
        self._build_display_origin_item()
        
        self.load_configuration(self._module_configuration)
    
    
    def _build_display_origin_item(self):
        self._state_of_display_origin_item = self.source_configuration._state_of_display_origin_item
        self._display_origin_item_format = self.source_configuration._display_origin_item_format
        if SERVICE_MODE.is_enable(self._state_of_display_origin_item) and self._display_origin_item_format:
            self._display_origin_item_format._load(self)
    
    
    def _build_mapping_origin_to_source(self):
        rule_component = component_manager.get_component(COMPONENT.RULE)
        self._state_of_mapping_origin_to_source = self.source_configuration._state_of_mapping_origin_to_source
        _origin_to_source_type_mapper = self.source_configuration._origin_to_source_type_mapper or GenericDictMapperOriginToSource
        _origin_to_source_mappers_name = list(self.source_configuration._origin_to_source_mappers_name)
        
        if SERVICE_MODE.is_enable(self._state_of_mapping_origin_to_source):
            self.mapper_origin_to_source = MapperOriginToSource(
                self.logger,
                self.translator,
                rule_component,
                self.file_loader,
                self._state_of_mapping_origin_to_source,
                _origin_to_source_type_mapper,
                self.origin_item_description,
                self.api_item_properties,
                _origin_to_source_mappers_name
            )
            
            self.tab_container.add(TabMappingOriginToSource(self.logger, self.translator, self.source_name, self.mapper_origin_to_source, self.type_to_import, self.source_info))
            self._reload_mapper = ReloadMapperEvent(self.mapper_origin_to_source)
            self.route_container.add(RouteReloadMappingOriginToSource(self.logger, self.source_name, self.mapper_origin_to_source, self._reload_mapper, self.source_info))
        else:
            self.mapper_origin_to_source = ProxyDisableMapper()
    
    
    def _build_host_template_binding_rule(self):
        self._state_of_host_template_binding_rule = self.source_configuration._state_of_host_template_binding_rule
        if SERVICE_MODE.is_enable(self._state_of_host_template_binding_rule):
            self.host_template_binding_rule_manager = SourceRulesManager(
                self.logger,
                self.file_loader,
                self._state_of_host_template_binding_rule,
                self.translator,
                self.api_item_properties
            )
            
            tab_host_template_binding_rules = TabHostTemplateBindingRules(
                self.logger,
                self.translator,
                self.source_name,
                self.host_template_binding_rule_manager,
                self.source_info
            )
            self.tab_container.add(tab_host_template_binding_rules)
            
            self._reload_source_rules_manager = ReloadSourceRulesManagerEvent(self.host_template_binding_rule_manager)
            route_reload_template_binding_rules = RouteReloadTemplateBindingRules(
                self.logger,
                self.source_name,
                self.host_template_binding_rule_manager,
                self._reload_source_rules_manager,
                self.source_info
            )
            self.route_container.add(route_reload_template_binding_rules)
    
    
    def _build_sync_keys_manager(self):
        properties_used_as_synckey = {}
        for item_type in ITEM_TYPE.ALL_TYPES:
            properties_used_as_synckey[item_type] = read_list_in_configuration(self._module_configuration, 'properties_used_as_synckey_for_%s' % item_type, '')
        
        self.sync_keys_manager = SyncKeysManager(self.logger, properties_used_as_synckey)
    
    
    def _build_origin_item_description(self):
        self._state_of_origin_item_properties_description = self.source_configuration._state_of_origin_item_properties_description
        origin_item_description_class = self.source_configuration._origin_item_description_class or OriginItemDescription
        self.origin_item_description = origin_item_description_class(
            self.logger,
            self.translator,
            self.file_loader,
            component_manager.get_configuration_component().lang,
            self._state_of_origin_item_properties_description
        )
    
    
    def _build_api_item_properties_filter(self):
        self._state_of_api_item_properties = self.source_configuration._state_of_api_item_properties
        api_item_properties_class = self.source_configuration._api_item_properties_class or ApiItemProperties
        self.api_item_properties = api_item_properties_class(self.logger, self.translator, self.file_loader, self._state_of_api_item_properties)
    
    
    # Call by Synchronizer after fork. Use for init source thread.
    def source_start(self):
        if self._reload_mapper:
            self._reload_mapper.start_thread()
        if self._reload_source_rules_manager:
            self._reload_source_rules_manager.start_thread()
    
    
    def build_items_containers(self):
        # type: () -> ItemsContainers
        containers = ItemsContainers()
        containers.set_sync_key_service(self.sync_keys_manager)
        if SERVICE_MODE.is_enable(self._state_of_display_origin_item) and self._display_origin_item_format:
            containers.set_display_origin_item_format(self._display_origin_item_format)
        if self._state_of_host_template_binding_rule in (SERVICE_MODE.ON, SERVICE_MODE.NOT_OVERLOAD_BY_USER):
            containers.set_source_rules_manager(self.host_template_binding_rule_manager)
        
        return containers
    
    
    def get_display_origin_item_format(self):
        return self._display_origin_item_format
    
    
    def load_configuration(self, configuration):
        raise NotImplementedError()
    
    
    def do_loop_turn(self):
        pass
    
    
    def loop_turn(self):
        pass
    
    
    @staticmethod
    def prepare_error_exit(message, level=SOURCE_STATE.CRITICAL):
        result = {
            'output'  : message,
            'state'   : level,
            'objects' : {},
            'errors'  : [message] if level == SOURCE_STATE.CRITICAL else [],
            'warnings': [message] if level == SOURCE_STATE.WARNING else [],
        }
        return result
    
    
    def get_objects(self):
        self.source_info.import_in_progressed = True
        ret = self._get_objects()
        self.source_info.import_in_progressed = False
        return ret
    
    
    def _get_objects(self):
        errors = []
        
        if SERVICE_MODE.is_enable(self._state_of_mapping_origin_to_source) and self.mapper_origin_to_source.validation_state.has_error():
            errors.extend(self.mapper_origin_to_source.validation_state.get_errors())
        
        if SERVICE_MODE.is_enable(self._state_of_api_item_properties) and self.api_item_properties.validation_state.has_error():
            errors.extend(self.api_item_properties.validation_state.get_errors())
        
        if SERVICE_MODE.is_enable(self._state_of_origin_item_properties_description) and self.origin_item_description.validation_state.has_error():
            errors.extend(self.origin_item_description.validation_state.get_errors())
        
        if SERVICE_MODE.is_enable(self._state_of_host_template_binding_rule) and self.host_template_binding_rule_manager.validation_state.has_error():
            errors.extend(self.host_template_binding_rule_manager.validation_state.get_errors())
        
        if self._load_error_message:
            errors.append(self._load_error_message)
        
        if errors:
            result = self.prepare_error_exit(',<br>'.join(errors))
            return result
        try:
            return self.import_source_items()
        except SourceException as e:
            return self.prepare_error_exit(e.message, e.level)
    
    
    def import_source_items(self):
        raise NotImplementedError()
